/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.cattleframework.utils.exception;

import java.lang.reflect.UndeclaredThrowableException;
import java.util.Set;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.cattleframework.exception.CattleException;
import org.cattleframework.exception.CattleResponseException;
import org.cattleframework.exception.ExceptionConstants;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 异常处理工具
 * 
 * @author orange
 *
 */
public final class ExceptionHandlingUtils {

    private static final Logger logger = LoggerFactory.getLogger(ExceptionHandlingUtils.class);

    private ExceptionHandlingUtils() {
    }

    public static ExceptionProcessResponse processException(Set<BaseExceptionCustomizer> exceptionCustomizers,
	    Throwable exception) {
	logger.error(exception.getMessage(), exception);
	if (exception instanceof UndeclaredThrowableException) {
	    exception = exception.getCause();
	}
	ExceptionProcessResponse exceptionProcessResponse = null;
	if (CollectionUtils.isNotEmpty(exceptionCustomizers)) {
	    for (BaseExceptionCustomizer exceptionCustomizer : exceptionCustomizers) {
		exceptionProcessResponse = exceptionCustomizer.process(exception);
		if (exceptionProcessResponse != null) {
		    break;
		}
	    }
	}
	if (exceptionProcessResponse == null) {
	    exceptionProcessResponse = getCattleExceptionResponse(exception);
	}
	Integer httpStatus = null;
	Integer exceptionCode = null;
	String message = null;
	String response = null;
	if (exceptionProcessResponse != null) {
	    httpStatus = exceptionProcessResponse.httpStatus();
	    exceptionCode = exceptionProcessResponse.exceptionCode();
	    message = exceptionProcessResponse.message();
	    response = exceptionProcessResponse.response();
	}
	if (StringUtils.isBlank(message)) {
	    message = exception.getMessage();
	}
	return new ExceptionProcessResponse(httpStatus, exceptionCode, message, response);
    }

    private static ExceptionProcessResponse getCattleExceptionResponse(Throwable e) {
	if (e instanceof CattleResponseException) {
	    CattleResponseException exception = (CattleResponseException) e;
	    Integer exceptionCode = null;
	    String message = null;
	    String response = null;
	    if (exception.getCode() != ExceptionConstants.GENERAL) {
		exceptionCode = exception.getCode();
	    }
	    if (StringUtils.isNotBlank(exception.getMessage())) {
		message = exception.getMessage();
	    }
	    if (StringUtils.isNotBlank(exception.getResponse())) {
		response = exception.getResponse();
	    }
	    return new ExceptionProcessResponse(null, exceptionCode, message, response);
	} else if (e instanceof CattleException) {
	    CattleException exception = (CattleException) e;
	    Integer exceptionCode = null;
	    String message = null;
	    if (exception.getCode() != ExceptionConstants.GENERAL) {
		exceptionCode = exception.getCode();
	    }
	    if (StringUtils.isNotBlank(exception.getMessage())) {
		message = exception.getMessage();
	    }
	    return new ExceptionProcessResponse(null, exceptionCode, message, null);
	} else {
	    if (e.getCause() != null) {
		return getCattleExceptionResponse(e.getCause());
	    }
	}
	return null;
    }
}