/*
 * Copyright (C) 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.cattleframework.utils.reflect;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.TypeVariable;
import java.lang.reflect.WildcardType;

import org.cattleframework.exception.CattleException;
import org.objenesis.Objenesis;
import org.objenesis.ObjenesisStd;

/**
 * 类的工具
 * 
 * @author orange
 *
 */
public final class ClassUtils {

    static final String NESTED_CLASS_SEPARATOR = "$";

    static final String CGLIB_CLASS_SEPARATOR = "$$";

    private static Objenesis objenesis = new ObjenesisStd(true);

    private ClassUtils() {
    }

    public static <T> T instanceEmpty(Class<T> type) {
	return objenesis.newInstance(type);
    }

    public static Class<?> getUserClass(Class<?> clazz) {
	String className = clazz.getName();
	if (className.contains(CGLIB_CLASS_SEPARATOR) || className.contains(NESTED_CLASS_SEPARATOR)) {
	    Class<?> superclass = clazz.getSuperclass();
	    if (superclass != null && superclass != Object.class) {
		return getUserClass(superclass);
	    }
	}
	return clazz;
    }

    @SuppressWarnings({ "unchecked", "rawtypes" })
    public static <T> Class<T> getTypeClass(java.lang.reflect.Type type) {
	if (type == null) {
	    return null;
	} else if (type instanceof Class<?>) {
	    return (Class<T>) type;
	} else if (type instanceof ParameterizedType) {
	    return (Class<T>) ((ParameterizedType) type).getRawType();
	} else if (type instanceof TypeVariable) {
	    return getTypeClass(((TypeVariable) type).getBounds()[0]);
	} else if (type instanceof WildcardType) {
	    return getTypeClass(((WildcardType) type).getUpperBounds()[0]);
	}
	throw new CattleException("不能获得java类型的类,从类: " + type);
    }
}