package com.zxl.ai.utils;

import lombok.extern.slf4j.Slf4j;

import javax.crypto.Cipher;
import java.io.ByteArrayOutputStream;
import java.security.*;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;

@Slf4j
public class RSAUtil {
    private static final String PUBLIC_KEYS = "MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQDtBhrQqENTp72Jim2k+qMkY64P24T5QN1eHevUpghRvGun5KTyLQnrH9ExMh0q14zabX0vnxbG978hWIuggJDM0OVqd7bUpagZU+jfq5HjiT2f05VIe+oeaqr8RZCmOKg4j6tFHhXjniQrQHf/gaJsxudOGF85ZBxHOBiFnH3osQIDAQAB";

    private static final String PRIVATE_KEYS = "MIICdwIBADANBgkqhkiG9w0BAQEFAASCAmEwggJdAgEAAoGBAO0GGtCoQ1OnvYmKbaT6oyRjrg/bhPlA3V4d69SmCFG8a6fkpPItCesf0TEyHSrXjNptfS+fFsb3vyFYi6CAkMzQ5Wp3ttSlqBlT6N+rkeOJPZ/TlUh76h5qqvxFkKY4qDiPq0UeFeOeJCtAd/+BomzG504YXzlkHEc4GIWcfeixAgMBAAECgYEAsxExoQH+G21WnFqgi0tvhUZpckF+LHD8zOX+DSGCl32qN6VdmDi+jYv0Fx6tOniuSTb+pMOkCZ2SIDctBY0b1qyFbsBar9LsAQurKlAM/iSYik4Z8Fv7pe0gl2PhE6PlzWgyXUlrbRvxnXbfWZ3rs4Bj3w39j+jtDkGPe84kZKECQQD9C7Gc4iMY0+5woNNDbkk4SKbL+WbrtawKpOA2S6HyXqMTdsCRZQbX1aiN8NlGsCjP7bf51A2lUQv9b3p7wLHvAkEA78qGUd4/Zg0a1yPdm5lf5wgBeZXZl0HhTv1bxyuUm7IvneBbCs4dOPk2Ftn0+u3fovfvSjMo5iV9v8fKUr8vXwJAf7q8X92CK9EmhbGYkmwEB6Yck8+DlS1x9zl8d8u7pnUAyh1iXMHrL12JJITTmCvpnfN3r2LLtIvxim8QG5GL/wJBAIZ4fHrMt6DzGGotosjvaeiVhL46WpuIRmGVkzJOLjhh5HIUrvmQPLMGjl6iO8+8Nf2eLILfzb/CppI16DAfJuECQFqs6GXTz2dJOsE4QtFjit1e39k5rzsUf9Xd6++ANDi1U62Ac1gflKqceeXvSBWra5MKeaeVmJlkQ0agopxFbZY=";

    public static final String KEY_ALGORITHM = "RSA";

    private static final String PUBLIC_KEY = "RSAPublicKey";

    private static final String PRIVATE_KEY = "RSAPrivateKey";

	// 1024 bits 的 RSA 密钥对，最大加密明文大小
    private static final int MAX_ENCRYPT_BLOCK = 117;

    // 1024 bits 的 RSA 密钥对，最大解密密文大小
    private static final int MAX_DECRYPT_BLOCK = 128;

    // 生成密钥对
    public static Map<String, Object> initKey(int keysize) throws Exception {
        KeyPairGenerator keyPairGen = KeyPairGenerator.getInstance(KEY_ALGORITHM);
        // 设置密钥对的 bit 数，越大越安全
        keyPairGen.initialize(keysize);
        KeyPair keyPair = keyPairGen.generateKeyPair();

        // 获取公钥
        RSAPublicKey publicKey = (RSAPublicKey) keyPair.getPublic();
        // 获取私钥
        RSAPrivateKey privateKey = (RSAPrivateKey) keyPair.getPrivate();
        Map<String, Object> keyMap = new HashMap<>(2);
        keyMap.put(PUBLIC_KEY, publicKey);
        keyMap.put(PRIVATE_KEY, privateKey);
        return keyMap;
    }

    // 获取公钥字符串
    public static String getPublicKeyStr(Map<String, Object> keyMap) {
        // 获得 map 中的公钥对象，转为 key 对象
        Key key = (Key) keyMap.get(PUBLIC_KEY);
        // 编码返回字符串
        return encryptBASE64(key.getEncoded());
    }

    // 获取私钥字符串
    public static String getPrivateKeyStr(Map<String, Object> keyMap) {
        // 获得 map 中的私钥对象，转为 key 对象
        Key key = (Key) keyMap.get(PRIVATE_KEY);
        // 编码返回字符串
        return encryptBASE64(key.getEncoded());
    }

    // 获取公钥
    public static PublicKey getPublicKey(String publicKeyString) throws NoSuchAlgorithmException, InvalidKeySpecException {
        byte[] publicKeyByte = Base64.getDecoder().decode(publicKeyString);
        X509EncodedKeySpec keySpec = new X509EncodedKeySpec(publicKeyByte);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
        return keyFactory.generatePublic(keySpec);
    }

    // 获取私钥
    public static PrivateKey getPrivateKey(String privateKeyString) throws Exception {
        byte[] privateKeyByte = Base64.getDecoder().decode(privateKeyString);
        PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(privateKeyByte);
        KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
        return keyFactory.generatePrivate(keySpec);
    }

    /**
     * BASE64 编码返回加密字符串
     *
     * @param key 需要编码的字节数组
     * @return 编码后的字符串
     */
    public static String encryptBASE64(byte[] key) {
        return new String(Base64.getEncoder().encode(key));
    }

	/**
     * BASE64 解码，返回字节数组
     *
     * @param key 待解码的字符串
     * @return 解码后的字节数组
     */
    public static byte[] decryptBASE64(String key) {
        return Base64.getDecoder().decode(key);
    }

    /**
     * 公钥加密
     *
     * @param text         待加密的明文字符串
     * @param publicKeyStr 公钥
     * @return 加密后的密文
     */
    public static String encrypt1(String text, String publicKeyStr) {
        try {
            log.info("明文字符串为:[{}]", text);
            Cipher cipher = Cipher.getInstance(KEY_ALGORITHM);
            cipher.init(Cipher.ENCRYPT_MODE, getPublicKey(publicKeyStr));
            byte[] tempBytes = cipher.doFinal(text.getBytes("UTF-8"));
            return Base64.getEncoder().encodeToString(tempBytes);
        } catch (Exception e) {
            throw new RuntimeException("加密字符串[" + text + "]时遇到异常", e);
        }
    }

    /**
     * 私钥解密
     *
     * @param secretText    待解密的密文字符串
     * @param privateKeyStr 私钥
     * @return 解密后的明文
     */
    public static String decrypt1(String secretText, String privateKeyStr) {
        try {
            // 生成私钥
            Cipher cipher = Cipher.getInstance(KEY_ALGORITHM);
            cipher.init(Cipher.DECRYPT_MODE, getPrivateKey(privateKeyStr));
            // 密文解码
            byte[] secretTextDecoded = Base64.getDecoder().decode(secretText.getBytes("UTF-8"));
            byte[] tempBytes = cipher.doFinal(secretTextDecoded);
            return new String(tempBytes);
        } catch (Exception e) {
            throw new RuntimeException("解密字符串[" + secretText + "]时遇到异常", e);
        }
    }

    public static void main(String[] args) throws Exception {
        Map<String, Object> keyMap;
        String cipherText;
        // 原始明文
        String content = "春江潮水连海平，海上明月共潮生。滟滟随波千万里，何处春江无月明。";

//        // 生成密钥对
//        keyMap = initKey(1024);
//        String publicKey = getPublicKeyStr(keyMap);
//        log.info("公钥:[{}]，长度:[{}]", publicKey, publicKey.length());
//        String privateKey = getPrivateKeyStr(keyMap);
//        log.info("私钥:[{}]，长度:[{}]", privateKey, privateKey.length());


        String publicKey = PUBLIC_KEYS;
        log.info("公钥:[{}]，长度:[{}]", publicKey, publicKey.length());
        String privateKey = PRIVATE_KEYS;
        log.info("私钥:[{}]，长度:[{}]", privateKey, privateKey.length());

        // 加密
        cipherText = encrypt1(content, publicKey);
        log.info("加密后的密文:[{}]，长度:[{}]", cipherText, cipherText.length());

        // 解密
        String plainText = decrypt1(cipherText, privateKey);
        log.info("解密后明文:[{}]", plainText);
    }
}

