/*
 * Decompiled with CFR 0.152.
 */
package io.gravitee.plugin.core.internal;

import com.google.common.annotations.VisibleForTesting;
import io.gravitee.common.event.EventManager;
import io.gravitee.common.service.AbstractService;
import io.gravitee.plugin.core.api.Plugin;
import io.gravitee.plugin.core.api.PluginEvent;
import io.gravitee.plugin.core.api.PluginManifest;
import io.gravitee.plugin.core.api.PluginManifestFactory;
import io.gravitee.plugin.core.api.PluginRegistry;
import io.gravitee.plugin.core.internal.PluginImpl;
import io.gravitee.plugin.core.internal.PluginRegistryConfiguration;
import io.gravitee.plugin.core.internal.PropertiesBasedPluginManifestValidator;
import io.gravitee.plugin.core.utils.FileUtils;
import io.gravitee.plugin.core.utils.GlobMatchingFileVisitor;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.file.DirectoryStream;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.env.Environment;
import org.springframework.util.StringUtils;

public class PluginRegistryImpl
extends AbstractService
implements PluginRegistry {
    public static final String PROPERTY_STRING_FORMAT = "%s.%s.enabled";
    private static final Logger LOGGER = LoggerFactory.getLogger(PluginRegistryImpl.class);
    private static final String JAR_EXTENSION = ".jar";
    private static final String JAR_GLOB = "*.jar";
    private static final String ZIP_EXTENSION = ".zip";
    private static final String ZIP_GLOB = "*.zip";
    private static final String PLUGIN_MANIFEST_FILE = "plugin.properties";
    private static final Map<String, String> PLUGIN_TYPE_PROPERTY_ALIASES = new HashMap<String, String>();
    @Autowired
    private PluginRegistryConfiguration configuration;
    @Autowired
    private Environment environment;
    @Autowired
    @Qualifier(value="corePluginExecutor")
    private ExecutorService executor;
    private boolean init = false;
    private List<Plugin> plugins = new ArrayList<Plugin>();
    @Autowired
    private EventManager eventManager;
    private String[] workspacesPath;

    public PluginRegistryImpl() {
    }

    public PluginRegistryImpl(String workspacePath) {
        this.workspacesPath = new String[]{workspacePath};
    }

    protected void doStart() throws Exception {
        super.doStart();
        if (!this.init) {
            LOGGER.info("Initializing plugin registry.");
            this.init();
            LOGGER.info("Plugins have been loaded and installed.");
        } else {
            LOGGER.warn("Plugin registry has already been initialized.");
        }
    }

    public void init() throws Exception {
        String[] pluginsPath = this.configuration.getPluginsPath();
        if ((pluginsPath == null || pluginsPath.length == 0) && this.workspacesPath == null) {
            LOGGER.error("No plugin registry configured.");
            throw new RuntimeException("No plugin registry configured.");
        }
        if (this.workspacesPath != null) {
            pluginsPath = this.workspacesPath;
        }
        for (String aWorkspacePath : pluginsPath) {
            this.loadPluginsFromRegistry(aWorkspacePath);
        }
        this.printPlugins();
        this.eventManager.publishEvent((Enum)PluginEvent.ENDED, null);
    }

    private void loadPluginsFromRegistry(String registryPath) throws Exception {
        File registryDir = new File(registryPath);
        if (!registryDir.isDirectory()) {
            LOGGER.error("Invalid registry directory, {} is not a directory.", (Object)registryDir.getAbsolutePath());
            throw new RuntimeException("Invalid registry directory. Not a directory: " + registryDir.getAbsolutePath());
        }
        this.loadPlugins(registryDir);
    }

    private void loadPlugins(File registryDir) throws Exception {
        Path registryPath = registryDir.toPath();
        LOGGER.info("Loading plugins from {}", (Object)registryDir);
        try (DirectoryStream stream = FileUtils.newDirectoryStream(registryPath, ZIP_GLOB);){
            ArrayList<CompletableFuture<Void>> futures = new ArrayList<CompletableFuture<Void>>();
            Iterator archiveIte = stream.iterator();
            if (archiveIte.hasNext()) {
                while (archiveIte.hasNext()) {
                    Path path = (Path)archiveIte.next();
                    futures.add(CompletableFuture.runAsync(() -> this.loadPlugin(registryDir, path), this.executor));
                }
            } else {
                LOGGER.warn("No plugin has been found in {}", (Object)registryDir);
            }
            CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).get();
            this.init = true;
        }
        catch (IOException ioe) {
            LOGGER.error("An unexpected error occurs", (Throwable)ioe);
            throw ioe;
        }
    }

    private void printPlugins() {
        this.plugins.stream().map(Plugin::type).distinct().forEach(this::printPluginByType);
    }

    private void printPluginByType(String pluginType) {
        LOGGER.info("List of available {}: ", (Object)pluginType.toLowerCase());
        this.plugins.stream().filter(plugin -> pluginType.equalsIgnoreCase(plugin.type())).forEach(plugin -> LOGGER.info("\t> {} [{}] has been loaded", (Object)plugin.id(), (Object)plugin.manifest().version()));
    }

    private void loadPlugin(File registryDir, Path pluginArchivePath) {
        LOGGER.debug("Loading plugin from {}", (Object)pluginArchivePath);
        try {
            String sPluginFile = pluginArchivePath.toFile().getName();
            sPluginFile = sPluginFile.substring(0, sPluginFile.lastIndexOf(ZIP_EXTENSION));
            Path workDir = FileSystems.getDefault().getPath(registryDir.getAbsolutePath(), ".work", sPluginFile);
            if (StringUtils.hasText((String)this.configuration.getPluginWorkDir()) && !(workDir = FileSystems.getDefault().getPath(this.configuration.getPluginWorkDir(), sPluginFile)).toFile().getParentFile().exists()) {
                workDir.toFile().getParentFile().mkdirs();
            }
            FileUtils.delete(workDir);
            FileUtils.unzip(pluginArchivePath.toString(), workDir);
            PluginManifest manifest = this.readPluginManifest(workDir);
            if (manifest != null && this.isEnabled(manifest)) {
                URL[] pluginDependencies = this.extractPluginDependencies(workDir);
                URL[] extDependencies = this.extractPluginExtensionDependencies(manifest, registryDir.toPath());
                if (extDependencies != null) {
                    URL[] dependencies = Arrays.copyOf(pluginDependencies, pluginDependencies.length + extDependencies.length);
                    System.arraycopy(extDependencies, 0, dependencies, pluginDependencies.length, extDependencies.length);
                    pluginDependencies = dependencies;
                }
                PluginImpl plugin = new PluginImpl(manifest);
                plugin.setPath(workDir);
                plugin.setDependencies(pluginDependencies);
                this.eventManager.publishEvent((Enum)PluginEvent.DEPLOYED, (Object)plugin);
                this.plugins.add(plugin);
            }
        }
        catch (IOException ioe) {
            LOGGER.error("An unexpected error occurs while loading plugin archive {}", (Object)pluginArchivePath, (Object)ioe);
        }
    }

    private boolean isEnabled(PluginManifest pluginManifest) {
        String propertyFromAlias = String.format(PROPERTY_STRING_FORMAT, PLUGIN_TYPE_PROPERTY_ALIASES.get(pluginManifest.type()), pluginManifest.id());
        boolean enabled = PLUGIN_TYPE_PROPERTY_ALIASES.containsKey(pluginManifest.type()) && this.environment.containsProperty(propertyFromAlias) ? ((Boolean)this.environment.getProperty(propertyFromAlias, Boolean.class, (Object)true)).booleanValue() : ((Boolean)this.environment.getProperty(String.format(PROPERTY_STRING_FORMAT, pluginManifest.type(), pluginManifest.id()), Boolean.class, (Object)true)).booleanValue();
        LOGGER.debug("Plugin {} is loaded in registry: {}", (Object)pluginManifest.id(), (Object)enabled);
        return enabled;
    }

    private URL[] extractPluginDependencies(Path pluginDirPath) {
        try {
            GlobMatchingFileVisitor visitor = new GlobMatchingFileVisitor(JAR_GLOB);
            Files.walkFileTree(pluginDirPath, visitor);
            List<Path> pluginDependencies = visitor.getMatchedPaths();
            return this.listToArray(pluginDependencies);
        }
        catch (IOException ioe) {
            LOGGER.error("Unexpected error while looking for plugin dependencies", (Throwable)ioe);
            return null;
        }
    }

    private URL[] extractPluginExtensionDependencies(PluginManifest manifest, Path registryPath) {
        try {
            GlobMatchingFileVisitor visitor = new GlobMatchingFileVisitor(JAR_GLOB);
            Path extPath = Paths.get(registryPath.toString(), "ext", manifest.id());
            if (extPath.toFile().exists()) {
                Files.walkFileTree(extPath, visitor);
                List<Path> pluginDependencies = visitor.getMatchedPaths();
                return this.listToArray(pluginDependencies);
            }
            return null;
        }
        catch (IOException ioe) {
            LOGGER.error("Unexpected error while looking for plugin dependencies", (Throwable)ioe);
            return null;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private PluginManifest readPluginManifest(Path pluginPath) {
        try (DirectoryStream stream = FileUtils.newDirectoryStream(pluginPath, JAR_GLOB);){
            Iterator iterator = stream.iterator();
            if (!iterator.hasNext()) {
                LOGGER.debug("Unable to find a jar in the root directory: {}", (Object)pluginPath);
                PluginManifest pluginManifest = null;
                return pluginManifest;
            }
            Path pluginJarPath = (Path)iterator.next();
            LOGGER.debug("Found a jar in the root directory, looking for a plugin manifest in {}", (Object)pluginJarPath);
            Properties pluginManifestProperties = this.loadPluginManifest(pluginJarPath.toString());
            if (pluginManifestProperties == null) {
                LOGGER.error("No plugin.properties found from {}", (Object)pluginJarPath);
                PluginManifest pluginManifest = null;
                return pluginManifest;
            }
            LOGGER.debug("A plugin manifest has been loaded from: {}", (Object)pluginJarPath);
            PropertiesBasedPluginManifestValidator validator = new PropertiesBasedPluginManifestValidator(pluginManifestProperties);
            if (!validator.validate()) {
                LOGGER.error("Plugin manifest not valid, skipping plugin registration.");
                PluginManifest pluginManifest = null;
                return pluginManifest;
            }
            PluginManifest pluginManifest = PluginManifestFactory.create(pluginManifestProperties);
            return pluginManifest;
        }
        catch (IOException ioe) {
            LOGGER.error("Unexpected error while trying to load plugin manifest", (Throwable)ioe);
            throw new IllegalStateException("Unexpected error while trying to load plugin manifest", ioe);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private Properties loadPluginManifest(String pluginPath) {
        try (FileSystem zipFileSystem = FileUtils.createZipFileSystem(pluginPath, false);){
            Path root = zipFileSystem.getPath("/", new String[0]);
            PluginManifestVisitor visitor = new PluginManifestVisitor();
            Files.walkFileTree(root, visitor);
            Path pluginManifestPath = visitor.getPluginManifest();
            if (pluginManifestPath == null) return null;
            Properties properties2 = new Properties();
            properties2.load(Files.newInputStream(pluginManifestPath, new OpenOption[0]));
            Properties properties = properties2;
            return properties;
        }
        catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    private URL[] listToArray(List<Path> paths) {
        URL[] urls = new URL[paths.size()];
        int idx = 0;
        for (Path path : paths) {
            try {
                urls[idx++] = path.toUri().toURL();
            }
            catch (IOException iOException) {}
        }
        return urls;
    }

    private List<File> getPluginsArchive(String directory) {
        DirectoryStream.Filter<Path> filter = file -> Files.isDirectory(file, new LinkOption[0]);
        ArrayList<File> files = new ArrayList<File>();
        Path dir = FileSystems.getDefault().getPath(directory, new String[0]);
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(dir, filter);){
            for (Path path : stream) {
                files.add(path.toFile());
            }
        }
        catch (IOException e) {
            e.printStackTrace();
        }
        return files;
    }

    @Override
    public Collection<Plugin> plugins() {
        return this.plugins;
    }

    @Override
    public Collection<Plugin> plugins(String type) {
        return this.plugins.stream().filter(pluginContext -> type.equalsIgnoreCase(pluginContext.type())).collect(Collectors.toSet());
    }

    public void setEventManager(EventManager eventManager) {
        this.eventManager = eventManager;
    }

    public void setExecutor(ExecutorService executor) {
        this.executor = executor;
    }

    public void setConfiguration(PluginRegistryConfiguration configuration) {
        this.configuration = configuration;
    }

    @VisibleForTesting
    void setEnvironment(Environment environment) {
        this.environment = environment;
    }

    static {
        PLUGIN_TYPE_PROPERTY_ALIASES.put("service", "services");
        PLUGIN_TYPE_PROPERTY_ALIASES.put("policy", "policies");
        PLUGIN_TYPE_PROPERTY_ALIASES.put("alert", "alerts");
        PLUGIN_TYPE_PROPERTY_ALIASES.put("fetcher", "fetchers");
        PLUGIN_TYPE_PROPERTY_ALIASES.put("connector", "connectors");
        PLUGIN_TYPE_PROPERTY_ALIASES.put("notifier", "notifiers");
        PLUGIN_TYPE_PROPERTY_ALIASES.put("service_discovery", "service-discoveries");
    }

    class PluginManifestVisitor
    extends SimpleFileVisitor<Path> {
        private Path pluginManifest = null;

        PluginManifestVisitor() {
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            if (file.getFileName().toString().equals(PluginRegistryImpl.PLUGIN_MANIFEST_FILE)) {
                this.pluginManifest = file;
                return FileVisitResult.TERMINATE;
            }
            return super.visitFile(file, attrs);
        }

        public Path getPluginManifest() {
            return this.pluginManifest;
        }
    }
}

