/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.netty;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import io.grpc.Status;
import io.grpc.internal.GrpcUtil;
import io.grpc.netty.GrpcSslContexts;
import io.grpc.netty.JettyTlsUtil;
import io.grpc.netty.ProtocolNegotiator;
import io.grpc.netty.Utils;
import io.netty.channel.ChannelDuplexHandler;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelHandlerAdapter;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInboundHandlerAdapter;
import io.netty.channel.ChannelPipeline;
import io.netty.channel.ChannelPromise;
import io.netty.handler.codec.http.DefaultHttpRequest;
import io.netty.handler.codec.http.HttpClientCodec;
import io.netty.handler.codec.http.HttpClientUpgradeHandler;
import io.netty.handler.codec.http.HttpMethod;
import io.netty.handler.codec.http.HttpVersion;
import io.netty.handler.codec.http2.Http2ClientUpgradeCodec;
import io.netty.handler.codec.http2.Http2ConnectionHandler;
import io.netty.handler.ssl.OpenSsl;
import io.netty.handler.ssl.OpenSslEngine;
import io.netty.handler.ssl.SslContext;
import io.netty.handler.ssl.SslHandler;
import io.netty.handler.ssl.SslHandshakeCompletionEvent;
import io.netty.util.AsciiString;
import io.netty.util.ReferenceCountUtil;
import java.net.URI;
import java.util.ArrayDeque;
import java.util.Arrays;
import java.util.Queue;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLParameters;

public final class ProtocolNegotiators {
    private static final Logger log = Logger.getLogger(ProtocolNegotiators.class.getName());

    private ProtocolNegotiators() {
    }

    public static ProtocolNegotiator serverPlaintext() {
        return new ProtocolNegotiator(){

            @Override
            public ProtocolNegotiator.Handler newHandler(final Http2ConnectionHandler handler) {
                class PlaintextHandler
                extends ChannelHandlerAdapter
                implements ProtocolNegotiator.Handler {
                    PlaintextHandler() {
                    }

                    public void handlerAdded(ChannelHandlerContext ctx) throws Exception {
                        ctx.pipeline().replace((ChannelHandler)this, null, (ChannelHandler)handler);
                    }

                    @Override
                    public AsciiString scheme() {
                        return Utils.HTTP;
                    }
                }
                return new PlaintextHandler();
            }
        };
    }

    public static ProtocolNegotiator serverTls(final SslContext sslContext) {
        Preconditions.checkNotNull((Object)sslContext, (Object)"sslContext");
        return new ProtocolNegotiator(){

            @Override
            public ProtocolNegotiator.Handler newHandler(Http2ConnectionHandler handler) {
                return new ServerTlsHandler(sslContext, (ChannelHandler)handler);
            }
        };
    }

    public static ProtocolNegotiator tls(SslContext sslContext, String authority) {
        int port;
        String host;
        Preconditions.checkNotNull((Object)sslContext, (Object)"sslContext");
        URI uri = GrpcUtil.authorityToUri((String)((String)Preconditions.checkNotNull((Object)authority, (Object)"authority")));
        if (uri.getHost() != null) {
            host = uri.getHost();
            port = uri.getPort();
        } else {
            host = authority;
            port = -1;
        }
        return new TlsNegotiator(sslContext, host, port);
    }

    public static ProtocolNegotiator plaintextUpgrade() {
        return new PlaintextUpgradeNegotiator();
    }

    public static ProtocolNegotiator plaintext() {
        return new PlaintextNegotiator();
    }

    private static RuntimeException unavailableException(String msg) {
        return Status.UNAVAILABLE.withDescription(msg).asRuntimeException();
    }

    @VisibleForTesting
    static void logSslEngineDetails(Level level, ChannelHandlerContext ctx, String msg, @Nullable Throwable t) {
        if (!log.isLoggable(level)) {
            return;
        }
        SslHandler sslHandler = (SslHandler)ctx.pipeline().get(SslHandler.class);
        SSLEngine engine = sslHandler.engine();
        StringBuilder builder = new StringBuilder(msg);
        builder.append("\nSSLEngine Details: [\n");
        if (engine instanceof OpenSslEngine) {
            builder.append("    OpenSSL, ");
            builder.append("Version: 0x").append(Integer.toHexString(OpenSsl.version()));
            builder.append(" (").append(OpenSsl.versionString()).append("), ");
            builder.append("ALPN supported: ").append(OpenSsl.isAlpnSupported());
        } else if (JettyTlsUtil.isJettyAlpnConfigured()) {
            builder.append("    Jetty ALPN");
        } else if (JettyTlsUtil.isJettyNpnConfigured()) {
            builder.append("    Jetty NPN");
        }
        builder.append("\n    TLS Protocol: ");
        builder.append(engine.getSession().getProtocol());
        builder.append("\n    Application Protocol: ");
        builder.append(sslHandler.applicationProtocol());
        builder.append("\n    Need Client Auth: ");
        builder.append(engine.getNeedClientAuth());
        builder.append("\n    Want Client Auth: ");
        builder.append(engine.getWantClientAuth());
        builder.append("\n    Supported protocols=");
        builder.append(Arrays.toString(engine.getSupportedProtocols()));
        builder.append("\n    Enabled protocols=");
        builder.append(Arrays.toString(engine.getEnabledProtocols()));
        builder.append("\n    Supported ciphers=");
        builder.append(Arrays.toString(engine.getSupportedCipherSuites()));
        builder.append("\n    Enabled ciphers=");
        builder.append(Arrays.toString(engine.getEnabledCipherSuites()));
        builder.append("\n]");
        log.log(level, builder.toString(), t);
    }

    private static class BufferingHttp2UpgradeHandler
    extends AbstractBufferingHandler
    implements ProtocolNegotiator.Handler {
        BufferingHttp2UpgradeHandler(ChannelHandler ... handlers) {
            super(handlers);
        }

        @Override
        public AsciiString scheme() {
            return Utils.HTTP;
        }

        public void channelActive(ChannelHandlerContext ctx) throws Exception {
            DefaultHttpRequest upgradeTrigger = new DefaultHttpRequest(HttpVersion.HTTP_1_1, HttpMethod.GET, "/");
            ctx.writeAndFlush((Object)upgradeTrigger);
            super.channelActive(ctx);
        }

        public void userEventTriggered(ChannelHandlerContext ctx, Object evt) throws Exception {
            if (evt == HttpClientUpgradeHandler.UpgradeEvent.UPGRADE_SUCCESSFUL) {
                this.writeBufferedAndRemove(ctx);
            } else if (evt == HttpClientUpgradeHandler.UpgradeEvent.UPGRADE_REJECTED) {
                this.fail(ctx, ProtocolNegotiators.unavailableException("HTTP/2 upgrade rejected"));
            }
            super.userEventTriggered(ctx, evt);
        }
    }

    private static class BufferUntilChannelActiveHandler
    extends AbstractBufferingHandler
    implements ProtocolNegotiator.Handler {
        BufferUntilChannelActiveHandler(ChannelHandler ... handlers) {
            super(handlers);
        }

        @Override
        public AsciiString scheme() {
            return Utils.HTTP;
        }

        public void handlerAdded(ChannelHandlerContext ctx) {
            this.writeBufferedAndRemove(ctx);
        }

        public void channelActive(ChannelHandlerContext ctx) throws Exception {
            this.writeBufferedAndRemove(ctx);
            super.channelActive(ctx);
        }
    }

    private static class BufferUntilTlsNegotiatedHandler
    extends AbstractBufferingHandler
    implements ProtocolNegotiator.Handler {
        BufferUntilTlsNegotiatedHandler(ChannelHandler ... handlers) {
            super(handlers);
        }

        @Override
        public AsciiString scheme() {
            return Utils.HTTPS;
        }

        public void userEventTriggered(ChannelHandlerContext ctx, Object evt) throws Exception {
            if (evt instanceof SslHandshakeCompletionEvent) {
                SslHandshakeCompletionEvent handshakeEvent = (SslHandshakeCompletionEvent)evt;
                if (handshakeEvent.isSuccess()) {
                    SslHandler handler = (SslHandler)ctx.pipeline().get(SslHandler.class);
                    if (GrpcSslContexts.HTTP2_VERSIONS.contains(handler.applicationProtocol())) {
                        ProtocolNegotiators.logSslEngineDetails(Level.FINER, ctx, "TLS negotiation succeeded.", null);
                        this.writeBufferedAndRemove(ctx);
                    } else {
                        Exception ex = new Exception("Failed ALPN negotiation: Unable to find compatible protocol.");
                        ProtocolNegotiators.logSslEngineDetails(Level.FINE, ctx, "TLS negotiation failed.", ex);
                        this.fail(ctx, ex);
                    }
                } else {
                    this.fail(ctx, handshakeEvent.cause());
                }
            }
            super.userEventTriggered(ctx, evt);
        }
    }

    public static abstract class AbstractBufferingHandler
    extends ChannelDuplexHandler {
        private ChannelHandler[] handlers;
        private Queue<ChannelWrite> bufferedWrites = new ArrayDeque<ChannelWrite>();
        private boolean writing;
        private boolean flushRequested;
        private Throwable failCause;

        protected AbstractBufferingHandler(ChannelHandler ... handlers) {
            this.handlers = handlers;
        }

        public void channelRegistered(ChannelHandlerContext ctx) throws Exception {
            if (this.handlers != null) {
                ctx.pipeline().addFirst(this.handlers);
                this.handlers = null;
            }
            super.channelRegistered(ctx);
        }

        public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) throws Exception {
            this.fail(ctx, cause);
        }

        public void channelInactive(ChannelHandlerContext ctx) throws Exception {
            this.fail(ctx, ProtocolNegotiators.unavailableException("Connection broken while performing protocol negotiation"));
            super.channelInactive(ctx);
        }

        public void write(ChannelHandlerContext ctx, Object msg, ChannelPromise promise) throws Exception {
            if (this.failCause != null) {
                promise.setFailure(this.failCause);
                ReferenceCountUtil.release((Object)msg);
            } else if (this.bufferedWrites == null) {
                super.write(ctx, msg, promise);
            } else {
                this.bufferedWrites.add(new ChannelWrite(msg, promise));
            }
        }

        public void flush(ChannelHandlerContext ctx) {
            if (this.bufferedWrites == null) {
                ctx.flush();
            } else {
                this.flushRequested = true;
            }
        }

        public void close(ChannelHandlerContext ctx, ChannelPromise future) throws Exception {
            this.fail(ctx, ProtocolNegotiators.unavailableException("Channel closed while performing protocol negotiation"));
        }

        protected final void fail(ChannelHandlerContext ctx, Throwable cause) {
            if (this.failCause == null) {
                this.failCause = cause;
            }
            if (this.bufferedWrites != null) {
                while (!this.bufferedWrites.isEmpty()) {
                    ChannelWrite write = this.bufferedWrites.poll();
                    write.promise.setFailure(cause);
                    ReferenceCountUtil.release((Object)write.msg);
                }
                this.bufferedWrites = null;
            }
            ctx.close();
        }

        protected final void writeBufferedAndRemove(ChannelHandlerContext ctx) {
            if (!ctx.channel().isActive() || this.writing) {
                return;
            }
            this.writing = true;
            while (!this.bufferedWrites.isEmpty()) {
                ChannelWrite write = this.bufferedWrites.poll();
                ctx.write(write.msg, write.promise);
            }
            assert (this.bufferedWrites.isEmpty());
            this.bufferedWrites = null;
            if (this.flushRequested) {
                ctx.flush();
            }
            ctx.pipeline().remove((ChannelHandler)this);
        }

        private static class ChannelWrite {
            Object msg;
            ChannelPromise promise;

            ChannelWrite(Object msg, ChannelPromise promise) {
                this.msg = msg;
                this.promise = promise;
            }
        }
    }

    static final class PlaintextNegotiator
    implements ProtocolNegotiator {
        PlaintextNegotiator() {
        }

        @Override
        public ProtocolNegotiator.Handler newHandler(Http2ConnectionHandler handler) {
            return new BufferUntilChannelActiveHandler(new ChannelHandler[]{handler});
        }
    }

    static final class PlaintextUpgradeNegotiator
    implements ProtocolNegotiator {
        PlaintextUpgradeNegotiator() {
        }

        @Override
        public ProtocolNegotiator.Handler newHandler(Http2ConnectionHandler handler) {
            Http2ClientUpgradeCodec upgradeCodec = new Http2ClientUpgradeCodec(handler);
            HttpClientCodec httpClientCodec = new HttpClientCodec();
            HttpClientUpgradeHandler upgrader = new HttpClientUpgradeHandler((HttpClientUpgradeHandler.SourceCodec)httpClientCodec, (HttpClientUpgradeHandler.UpgradeCodec)upgradeCodec, 1000);
            return new BufferingHttp2UpgradeHandler(new ChannelHandler[]{upgrader});
        }
    }

    static final class TlsNegotiator
    implements ProtocolNegotiator {
        private final SslContext sslContext;
        private final String host;
        private final int port;

        TlsNegotiator(SslContext sslContext, String host, int port) {
            this.sslContext = (SslContext)Preconditions.checkNotNull((Object)sslContext);
            this.host = (String)Preconditions.checkNotNull((Object)host);
            this.port = port;
        }

        @VisibleForTesting
        String getHost() {
            return this.host;
        }

        @VisibleForTesting
        int getPort() {
            return this.port;
        }

        @Override
        public ProtocolNegotiator.Handler newHandler(Http2ConnectionHandler handler) {
            ChannelHandlerAdapter sslBootstrap = new ChannelHandlerAdapter(){

                public void handlerAdded(ChannelHandlerContext ctx) throws Exception {
                    SSLEngine sslEngine = TlsNegotiator.this.sslContext.newEngine(ctx.alloc(), TlsNegotiator.this.host, TlsNegotiator.this.port);
                    SSLParameters sslParams = new SSLParameters();
                    sslParams.setEndpointIdentificationAlgorithm("HTTPS");
                    sslEngine.setSSLParameters(sslParams);
                    ctx.pipeline().replace((ChannelHandler)this, null, (ChannelHandler)new SslHandler(sslEngine, false));
                }
            };
            return new BufferUntilTlsNegotiatedHandler(new ChannelHandler[]{sslBootstrap, handler});
        }
    }

    @VisibleForTesting
    static final class ServerTlsHandler
    extends ChannelInboundHandlerAdapter
    implements ProtocolNegotiator.Handler {
        private final ChannelHandler grpcHandler;
        private final SslContext sslContext;

        ServerTlsHandler(SslContext sslContext, ChannelHandler grpcHandler) {
            this.sslContext = sslContext;
            this.grpcHandler = grpcHandler;
        }

        public void handlerAdded(ChannelHandlerContext ctx) throws Exception {
            super.handlerAdded(ctx);
            SSLEngine sslEngine = this.sslContext.newEngine(ctx.alloc());
            ctx.pipeline().addFirst(new ChannelHandler[]{new SslHandler(sslEngine, false)});
        }

        public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) throws Exception {
            this.fail(ctx, cause);
        }

        public void userEventTriggered(ChannelHandlerContext ctx, Object evt) throws Exception {
            if (evt instanceof SslHandshakeCompletionEvent) {
                SslHandshakeCompletionEvent handshakeEvent = (SslHandshakeCompletionEvent)evt;
                if (handshakeEvent.isSuccess()) {
                    if (GrpcSslContexts.HTTP2_VERSIONS.contains(this.sslHandler(ctx.pipeline()).applicationProtocol())) {
                        ctx.pipeline().replace((ChannelHandler)this, null, this.grpcHandler);
                    } else {
                        this.fail(ctx, new Exception("Failed protocol negotiation: Unable to find compatible protocol."));
                    }
                } else {
                    this.fail(ctx, handshakeEvent.cause());
                }
            }
            super.userEventTriggered(ctx, evt);
        }

        private SslHandler sslHandler(ChannelPipeline pipeline) {
            return (SslHandler)pipeline.get(SslHandler.class);
        }

        private void fail(ChannelHandlerContext ctx, Throwable exception) {
            ProtocolNegotiators.logSslEngineDetails(Level.FINE, ctx, "TLS negotiation failed for new client.", exception);
            ctx.close();
        }

        @Override
        public AsciiString scheme() {
            return Utils.HTTPS;
        }
    }
}

