/*
 * Copyright 2016 Game Server Services, Inc. or its affiliates. All Rights
 * Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package io.gs2.quest;

import java.util.ArrayList;
import java.util.List;

import io.gs2.model.Region;
import io.gs2.util.EncodingUtil;
import org.apache.http.NameValuePair;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.message.BasicNameValuePair;

import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.JsonNode;
import org.json.JSONObject;

import io.gs2.AbstractGs2Client;
import io.gs2.Gs2Constant;
import io.gs2.model.IGs2Credential;
import io.gs2.quest.request.*;
import io.gs2.quest.result.*;
import io.gs2.quest.model.*;

/**
 * GS2 Quest API クライアント
 *
 * @author Game Server Services, Inc.
 *
 */
public class Gs2QuestRestClient extends AbstractGs2Client<Gs2QuestRestClient> {

	public static String ENDPOINT = "quest";

	/**
	 * コンストラクタ。
	 *
	 * @param credential 認証情報
	 */
	public Gs2QuestRestClient(IGs2Credential credential) {
		super(credential);
	}

	/**
	 * コンストラクタ。
	 *
	 * @param credential 認証情報
	 * @param region リージョン
	 */
	public Gs2QuestRestClient(IGs2Credential credential, Region region) {
		super(credential, region);
	}

	/**
	 * コンストラクタ。
	 *
	 * @param credential 認証情報
	 * @param region リージョン
	 */
	public Gs2QuestRestClient(IGs2Credential credential, String region) {
		super(credential, region);

	}

	/**
	 * クエストを分類するカテゴリーの一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeNamespacesResult describeNamespaces(DescribeNamespacesRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FNamespaceFunctionHandler.describeNamespaces";
        String pageToken = String.valueOf(request.getPageToken());
        String limit = String.valueOf(request.getLimit());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(pageToken != null) {
            _queryString.add(new BasicNameValuePair("pageToken", String.valueOf(pageToken)));
        }
        if(limit != null) {
            _queryString.add(new BasicNameValuePair("limit", String.valueOf(limit)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DescribeNamespacesResult.class);
    }

	/**
	 * クエストを分類するカテゴリーを新規作成<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public CreateNamespaceResult createNamespace(CreateNamespaceRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FNamespaceFunctionHandler.createNamespace";
        String name = null;
        if(request.getName() != null) {
            name = request.getName();
        }
        String description = null;
        if(request.getDescription() != null) {
            description = request.getDescription();
        }
        String startQuestTriggerScriptId = null;
        if(request.getStartQuestTriggerScriptId() != null) {
            startQuestTriggerScriptId = request.getStartQuestTriggerScriptId();
        }
        String startQuestDoneTriggerScriptId = null;
        if(request.getStartQuestDoneTriggerScriptId() != null) {
            startQuestDoneTriggerScriptId = request.getStartQuestDoneTriggerScriptId();
        }
        String startQuestDoneTriggerQueueNamespaceId = null;
        if(request.getStartQuestDoneTriggerQueueNamespaceId() != null) {
            startQuestDoneTriggerQueueNamespaceId = request.getStartQuestDoneTriggerQueueNamespaceId();
        }
        String completeQuestTriggerScriptId = null;
        if(request.getCompleteQuestTriggerScriptId() != null) {
            completeQuestTriggerScriptId = request.getCompleteQuestTriggerScriptId();
        }
        String completeQuestDoneTriggerScriptId = null;
        if(request.getCompleteQuestDoneTriggerScriptId() != null) {
            completeQuestDoneTriggerScriptId = request.getCompleteQuestDoneTriggerScriptId();
        }
        String completeQuestDoneTriggerQueueNamespaceId = null;
        if(request.getCompleteQuestDoneTriggerQueueNamespaceId() != null) {
            completeQuestDoneTriggerQueueNamespaceId = request.getCompleteQuestDoneTriggerQueueNamespaceId();
        }
        String failedQuestTriggerScriptId = null;
        if(request.getFailedQuestTriggerScriptId() != null) {
            failedQuestTriggerScriptId = request.getFailedQuestTriggerScriptId();
        }
        String failedQuestDoneTriggerScriptId = null;
        if(request.getFailedQuestDoneTriggerScriptId() != null) {
            failedQuestDoneTriggerScriptId = request.getFailedQuestDoneTriggerScriptId();
        }
        String failedQuestDoneTriggerQueueNamespaceId = null;
        if(request.getFailedQuestDoneTriggerQueueNamespaceId() != null) {
            failedQuestDoneTriggerQueueNamespaceId = request.getFailedQuestDoneTriggerQueueNamespaceId();
        }
        String queueNamespaceId = null;
        if(request.getQueueNamespaceId() != null) {
            queueNamespaceId = request.getQueueNamespaceId();
        }
        String keyId = null;
        if(request.getKeyId() != null) {
            keyId = request.getKeyId();
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(name != null) {
            _body.put("name", name);
        }
        if(description != null) {
            _body.put("description", description);
        }
        if(startQuestTriggerScriptId != null) {
            _body.put("startQuestTriggerScriptId", startQuestTriggerScriptId);
        }
        if(startQuestDoneTriggerScriptId != null) {
            _body.put("startQuestDoneTriggerScriptId", startQuestDoneTriggerScriptId);
        }
        if(startQuestDoneTriggerQueueNamespaceId != null) {
            _body.put("startQuestDoneTriggerQueueNamespaceId", startQuestDoneTriggerQueueNamespaceId);
        }
        if(completeQuestTriggerScriptId != null) {
            _body.put("completeQuestTriggerScriptId", completeQuestTriggerScriptId);
        }
        if(completeQuestDoneTriggerScriptId != null) {
            _body.put("completeQuestDoneTriggerScriptId", completeQuestDoneTriggerScriptId);
        }
        if(completeQuestDoneTriggerQueueNamespaceId != null) {
            _body.put("completeQuestDoneTriggerQueueNamespaceId", completeQuestDoneTriggerQueueNamespaceId);
        }
        if(failedQuestTriggerScriptId != null) {
            _body.put("failedQuestTriggerScriptId", failedQuestTriggerScriptId);
        }
        if(failedQuestDoneTriggerScriptId != null) {
            _body.put("failedQuestDoneTriggerScriptId", failedQuestDoneTriggerScriptId);
        }
        if(failedQuestDoneTriggerQueueNamespaceId != null) {
            _body.put("failedQuestDoneTriggerQueueNamespaceId", failedQuestDoneTriggerQueueNamespaceId);
        }
        if(queueNamespaceId != null) {
            _body.put("queueNamespaceId", queueNamespaceId);
        }
        if(keyId != null) {
            _body.put("keyId", keyId);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, CreateNamespaceResult.class);
    }

	/**
	 * クエストを分類するカテゴリーの状態を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetNamespaceStatusResult getNamespaceStatus(GetNamespaceStatusRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FNamespaceFunctionHandler.getNamespaceStatus";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetNamespaceStatusResult.class);
    }

	/**
	 * クエストを分類するカテゴリーを取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetNamespaceResult getNamespace(GetNamespaceRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FNamespaceFunctionHandler.getNamespace";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetNamespaceResult.class);
    }

	/**
	 * クエストを分類するカテゴリーを更新<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public UpdateNamespaceResult updateNamespace(UpdateNamespaceRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FNamespaceFunctionHandler.updateNamespace";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String description = null;
        if(request.getDescription() != null) {
            description = request.getDescription();
        }
        String startQuestTriggerScriptId = null;
        if(request.getStartQuestTriggerScriptId() != null) {
            startQuestTriggerScriptId = request.getStartQuestTriggerScriptId();
        }
        String startQuestDoneTriggerScriptId = null;
        if(request.getStartQuestDoneTriggerScriptId() != null) {
            startQuestDoneTriggerScriptId = request.getStartQuestDoneTriggerScriptId();
        }
        String startQuestDoneTriggerQueueNamespaceId = null;
        if(request.getStartQuestDoneTriggerQueueNamespaceId() != null) {
            startQuestDoneTriggerQueueNamespaceId = request.getStartQuestDoneTriggerQueueNamespaceId();
        }
        String completeQuestTriggerScriptId = null;
        if(request.getCompleteQuestTriggerScriptId() != null) {
            completeQuestTriggerScriptId = request.getCompleteQuestTriggerScriptId();
        }
        String completeQuestDoneTriggerScriptId = null;
        if(request.getCompleteQuestDoneTriggerScriptId() != null) {
            completeQuestDoneTriggerScriptId = request.getCompleteQuestDoneTriggerScriptId();
        }
        String completeQuestDoneTriggerQueueNamespaceId = null;
        if(request.getCompleteQuestDoneTriggerQueueNamespaceId() != null) {
            completeQuestDoneTriggerQueueNamespaceId = request.getCompleteQuestDoneTriggerQueueNamespaceId();
        }
        String failedQuestTriggerScriptId = null;
        if(request.getFailedQuestTriggerScriptId() != null) {
            failedQuestTriggerScriptId = request.getFailedQuestTriggerScriptId();
        }
        String failedQuestDoneTriggerScriptId = null;
        if(request.getFailedQuestDoneTriggerScriptId() != null) {
            failedQuestDoneTriggerScriptId = request.getFailedQuestDoneTriggerScriptId();
        }
        String failedQuestDoneTriggerQueueNamespaceId = null;
        if(request.getFailedQuestDoneTriggerQueueNamespaceId() != null) {
            failedQuestDoneTriggerQueueNamespaceId = request.getFailedQuestDoneTriggerQueueNamespaceId();
        }
        String queueNamespaceId = null;
        if(request.getQueueNamespaceId() != null) {
            queueNamespaceId = request.getQueueNamespaceId();
        }
        String keyId = null;
        if(request.getKeyId() != null) {
            keyId = request.getKeyId();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(description != null) {
            _body.put("description", description);
        }
        if(startQuestTriggerScriptId != null) {
            _body.put("startQuestTriggerScriptId", startQuestTriggerScriptId);
        }
        if(startQuestDoneTriggerScriptId != null) {
            _body.put("startQuestDoneTriggerScriptId", startQuestDoneTriggerScriptId);
        }
        if(startQuestDoneTriggerQueueNamespaceId != null) {
            _body.put("startQuestDoneTriggerQueueNamespaceId", startQuestDoneTriggerQueueNamespaceId);
        }
        if(completeQuestTriggerScriptId != null) {
            _body.put("completeQuestTriggerScriptId", completeQuestTriggerScriptId);
        }
        if(completeQuestDoneTriggerScriptId != null) {
            _body.put("completeQuestDoneTriggerScriptId", completeQuestDoneTriggerScriptId);
        }
        if(completeQuestDoneTriggerQueueNamespaceId != null) {
            _body.put("completeQuestDoneTriggerQueueNamespaceId", completeQuestDoneTriggerQueueNamespaceId);
        }
        if(failedQuestTriggerScriptId != null) {
            _body.put("failedQuestTriggerScriptId", failedQuestTriggerScriptId);
        }
        if(failedQuestDoneTriggerScriptId != null) {
            _body.put("failedQuestDoneTriggerScriptId", failedQuestDoneTriggerScriptId);
        }
        if(failedQuestDoneTriggerQueueNamespaceId != null) {
            _body.put("failedQuestDoneTriggerQueueNamespaceId", failedQuestDoneTriggerQueueNamespaceId);
        }
        if(queueNamespaceId != null) {
            _body.put("queueNamespaceId", queueNamespaceId);
        }
        if(keyId != null) {
            _body.put("keyId", keyId);
        }
		HttpPut http = createHttpPut(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, UpdateNamespaceResult.class);
    }

	/**
	 * クエストを分類するカテゴリーを削除<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DeleteNamespaceResult deleteNamespace(DeleteNamespaceRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FNamespaceFunctionHandler.deleteNamespace";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpDelete http = createHttpDelete(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DeleteNamespaceResult.class);
    }

	/**
	 * クエストグループマスターの一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeQuestGroupModelMastersResult describeQuestGroupModelMasters(DescribeQuestGroupModelMastersRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FQuestGroupModelMasterFunctionHandler.describeQuestGroupModelMasters";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String pageToken = String.valueOf(request.getPageToken());
        String limit = String.valueOf(request.getLimit());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(pageToken != null) {
            _queryString.add(new BasicNameValuePair("pageToken", String.valueOf(pageToken)));
        }
        if(limit != null) {
            _queryString.add(new BasicNameValuePair("limit", String.valueOf(limit)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DescribeQuestGroupModelMastersResult.class);
    }

	/**
	 * クエストグループマスターを新規作成<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public CreateQuestGroupModelMasterResult createQuestGroupModelMaster(CreateQuestGroupModelMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FQuestGroupModelMasterFunctionHandler.createQuestGroupModelMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String name = null;
        if(request.getName() != null) {
            name = request.getName();
        }
        String description = null;
        if(request.getDescription() != null) {
            description = request.getDescription();
        }
        String metadata = null;
        if(request.getMetadata() != null) {
            metadata = request.getMetadata();
        }
        String challengePeriodEventId = null;
        if(request.getChallengePeriodEventId() != null) {
            challengePeriodEventId = request.getChallengePeriodEventId();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(name != null) {
            _body.put("name", name);
        }
        if(description != null) {
            _body.put("description", description);
        }
        if(metadata != null) {
            _body.put("metadata", metadata);
        }
        if(challengePeriodEventId != null) {
            _body.put("challengePeriodEventId", challengePeriodEventId);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, CreateQuestGroupModelMasterResult.class);
    }

	/**
	 * クエストグループマスターを取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetQuestGroupModelMasterResult getQuestGroupModelMaster(GetQuestGroupModelMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FQuestGroupModelMasterFunctionHandler.getQuestGroupModelMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String questGroupName = null;
        if(request.getQuestGroupName() != null) {
            questGroupName = request.getQuestGroupName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(questGroupName != null) {
            _queryString.add(new BasicNameValuePair("questGroupName", String.valueOf(questGroupName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetQuestGroupModelMasterResult.class);
    }

	/**
	 * クエストグループマスターを更新<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public UpdateQuestGroupModelMasterResult updateQuestGroupModelMaster(UpdateQuestGroupModelMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FQuestGroupModelMasterFunctionHandler.updateQuestGroupModelMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String questGroupName = null;
        if(request.getQuestGroupName() != null) {
            questGroupName = request.getQuestGroupName();
        }
        String description = null;
        if(request.getDescription() != null) {
            description = request.getDescription();
        }
        String metadata = null;
        if(request.getMetadata() != null) {
            metadata = request.getMetadata();
        }
        String challengePeriodEventId = null;
        if(request.getChallengePeriodEventId() != null) {
            challengePeriodEventId = request.getChallengePeriodEventId();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(questGroupName != null) {
            _queryString.add(new BasicNameValuePair("questGroupName", String.valueOf(questGroupName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(description != null) {
            _body.put("description", description);
        }
        if(metadata != null) {
            _body.put("metadata", metadata);
        }
        if(challengePeriodEventId != null) {
            _body.put("challengePeriodEventId", challengePeriodEventId);
        }
		HttpPut http = createHttpPut(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, UpdateQuestGroupModelMasterResult.class);
    }

	/**
	 * クエストグループマスターを削除<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DeleteQuestGroupModelMasterResult deleteQuestGroupModelMaster(DeleteQuestGroupModelMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FQuestGroupModelMasterFunctionHandler.deleteQuestGroupModelMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String questGroupName = null;
        if(request.getQuestGroupName() != null) {
            questGroupName = request.getQuestGroupName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(questGroupName != null) {
            _queryString.add(new BasicNameValuePair("questGroupName", String.valueOf(questGroupName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpDelete http = createHttpDelete(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DeleteQuestGroupModelMasterResult.class);
    }

	/**
	 * クエストモデルマスターの一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeQuestModelMastersResult describeQuestModelMasters(DescribeQuestModelMastersRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FQuestModelMasterFunctionHandler.describeQuestModelMasters";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String questGroupName = null;
        if(request.getQuestGroupName() != null) {
            questGroupName = request.getQuestGroupName();
        }
        String pageToken = String.valueOf(request.getPageToken());
        String limit = String.valueOf(request.getLimit());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(questGroupName != null) {
            _queryString.add(new BasicNameValuePair("questGroupName", String.valueOf(questGroupName)));
        }
        if(pageToken != null) {
            _queryString.add(new BasicNameValuePair("pageToken", String.valueOf(pageToken)));
        }
        if(limit != null) {
            _queryString.add(new BasicNameValuePair("limit", String.valueOf(limit)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DescribeQuestModelMastersResult.class);
    }

	/**
	 * クエストモデルマスターを新規作成<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public CreateQuestModelMasterResult createQuestModelMaster(CreateQuestModelMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FQuestModelMasterFunctionHandler.createQuestModelMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String questGroupName = null;
        if(request.getQuestGroupName() != null) {
            questGroupName = request.getQuestGroupName();
        }
        String name = null;
        if(request.getName() != null) {
            name = request.getName();
        }
        String description = null;
        if(request.getDescription() != null) {
            description = request.getDescription();
        }
        String metadata = null;
        if(request.getMetadata() != null) {
            metadata = request.getMetadata();
        }
        ArrayNode contents = null;
        if(request.getContents() != null) {
            List<JsonNode> node = new ArrayList<>();
            for(Contents item : request.getContents()) {
                node.add(item.toJson());
            }
            contents = JsonNodeFactory.instance.arrayNode().addAll(node);
        }
        String challengePeriodEventId = null;
        if(request.getChallengePeriodEventId() != null) {
            challengePeriodEventId = request.getChallengePeriodEventId();
        }
        ArrayNode consumeActions = null;
        if(request.getConsumeActions() != null) {
            List<JsonNode> node = new ArrayList<>();
            for(ConsumeAction item : request.getConsumeActions()) {
                node.add(item.toJson());
            }
            consumeActions = JsonNodeFactory.instance.arrayNode().addAll(node);
        }
        ArrayNode failedAcquireActions = null;
        if(request.getFailedAcquireActions() != null) {
            List<JsonNode> node = new ArrayList<>();
            for(AcquireAction item : request.getFailedAcquireActions()) {
                node.add(item.toJson());
            }
            failedAcquireActions = JsonNodeFactory.instance.arrayNode().addAll(node);
        }
        ArrayNode premiseQuestNames = null;
        if(request.getPremiseQuestNames() != null) {
            List<JsonNode> node = new ArrayList<>();
            for(String item : request.getPremiseQuestNames()) {
                node.add(JsonNodeFactory.instance.textNode(item));
            }
            premiseQuestNames = JsonNodeFactory.instance.arrayNode().addAll(node);
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(questGroupName != null) {
            _queryString.add(new BasicNameValuePair("questGroupName", String.valueOf(questGroupName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(name != null) {
            _body.put("name", name);
        }
        if(description != null) {
            _body.put("description", description);
        }
        if(metadata != null) {
            _body.put("metadata", metadata);
        }
        if(contents != null) {
            _body.put("contents", contents);
        }
        if(challengePeriodEventId != null) {
            _body.put("challengePeriodEventId", challengePeriodEventId);
        }
        if(consumeActions != null) {
            _body.put("consumeActions", consumeActions);
        }
        if(failedAcquireActions != null) {
            _body.put("failedAcquireActions", failedAcquireActions);
        }
        if(premiseQuestNames != null) {
            _body.put("premiseQuestNames", premiseQuestNames);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, CreateQuestModelMasterResult.class);
    }

	/**
	 * クエストモデルマスターを取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetQuestModelMasterResult getQuestModelMaster(GetQuestModelMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FQuestModelMasterFunctionHandler.getQuestModelMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String questGroupName = null;
        if(request.getQuestGroupName() != null) {
            questGroupName = request.getQuestGroupName();
        }
        String questName = null;
        if(request.getQuestName() != null) {
            questName = request.getQuestName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(questGroupName != null) {
            _queryString.add(new BasicNameValuePair("questGroupName", String.valueOf(questGroupName)));
        }
        if(questName != null) {
            _queryString.add(new BasicNameValuePair("questName", String.valueOf(questName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetQuestModelMasterResult.class);
    }

	/**
	 * クエストモデルマスターを更新<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public UpdateQuestModelMasterResult updateQuestModelMaster(UpdateQuestModelMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FQuestModelMasterFunctionHandler.updateQuestModelMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String questGroupName = null;
        if(request.getQuestGroupName() != null) {
            questGroupName = request.getQuestGroupName();
        }
        String questName = null;
        if(request.getQuestName() != null) {
            questName = request.getQuestName();
        }
        String description = null;
        if(request.getDescription() != null) {
            description = request.getDescription();
        }
        String metadata = null;
        if(request.getMetadata() != null) {
            metadata = request.getMetadata();
        }
        ArrayNode contents = null;
        if(request.getContents() != null) {
            List<JsonNode> node = new ArrayList<>();
            for(Contents item : request.getContents()) {
                node.add(item.toJson());
            }
            contents = JsonNodeFactory.instance.arrayNode().addAll(node);
        }
        String challengePeriodEventId = null;
        if(request.getChallengePeriodEventId() != null) {
            challengePeriodEventId = request.getChallengePeriodEventId();
        }
        ArrayNode consumeActions = null;
        if(request.getConsumeActions() != null) {
            List<JsonNode> node = new ArrayList<>();
            for(ConsumeAction item : request.getConsumeActions()) {
                node.add(item.toJson());
            }
            consumeActions = JsonNodeFactory.instance.arrayNode().addAll(node);
        }
        ArrayNode failedAcquireActions = null;
        if(request.getFailedAcquireActions() != null) {
            List<JsonNode> node = new ArrayList<>();
            for(AcquireAction item : request.getFailedAcquireActions()) {
                node.add(item.toJson());
            }
            failedAcquireActions = JsonNodeFactory.instance.arrayNode().addAll(node);
        }
        ArrayNode premiseQuestNames = null;
        if(request.getPremiseQuestNames() != null) {
            List<JsonNode> node = new ArrayList<>();
            for(String item : request.getPremiseQuestNames()) {
                node.add(JsonNodeFactory.instance.textNode(item));
            }
            premiseQuestNames = JsonNodeFactory.instance.arrayNode().addAll(node);
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(questGroupName != null) {
            _queryString.add(new BasicNameValuePair("questGroupName", String.valueOf(questGroupName)));
        }
        if(questName != null) {
            _queryString.add(new BasicNameValuePair("questName", String.valueOf(questName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(description != null) {
            _body.put("description", description);
        }
        if(metadata != null) {
            _body.put("metadata", metadata);
        }
        if(contents != null) {
            _body.put("contents", contents);
        }
        if(challengePeriodEventId != null) {
            _body.put("challengePeriodEventId", challengePeriodEventId);
        }
        if(consumeActions != null) {
            _body.put("consumeActions", consumeActions);
        }
        if(failedAcquireActions != null) {
            _body.put("failedAcquireActions", failedAcquireActions);
        }
        if(premiseQuestNames != null) {
            _body.put("premiseQuestNames", premiseQuestNames);
        }
		HttpPut http = createHttpPut(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, UpdateQuestModelMasterResult.class);
    }

	/**
	 * クエストモデルマスターを削除<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DeleteQuestModelMasterResult deleteQuestModelMaster(DeleteQuestModelMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FQuestModelMasterFunctionHandler.deleteQuestModelMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String questGroupName = null;
        if(request.getQuestGroupName() != null) {
            questGroupName = request.getQuestGroupName();
        }
        String questName = null;
        if(request.getQuestName() != null) {
            questName = request.getQuestName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(questGroupName != null) {
            _queryString.add(new BasicNameValuePair("questGroupName", String.valueOf(questGroupName)));
        }
        if(questName != null) {
            _queryString.add(new BasicNameValuePair("questName", String.valueOf(questName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpDelete http = createHttpDelete(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DeleteQuestModelMasterResult.class);
    }

	/**
	 * 現在有効なクエストマスターのマスターデータをエクスポートします<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public ExportMasterResult exportMaster(ExportMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FCurrentQuestMasterFunctionHandler.exportMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, ExportMasterResult.class);
    }

	/**
	 * 現在有効な現在有効なクエストマスターを取得します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetCurrentQuestMasterResult getCurrentQuestMaster(GetCurrentQuestMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FCurrentQuestMasterFunctionHandler.getCurrentQuestMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetCurrentQuestMasterResult.class);
    }

	/**
	 * 現在有効な現在有効なクエストマスターを更新します<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public UpdateCurrentQuestMasterResult updateCurrentQuestMaster(UpdateCurrentQuestMasterRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FCurrentQuestMasterFunctionHandler.updateCurrentQuestMaster";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String settings = null;
        if(request.getSettings() != null) {
            settings = request.getSettings();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(settings != null) {
            _body.put("settings", settings);
        }
		HttpPut http = createHttpPut(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, UpdateCurrentQuestMasterResult.class);
    }

	/**
	 * クエスト挑戦の一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeProgressesByUserIdResult describeProgressesByUserId(DescribeProgressesByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FProgressFunctionHandler.describeProgressesByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String userId = String.valueOf(request.getUserId());
        String pageToken = String.valueOf(request.getPageToken());
        String limit = String.valueOf(request.getLimit());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(pageToken != null) {
            _queryString.add(new BasicNameValuePair("pageToken", String.valueOf(pageToken)));
        }
        if(limit != null) {
            _queryString.add(new BasicNameValuePair("limit", String.valueOf(limit)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, DescribeProgressesByUserIdResult.class);
    }

	/**
	 * ユーザIDを指定してクエスト挑戦を作成<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public CreateProgressByUserIdResult createProgressByUserId(CreateProgressByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FProgressFunctionHandler.createProgressByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        String questModelId = null;
        if(request.getQuestModelId() != null) {
            questModelId = request.getQuestModelId();
        }
        Boolean force = null;
        if(request.getForce() != null) {
            force = request.getForce();
        }
        ArrayNode config = null;
        if(request.getConfig() != null) {
            List<JsonNode> node = new ArrayList<>();
            for(Config item : request.getConfig()) {
                node.add(item.toJson());
            }
            config = JsonNodeFactory.instance.arrayNode().addAll(node);
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(questModelId != null) {
            _body.put("questModelId", questModelId);
        }
        if(force != null) {
            _body.put("force", force);
        }
        if(config != null) {
            _body.put("config", config);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, CreateProgressByUserIdResult.class);
    }

	/**
	 * クエスト挑戦を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetProgressResult getProgress(GetProgressRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FProgressFunctionHandler.getProgress";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        http.setHeader("X-GS2-ACCESS-TOKEN", request.getAccessToken());
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, GetProgressResult.class);
    }

	/**
	 * ユーザIDを指定してクエスト挑戦を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetProgressByUserIdResult getProgressByUserId(GetProgressByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FProgressFunctionHandler.getProgressByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, GetProgressByUserIdResult.class);
    }

	/**
	 * クエストを開始<br>
	 *   <br>
	 *   同一カテゴリ内でゲームプレイヤーは同時に1つのクエストを実行できます。<br>
	 *   すでに同一カテゴリ内で実行中のクエストがある場合、このAPIはエラーを返します。<br>
	 *   進行中のクエストを取得するAPIを呼び出すことで、クエストを再開するために必要な情報を得ることができます。<br>
	 *   強制的にクエストを開始するには force パラメータに true を指定することで強制的にクエストを開始できます。<br>
	 *   <br>
	 *   クエストが正常に開始できた場合、Progress オブジェクトを応答します。<br>
	 *   Progress オブジェクトはクエストを実行するために必要ないくつかの情報を応答します。<br>
	 *   <br>
	 *   transactionId は実行中のクエスト固有のIDです。<br>
	 *   クエストの完了報告にはこのIDを指定する必要があります。<br>
	 *   <br>
	 *   randomSeed はクエストの内容を決定するために使用できる乱数シードです。<br>
	 *   クエストを開始するたびに異なる乱数が払い出されますので、この値をシード値としてゲームを進行させることで<br>
	 *   クエスト中にアプリケーションを強制終了したとしても同一条件で再開することができます。<br>
	 *   <br>
	 *   rewards にはこのクエストにおいて入手可能な報酬とその数量の"最大値"が得られます。<br>
	 *   クエストの完了報告にも rewards を渡すことができ、そこでクエスト中に実際に入手したアイテムの数量を指定します。<br>
	 *   詳細はクエストの完了報告APIのドキュメントを参照してください。<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public StartResult start(StartRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FProgressFunctionHandler.start";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String questGroupName = null;
        if(request.getQuestGroupName() != null) {
            questGroupName = request.getQuestGroupName();
        }
        String questName = null;
        if(request.getQuestName() != null) {
            questName = request.getQuestName();
        }
        Boolean force = null;
        if(request.getForce() != null) {
            force = request.getForce();
        }
        ArrayNode config = null;
        if(request.getConfig() != null) {
            List<JsonNode> node = new ArrayList<>();
            for(Config item : request.getConfig()) {
                node.add(item.toJson());
            }
            config = JsonNodeFactory.instance.arrayNode().addAll(node);
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(questGroupName != null) {
            _queryString.add(new BasicNameValuePair("questGroupName", String.valueOf(questGroupName)));
        }
        if(questName != null) {
            _queryString.add(new BasicNameValuePair("questName", String.valueOf(questName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(force != null) {
            _body.put("force", force);
        }
        if(config != null) {
            _body.put("config", config);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        http.setHeader("X-GS2-ACCESS-TOKEN", request.getAccessToken());
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, StartResult.class);
    }

	/**
	 * ユーザIDを指定してクエストを開始<br>
	 *   <br>
	 *   同一カテゴリ内でゲームプレイヤーは同時に1つのクエストを実行できます。<br>
	 *   すでに同一カテゴリ内で実行中のクエストがある場合、このAPIはエラーを返します。<br>
	 *   進行中のクエストを取得するAPIを呼び出すことで、クエストを再開するために必要な情報を得ることができます。<br>
	 *   強制的にクエストを開始するには force パラメータに true を指定することで強制的にクエストを開始できます。<br>
	 *   <br>
	 *   クエストが正常に開始できた場合、Progress オブジェクトを応答します。<br>
	 *   Progress オブジェクトはクエストを実行するために必要ないくつかの情報を応答します。<br>
	 *   <br>
	 *   transactionId は実行中のクエスト固有のIDです。<br>
	 *   クエストの完了報告にはこのIDを指定する必要があります。<br>
	 *   <br>
	 *   randomSeed はクエストの内容を決定するために使用できる乱数シードです。<br>
	 *   クエストを開®®始するたびに異なる乱数が払い出されますので、この値をシード値としてゲームを進行させることで<br>
	 *   クエスト中にアプリケーションを強制終了したとしても同一条件で再開することができます。<br>
	 *   <br>
	 *   rewards にはこのクエストにおいて入手可能な報酬とその数量の"最大値"が得られます。<br>
	 *   クエストの完了報告にも rewards を渡すことができ、そこでクエスト中に実際に入手したアイテムの数量を指定します。<br>
	 *   詳細はクエストの完了報告APIのドキュメントを参照してください。<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public StartByUserIdResult startByUserId(StartByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FProgressFunctionHandler.startByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String questGroupName = null;
        if(request.getQuestGroupName() != null) {
            questGroupName = request.getQuestGroupName();
        }
        String questName = null;
        if(request.getQuestName() != null) {
            questName = request.getQuestName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        Boolean force = null;
        if(request.getForce() != null) {
            force = request.getForce();
        }
        ArrayNode config = null;
        if(request.getConfig() != null) {
            List<JsonNode> node = new ArrayList<>();
            for(Config item : request.getConfig()) {
                node.add(item.toJson());
            }
            config = JsonNodeFactory.instance.arrayNode().addAll(node);
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(questGroupName != null) {
            _queryString.add(new BasicNameValuePair("questGroupName", String.valueOf(questGroupName)));
        }
        if(questName != null) {
            _queryString.add(new BasicNameValuePair("questName", String.valueOf(questName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(force != null) {
            _body.put("force", force);
        }
        if(config != null) {
            _body.put("config", config);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, StartByUserIdResult.class);
    }

	/**
	 * クエストを完了<br>
	 *   <br>
	 *   開始時に受け取ったクエストにおいて入手可能な報酬とその数量の"最大値"のうち、クエスト内で実際に入手した報酬を rewards で報告します。<br>
	 *   isComplete にはクエストをクリアできたかを報告します。クエストに失敗した場合、rewards の値は無視してクエストに設定された失敗した場合の報酬が付与されます。<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public EndResult end(EndRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FProgressFunctionHandler.end";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String transactionId = null;
        if(request.getTransactionId() != null) {
            transactionId = request.getTransactionId();
        }
        ArrayNode rewards = null;
        if(request.getRewards() != null) {
            List<JsonNode> node = new ArrayList<>();
            for(Reward item : request.getRewards()) {
                node.add(item.toJson());
            }
            rewards = JsonNodeFactory.instance.arrayNode().addAll(node);
        }
        Boolean isComplete = null;
        if(request.getIsComplete() != null) {
            isComplete = request.getIsComplete();
        }
        ArrayNode config = null;
        if(request.getConfig() != null) {
            List<JsonNode> node = new ArrayList<>();
            for(Config item : request.getConfig()) {
                node.add(item.toJson());
            }
            config = JsonNodeFactory.instance.arrayNode().addAll(node);
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(transactionId != null) {
            _body.put("transactionId", transactionId);
        }
        if(rewards != null) {
            _body.put("rewards", rewards);
        }
        if(isComplete != null) {
            _body.put("isComplete", isComplete);
        }
        if(config != null) {
            _body.put("config", config);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        http.setHeader("X-GS2-ACCESS-TOKEN", request.getAccessToken());
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, EndResult.class);
    }

	/**
	 * ユーザIDを指定してクエストを完了<br>
	 *   <br>
	 *   開始時に受け取ったクエストにおいて入手可能な報酬とその数量の"最大値"のうち、クエスト内で実際に入手した報酬を rewards で報告します。<br>
	 *   isComplete にはクエストをクリアできたかを報告します。クエストに失敗した場合、rewards の値は無視してクエストに設定された失敗した場合の報酬が付与されます。<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public EndByUserIdResult endByUserId(EndByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FProgressFunctionHandler.endByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        String transactionId = null;
        if(request.getTransactionId() != null) {
            transactionId = request.getTransactionId();
        }
        ArrayNode rewards = null;
        if(request.getRewards() != null) {
            List<JsonNode> node = new ArrayList<>();
            for(Reward item : request.getRewards()) {
                node.add(item.toJson());
            }
            rewards = JsonNodeFactory.instance.arrayNode().addAll(node);
        }
        Boolean isComplete = null;
        if(request.getIsComplete() != null) {
            isComplete = request.getIsComplete();
        }
        ArrayNode config = null;
        if(request.getConfig() != null) {
            List<JsonNode> node = new ArrayList<>();
            for(Config item : request.getConfig()) {
                node.add(item.toJson());
            }
            config = JsonNodeFactory.instance.arrayNode().addAll(node);
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(transactionId != null) {
            _body.put("transactionId", transactionId);
        }
        if(rewards != null) {
            _body.put("rewards", rewards);
        }
        if(isComplete != null) {
            _body.put("isComplete", isComplete);
        }
        if(config != null) {
            _body.put("config", config);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, EndByUserIdResult.class);
    }

	/**
	 * クエスト挑戦を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DeleteProgressResult deleteProgress(DeleteProgressRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FProgressFunctionHandler.deleteProgress";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpDelete http = createHttpDelete(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        http.setHeader("X-GS2-ACCESS-TOKEN", request.getAccessToken());
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, DeleteProgressResult.class);
    }

	/**
	 * ユーザIDを指定してクエスト挑戦を削除<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DeleteProgressByUserIdResult deleteProgressByUserId(DeleteProgressByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FProgressFunctionHandler.deleteProgressByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpDelete http = createHttpDelete(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, DeleteProgressByUserIdResult.class);
    }

	/**
	 * スタンプシートでクエストを開始<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public CreateProgressByStampSheetResult createProgressByStampSheet(CreateProgressByStampSheetRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FProgressFunctionHandler.createProgressByStampSheet";
        String stampSheet = null;
        if(request.getStampSheet() != null) {
            stampSheet = request.getStampSheet();
        }
        String keyId = null;
        if(request.getKeyId() != null) {
            keyId = request.getKeyId();
        }
        ArrayNode config = null;
        if(request.getConfig() != null) {
            List<JsonNode> node = new ArrayList<>();
            for(Config item : request.getConfig()) {
                node.add(item.toJson());
            }
            config = JsonNodeFactory.instance.arrayNode().addAll(node);
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(stampSheet != null) {
            _body.put("stampSheet", stampSheet);
        }
        if(keyId != null) {
            _body.put("keyId", keyId);
        }
        if(config != null) {
            _body.put("config", config);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, CreateProgressByStampSheetResult.class);
    }

	/**
	 * クエスト進行の一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeCompletedQuestListsResult describeCompletedQuestLists(DescribeCompletedQuestListsRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FCompletedQuestListFunctionHandler.describeCompletedQuestLists";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String pageToken = String.valueOf(request.getPageToken());
        String limit = String.valueOf(request.getLimit());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(pageToken != null) {
            _queryString.add(new BasicNameValuePair("pageToken", String.valueOf(pageToken)));
        }
        if(limit != null) {
            _queryString.add(new BasicNameValuePair("limit", String.valueOf(limit)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        http.setHeader("X-GS2-ACCESS-TOKEN", request.getAccessToken());
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, DescribeCompletedQuestListsResult.class);
    }

	/**
	 * ユーザIDを指定してクエスト進行の一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeCompletedQuestListsByUserIdResult describeCompletedQuestListsByUserId(DescribeCompletedQuestListsByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FCompletedQuestListFunctionHandler.describeCompletedQuestListsByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        String pageToken = String.valueOf(request.getPageToken());
        String limit = String.valueOf(request.getLimit());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(pageToken != null) {
            _queryString.add(new BasicNameValuePair("pageToken", String.valueOf(pageToken)));
        }
        if(limit != null) {
            _queryString.add(new BasicNameValuePair("limit", String.valueOf(limit)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, DescribeCompletedQuestListsByUserIdResult.class);
    }

	/**
	 * クエスト進行を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetCompletedQuestListResult getCompletedQuestList(GetCompletedQuestListRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FCompletedQuestListFunctionHandler.getCompletedQuestList";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String questGroupName = null;
        if(request.getQuestGroupName() != null) {
            questGroupName = request.getQuestGroupName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(questGroupName != null) {
            _queryString.add(new BasicNameValuePair("questGroupName", String.valueOf(questGroupName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        http.setHeader("X-GS2-ACCESS-TOKEN", request.getAccessToken());
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, GetCompletedQuestListResult.class);
    }

	/**
	 * ユーザIDを指定してクエスト進行を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetCompletedQuestListByUserIdResult getCompletedQuestListByUserId(GetCompletedQuestListByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FCompletedQuestListFunctionHandler.getCompletedQuestListByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String questGroupName = null;
        if(request.getQuestGroupName() != null) {
            questGroupName = request.getQuestGroupName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(questGroupName != null) {
            _queryString.add(new BasicNameValuePair("questGroupName", String.valueOf(questGroupName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, GetCompletedQuestListByUserIdResult.class);
    }

	/**
	 * ユーザIDを指定してクエスト進行を削除<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DeleteCompletedQuestListByUserIdResult deleteCompletedQuestListByUserId(DeleteCompletedQuestListByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FCompletedQuestListFunctionHandler.deleteCompletedQuestListByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String questGroupName = null;
        if(request.getQuestGroupName() != null) {
            questGroupName = request.getQuestGroupName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(questGroupName != null) {
            _queryString.add(new BasicNameValuePair("questGroupName", String.valueOf(questGroupName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpDelete http = createHttpDelete(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, DeleteCompletedQuestListByUserIdResult.class);
    }

	/**
	 * クエストグループの一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeQuestGroupModelsResult describeQuestGroupModels(DescribeQuestGroupModelsRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FQuestGroupModelFunctionHandler.describeQuestGroupModels";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DescribeQuestGroupModelsResult.class);
    }

	/**
	 * クエストグループを取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetQuestGroupModelResult getQuestGroupModel(GetQuestGroupModelRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FQuestGroupModelFunctionHandler.getQuestGroupModel";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String questGroupName = null;
        if(request.getQuestGroupName() != null) {
            questGroupName = request.getQuestGroupName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(questGroupName != null) {
            _queryString.add(new BasicNameValuePair("questGroupName", String.valueOf(questGroupName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetQuestGroupModelResult.class);
    }

	/**
	 * クエストモデルの一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeQuestModelsResult describeQuestModels(DescribeQuestModelsRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FQuestModelFunctionHandler.describeQuestModels";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String questGroupName = null;
        if(request.getQuestGroupName() != null) {
            questGroupName = request.getQuestGroupName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(questGroupName != null) {
            _queryString.add(new BasicNameValuePair("questGroupName", String.valueOf(questGroupName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DescribeQuestModelsResult.class);
    }

	/**
	 * クエストモデルを取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetQuestModelResult getQuestModel(GetQuestModelRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/quest-handler?handler=gs2_quest%2Fhandler%2FQuestModelFunctionHandler.getQuestModel";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String questGroupName = null;
        if(request.getQuestGroupName() != null) {
            questGroupName = request.getQuestGroupName();
        }
        String questName = null;
        if(request.getQuestName() != null) {
            questName = request.getQuestName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(questGroupName != null) {
            _queryString.add(new BasicNameValuePair("questGroupName", String.valueOf(questGroupName)));
        }
        if(questName != null) {
            _queryString.add(new BasicNameValuePair("questName", String.valueOf(questName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetQuestModelResult.class);
    }
}