<doc-view>

<v-layout row wrap>
<v-flex xs12 sm10 lg10>
<v-card class="section-def" v-bind:color="$store.state.currentColor">
<v-card-text class="pa-3">
<v-card class="section-def__card">
<v-card-text>
<dl>
<dt slot=title>Helidon SE DB Client Guide</dt>
<dd slot="desc"><p>This guide describes the features of Helidon&#8217;s DB Client and how to create a sample Helidon SE project
that can be used to run some basic examples using the Helidon DB Client.</p>
</dd>
</dl>
</v-card-text>
</v-card>
</v-card-text>
</v-card>
</v-flex>
</v-layout>


<h2 id="_what_you_need">What You Need</h2>
<div class="section">
<p>For this 15 minute tutorial, you will need the following:</p>


<div class="table__overflow elevation-1  flex sm7
">
<table class="datatable table">
<colgroup>
<col style="width: 50%;">
<col style="width: 50%;">
</colgroup>
<thead>
</thead>
<tbody>
<tr>
<td class="">A Helidon SE Application</td>
<td class="">You can use your own application or use the
 <router-link to="/se/guides/quickstart">Helidon SE Quickstart</router-link> to create a sample application.</td>
</tr>
<tr>
<td class=""><a target="_blank" href="https://www.oracle.com/technetwork/java/javase/downloads">Java&#160;SE&#160;21</a> (<a target="_blank" href="http://jdk.java.net">Open&#160;JDK&#160;21</a>)</td>
<td class="">Helidon requires Java 21+.</td>
</tr>
<tr>
<td class=""><a target="_blank" href="https://maven.apache.org/download.cgi">Maven 3.8+</a></td>
<td class="">Helidon requires Maven 3.8+.</td>
</tr>
<tr>
<td class=""><a target="_blank" href="https://docs.docker.com/install/">Docker 18.09+</a></td>
<td class="">You need Docker if you
want to build and deploy Docker containers.</td>
</tr>
<tr>
<td class=""><a target="_blank" href="https://kubernetes.io/docs/tasks/tools/install-kubectl/">Kubectl 1.16.5+</a></td>
<td class="">If you want to
deploy to Kubernetes, you need <code>kubectl</code> and a Kubernetes cluster (you can
<router-link to="/about/kubernetes">install one on your desktop</router-link>.</td>
</tr>
</tbody>
</table>
</div>

<markup
lang="bash"
title="Verify Prerequisites"
>java -version
mvn --version
docker --version
kubectl version --short</markup>

<markup
lang="bash"
title="Setting JAVA_HOME"
># On Mac
export JAVA_HOME=`/usr/libexec/java_home -v 21`

# On Linux
# Use the appropriate path to your JDK
export JAVA_HOME=/usr/lib/jvm/jdk-21</markup>

</div>


<h2 id="_introduction">Introduction</h2>
<div class="section">
<p>The Helidon DB Client provides a unified API for working with databases.</p>


<h3 id="_main_features">Main Features</h3>
<div class="section">
<p>The main features of  Helidon DB Client are:</p>

<ul class="ulist">
<li>
<p><strong>Unified API for data access and query</strong>:
The API was implemented as a layer above JDBC or MongoDB Java Driver, so any relational databases
with JDBC driver or MongoDB are supported.</p>

</li>
<li>
<p><strong>Observability</strong>:
Support for health checks, metrics and tracing.</p>

</li>
<li>
<p><strong>Portability between relational database drivers</strong>:
Works with native database statements that can be used inline in the code or defined as named statements
in database configuration. By moving the native query code to configuration files, the Helidon DB Client allows you to
switch to another database by changing the configuration files, not the code.</p>

</li>
</ul>

</div>

</div>


<h2 id="_getting_started_with_helidon_db_client">Getting Started with Helidon DB Client</h2>
<div class="section">
<p>This section describes how to configure and use the key features of the Helidon DB Client.</p>


<h3 id="_set_up_the_h2_database">Set Up the H2 Database</h3>
<div class="section">

<h4 id="_from_docker">From Docker</h4>
<div class="section">
<p>Create a new file in <code>helidon-quickstart-se</code> named <code>Dockerfile.h2</code>. It will be used to create the H2 docker image
to run H2 in a container.</p>

<markup
lang="dockerfile"
title="Write the following content into the new file created"
>FROM openjdk:11-jre-slim

ENV H2_VERSION "1.4.199"

ADD "https://repo1.maven.org/maven2/com/h2database/h2/${H2_VERSION}/h2-${H2_VERSION}.jar" /opt/h2.jar

COPY h2.server.properties /root/.h2.server.properties

EXPOSE 8082
EXPOSE 9092

CMD java \
       -cp /opt/h2.jar \
       org.h2.tools.Server \
       -web -webDaemon -webAllowOthers -webPort 8082 \
       -tcp -tcpAllowOthers -tcpPort 9092 \
       -ifNotExists</markup>

<p>Create a new file <code>h2.server.properties</code> in the current directory.</p>

<markup
lang="properties"
title="Copy the properties into the properties file."
>webSSL=false
webAllowOthers=true
webPort=8082
0=Generic H2 (Server)|org.h2.Driver|jdbc\:h2\:tcp\://localhost\:9092/~/test|sa</markup>

<markup
lang="bash"
title="Build the H2 docker image"
>docker build -f Dockerfile.h2 . -t h2db</markup>

<markup
lang="bash"
title="Run the H2 docker image"
>docker run --rm -p 8082:8082 -p 9092:9092 --name=h2 -it h2db</markup>

</div>


<h4 id="_from_the_command_line">From the Command Line</h4>
<div class="section">
<p>A database stores the books from the library. H2 is a java SQL database that is easy to use and lightweight.
If H2 is not installed on your machine, here are few steps to quickly download and set it up:</p>

<ol style="margin-left: 15px;">
<li>
Download the latest H2 version from the official website: <a target="_blank" href="https://www.h2database.com/html/main.html" class="bare">https://www.h2database.com/html/main.html</a>
<ul class="ulist">
<li>
<p>Note: Windows operating system users can download the Windows Installer.</p>

</li>
</ul>

</li>
<li>
Unzip the downloaded file into your directory.
<ul class="ulist">
<li>
<p>Only the h2-{latest-version}.jar, located in the h2/bin folder, will be needed.</p>

</li>
</ul>

</li>
<li>
Open a terminal window and run the following command to start H2:.

</li>
</ol>

<markup
lang="bash"
title="Replace <code>{latest-version}</code> with your current H2 version:"
>java -cp h2-{latest-version}.jar org.h2.tools.Shell -url dbc:h2:~/test -user sa -password "" -sql "" <span class="conum" data-value="1" />
java -jar h2-{latest-version}.jar -webAllowOthers -tcpAllowOthers -web -tcp <span class="conum" data-value="2" /></markup>

<ul class="colist">
<li data-value="1">Pre-create the database (optional if the file <code>~/test</code> already exists)</li>
<li data-value="2">Start the database</li>
</ul>

</div>

</div>


<h3 id="_connect_to_the_database">Connect to the Database</h3>
<div class="section">
<p>Open the console at <a target="_blank" href="http://127.0.0.1:8082" class="bare">http://127.0.0.1:8082</a> in your favorite browser. It displays a login window.
Select <code>Generic H2</code> from <code>Saved Settings</code>. The following settings should be set by default:</p>

<ul class="ulist">
<li>
<p>Driver Class: org.h2.Driver</p>

</li>
<li>
<p>JDBC URL: jdbc:h2:tcp://localhost:9092/~/test</p>

</li>
<li>
<p>User Name: sa</p>

</li>
<li>
<p>Password:</p>

</li>
</ul>

<p>Password must stay empty. Click <strong>Connect</strong>, the browser displays a web page. The database is correctly set and running.</p>

</div>


<h3 id="_create_a_sample_se_project_using_maven_archetype">Create a Sample SE Project Using Maven Archetype</h3>
<div class="section">
<p>Generate the project sources using the Helidon SE Maven archetype.
The result is a simple project that can be used for the examples in this guide.</p>

<markup
lang="bash"
title="Run the Maven archetype:"
>mvn -U archetype:generate -DinteractiveMode=false \
    -DarchetypeGroupId=io.helidon.archetypes \
    -DarchetypeArtifactId=helidon-quickstart-se \
    -DarchetypeVersion=4.0.8 \
    -DgroupId=io.helidon.examples \
    -DartifactId=helidon-quickstart-se \
    -Dpackage=io.helidon.examples.quickstart.se</markup>

<p>A new directory named <code>helidon-quickstart-se</code> is created.</p>

<markup
lang="bash"
title="Enter into this directory:"
>cd helidon-quickstart-se</markup>

</div>


<h3 id="_add_dependencies">Add Dependencies</h3>
<div class="section">
<p>Navigate to the <code>helidon-quickstart-se</code> directory and open the <code>pom.xml</code> file to add the following Helidon dependencies required to use the DB Client:</p>

<markup
lang="xml"
title="Copy these dependencies to pom.xml:"
>&lt;dependencies&gt;
    &lt;!-- ... --&gt;
    &lt;dependency&gt;
        &lt;groupId&gt;io.helidon.dbclient&lt;/groupId&gt;
        &lt;artifactId&gt;helidon-dbclient&lt;/artifactId&gt; <span class="conum" data-value="1" />
    &lt;/dependency&gt;
    &lt;dependency&gt;
        &lt;groupId&gt;io.helidon.dbclient&lt;/groupId&gt;
        &lt;artifactId&gt;helidon-dbclient-jdbc&lt;/artifactId&gt; <span class="conum" data-value="2" />
    &lt;/dependency&gt;
    &lt;dependency&gt;
        &lt;groupId&gt;io.helidon.dbclient&lt;/groupId&gt;
        &lt;artifactId&gt;helidon-dbclient-hikari&lt;/artifactId&gt; <span class="conum" data-value="3" />
    &lt;/dependency&gt;
    &lt;dependency&gt;
        &lt;groupId&gt;io.helidon.integrations.db&lt;/groupId&gt;
        &lt;artifactId&gt;h2&lt;/artifactId&gt; <span class="conum" data-value="4" />
    &lt;/dependency&gt;
    &lt;dependency&gt;
        &lt;groupId&gt;org.slf4j&lt;/groupId&gt;
        &lt;artifactId&gt;slf4j-jdk14&lt;/artifactId&gt;
    &lt;/dependency&gt;
    &lt;dependency&gt;
        &lt;groupId&gt;io.helidon.dbclient&lt;/groupId&gt;
        &lt;artifactId&gt;helidon-dbclient-health&lt;/artifactId&gt; <span class="conum" data-value="5" />
    &lt;/dependency&gt;
    &lt;dependency&gt;
        &lt;groupId&gt;io.helidon.dbclient&lt;/groupId&gt;
        &lt;artifactId&gt;helidon-dbclient-metrics&lt;/artifactId&gt; <span class="conum" data-value="6" />
    &lt;/dependency&gt;
    &lt;dependency&gt;
        &lt;groupId&gt;io.helidon.dbclient&lt;/groupId&gt;
        &lt;artifactId&gt;helidon-dbclient-metrics-hikari&lt;/artifactId&gt;
    &lt;/dependency&gt;
    &lt;dependency&gt;
        &lt;groupId&gt;io.helidon.dbclient&lt;/groupId&gt;
        &lt;artifactId&gt;helidon-dbclient-jsonp&lt;/artifactId&gt; <span class="conum" data-value="7" />
    &lt;/dependency&gt;
    &lt;!-- ... --&gt;
&lt;/dependencies&gt;</markup>

<ul class="colist">
<li data-value="1">DB Client API dependency.</li>
<li data-value="2">Using JDBC driver for this example.</li>
<li data-value="3">Using HikariCP as a connection pool.</li>
<li data-value="4">H2 driver dependency.</li>
<li data-value="5">Support for health check.</li>
<li data-value="6">Support for metrics.</li>
<li data-value="7">Support for Jsonp.</li>
</ul>

</div>


<h3 id="_configure_the_db_client">Configure the DB Client</h3>
<div class="section">
<p>To configure the application, Helidon uses the <code>application.yaml</code>. The DB Client configuration can be joined in the same
file and is located here: <code>src/main/resources</code>.</p>

<markup
lang="yaml"
title="Copy these properties into application.yaml"
>db:
  source: jdbc <span class="conum" data-value="1" />
  connection: <span class="conum" data-value="2" />
    url: "jdbc:h2:tcp://localhost:9092/~/test"
    username: "sa"
    password:
  statements: <span class="conum" data-value="3" />
    health-check: "SELECT 0"
    create-table: "CREATE TABLE IF NOT EXISTS LIBRARY (NAME VARCHAR NOT NULL, INFO VARCHAR NOT NULL)"
    insert-book: "INSERT INTO LIBRARY (NAME, INFO) VALUES (:name, :info)"
    select-book: "SELECT INFO FROM LIBRARY WHERE NAME = ?"
    delete-book: "DELETE FROM LIBRARY WHERE NAME = ?"
  health-check:
    type: "query"
    statementName: "health-check"
  services:
    metrics:
      - type: COUNTER <span class="conum" data-value="4" />
        statement-names: [ "select-book" ]</markup>

<ul class="colist">
<li data-value="1">Source property support two values: jdbc and mongo.</li>
<li data-value="2">Connection detail we used to set up H2.</li>
<li data-value="3">SQL statements to manage the database.</li>
<li data-value="4">Add a counter for metrics only for the <code>select-book</code> statement.</li>
</ul>

<markup
lang="yaml"
title="Copy these properties into application-test.yaml"
>db:
  connection:
    url: "jdbc:h2:mem:test" <span class="conum" data-value="1" /></markup>

<ul class="colist">
<li data-value="1">Override the JDBC URL to use an in-memory database for the tests</li>
</ul>

</div>


<h3 id="_set_up_helidon_db_client">Set Up Helidon DB Client</h3>
<div class="section">
<markup
lang="java"
title="Update <code>Main#main</code>:"
>public static void main(String[] args) {

    // load logging configuration
    LogConfig.configureRuntime();

    // initialize global config from default configuration
    Config config = Config.create();
    Config.global(config);

    DbClient dbClient = DbClient.create(config.get("db")); <span class="conum" data-value="1" />
    Contexts.globalContext().register(dbClient); <span class="conum" data-value="2" />

    HealthObserver healthObserver = HealthObserver.builder()
            .useSystemServices(false)
            .details(true)
            .addCheck(DbClientHealthCheck.create(dbClient, config.get("db.health-check"))) <span class="conum" data-value="3" />
            .build();

    ObserveFeature observe = ObserveFeature.builder()
            .config(config.get("server.features.observe"))
            .addObserver(healthObserver) <span class="conum" data-value="4" />
            .build();

    WebServer server = WebServer.builder()
            .config(config.get("server"))
            .addFeature(observe) <span class="conum" data-value="5" />
            .routing(Main::routing)
            .build()
            .start();

    System.out.println("WEB server is up! http://localhost:" + server.port() + "/simple-greet");
}</markup>

<ul class="colist">
<li data-value="1">Create the DbClient instance</li>
<li data-value="2">Register it in the global context</li>
<li data-value="3">Create an instance of HealthObserver to register a DbClientHealthCheck</li>
<li data-value="4">Add the <code>HealthObserver</code> to the <code>ObserveFeature</code></li>
<li data-value="5">Register the ObserveFeature on the server</li>
</ul>

</div>


<h3 id="_create_the_library_service">Create the Library service</h3>
<div class="section">
<p>Create LibraryService class into <code>io.helidon.examples.quickstart.se</code> package.</p>

<markup
lang="java"
title="LibraryService class looks like this:"
>public class LibraryService implements HttpService {

    private final DbClient dbClient;    <span class="conum" data-value="1" />

    LibraryService() {
        dbClient = Contexts.globalContext()
                .get(DbClient.class)
                .orElseGet(this::newDbClient); <span class="conum" data-value="2" />
        dbClient.execute()
                .namedDml("create-table"); <span class="conum" data-value="3" />
    }

    private DbClient newDbClient() {
        return DbClient.create(Config.global().get("db"));
    }

    @Override
    public void routing(HttpRules rules) {
        // TODO
    }
}</markup>

<ul class="colist">
<li data-value="1">Declare the DB Client instance</li>
<li data-value="2">Initialize the DB Client instance using global config</li>
<li data-value="3">Initialize the database schema</li>
</ul>

<p>As the LibraryService implements <code>io.helidon.webserver.HttpService</code>, the <code>routing(HttpRules)</code> method has to be implemented.
It defines application endpoints and Http request which can be reached by clients.</p>

<markup
lang="java"
title="Add update method to LibraryService"
>@Override
public void routing(HttpRules rules) {
    rules
            .get("/{name}", this::getBook)      <span class="conum" data-value="1" />
            .put("/{name}", this::addBook)      <span class="conum" data-value="2" />
            .delete("/{name}", this::deleteBook)   <span class="conum" data-value="3" />
            .get("/json/{name}", this::getJsonBook); <span class="conum" data-value="4" />
}</markup>

<ul class="colist">
<li data-value="1">Return information about the required book from the database.</li>
<li data-value="2">Add a book to the library.</li>
<li data-value="3">Remove a book from the library.</li>
<li data-value="4">Return the book information in Json format.</li>
</ul>

<p>To summarize, there is one endpoint that can manipulate books.
The number of endpoints and application
features can be changed from these rules by creating or modifying methods. <code>{name}</code> is a path parameter for the book
name. The architecture of the application is defined, so the next step is to create these features.</p>

<markup
lang="java"
title="Add getBook to the LibraryService:"
>private void getBook(ServerRequest request,
                     ServerResponse response) {

    String bookName = request.path()
            .pathParameters()
            .get("name"); <span class="conum" data-value="1" />

    String bookInfo = dbClient.execute()
            .namedGet("select-book", bookName)   <span class="conum" data-value="2" />
            .map(row -&gt; row.column("INFO").asString().get())
            .orElseThrow(() -&gt; new NotFoundException(
                    "Book not found: " + bookName)); <span class="conum" data-value="3" />
    response.send(bookInfo); <span class="conum" data-value="4" />
}</markup>

<ul class="colist">
<li data-value="1">Get the book name from the path in the URL.</li>
<li data-value="2">Helidon DB Client executes the <code>select-book</code> SQL script from application.yaml.</li>
<li data-value="3">Sends 404 HTTP status if no book was found for the given name.</li>
<li data-value="4">Sends book information to the client.</li>
</ul>

<p>The <code>getBook</code> method reach the book from the database and send the information to the client. The name of the book is
located into the url path. If the book is not present in the database, an HTTP 404 is sent back.
The <code>execute()</code> method is called on the dbClient instance to execute one statement.
Nevertheless, it is possible to execute a set of tasks into a single execution unit by using the
 <code>transaction()</code> method.</p>

<p>DbExecute class provides many builders to create statements such as, DML, insert, update, delete, query and get
statements. For each statement there are two builders which can be regrouped in 2 categories. Builders with methods
containing <code>Named</code> keyword, they use a statement defined in the configuration file.</p>

<p>And builders without <code>Named</code> keyword, they use a statement passed as an argument. More information on the Helidon DB
Client <router-link to="/se/dbclient">here</router-link>.</p>

<markup
lang="java"
title="Add getJsonBook to the LibraryService:"
>private void getJsonBook(ServerRequest request,
                         ServerResponse response) {

    String bookName = request.path()
            .pathParameters()
            .get("name");

    JsonObject bookJson = dbClient.execute()
            .namedGet("select-book", bookName)
            .map(row -&gt; row.as(JsonObject.class))
            .orElseThrow(() -&gt; new NotFoundException(
                    "Book not found: " + bookName));
    response.send(bookJson);
}</markup>

<p>Instead of sending the <code>INFO</code> content of the targeted book, the <code>getJsonBook</code> method send the whole row of the
database as a <code>JsonObject</code>.</p>

<markup
lang="java"
title="Add addBook to the LibraryService:"
>private void addBook(ServerRequest request,
                     ServerResponse response) {

    String bookName = request.path()
            .pathParameters()
            .get("name");

    String newValue = request.content().as(String.class);
    dbClient.execute()
            .createNamedInsert("insert-book")
            .addParam("name", bookName) <span class="conum" data-value="1" />
            .addParam("info", newValue)
            .execute();
    response.status(Status.CREATED_201).send(); <span class="conum" data-value="2" />
}</markup>

<ul class="colist">
<li data-value="1">The SQL statement requires the book name and its information. They are provided with <code>addParam</code> method.</li>
<li data-value="2">A new book was added to library, so an HTTP 201 code is returned.</li>
</ul>

<p>When a user adds a new book, it uses HTTP PUT method where the book name is in the URL and the information in the
request content. To catch this content, the information is retrieved as a string and then the DB Client execute the
<code>insert-book</code> script to add the book to the library. It requires two parameters, the book name and information which are
passed to the dbClient thanks to <code>addParam</code> method. An HTTP 201 is sent back as a confirmation.</p>

<markup
lang="java"
title="Add deleteBook to LibraryService:"
>private void deleteBook(ServerRequest request,
                        ServerResponse response) {

    String bookName = request.path()
            .pathParameters()
            .get("name");

    dbClient.execute().namedDelete("delete-book", bookName); <span class="conum" data-value="1" />
    response.status(Status.NO_CONTENT_204).send(); <span class="conum" data-value="2" />
}</markup>

<ul class="colist">
<li data-value="1">Execute SQL script from application.yaml to remove a book from the library by its name.</li>
<li data-value="2">The required book was removed, so an HTTP 204 is sent.</li>
</ul>

<p>To remove a book from the library, use the "delete-book" script in the way than previously. If the book is removed
successfully, an HTTP 204 is sent back.</p>

</div>


<h3 id="_set_up_routing">Set Up Routing</h3>
<div class="section">
<markup
lang="java"
title="Modify the <code>routing</code> method in <code>Main.java</code>:"
>static void routing(HttpRouting.Builder routing) {
    routing
            .register("/greet", new GreetService())
            .register("/library", new LibraryService()) <span class="conum" data-value="1" />
            .get("/simple-greet", (req, res) -&gt; res.send("Hello World!"));
}</markup>

<ul class="colist">
<li data-value="1">Register the LibraryService to the Routing.</li>
</ul>

<p>The library service does not yet exist, but you&#8217;ll create it in the next step of the guide.</p>

</div>

</div>


<h2 id="_build_and_run_the_library_application">Build and Run the Library Application</h2>
<div class="section">
<p>The application is ready to be built and run.</p>

<markup
lang="bash"
title="Run the following to build the application:"
>mvn package</markup>

<p>Note that the tests are passing as the <code>GreetService</code> process was not modified. For the purposes of this demonstration,
we only added independent new content to the existing application.
Make sure H2 is running and start the Helidon quickstart with this command:</p>

<markup
lang="bash"
title="Run the application"
>java -jar target/helidon-quickstart-se.jar</markup>

<p>Once the application starts, check the table LIBRARY is created in the H2 database. To do so, go to the
H2 Server console and LIBRARY table should be present in the left column under <code>jdbc:h2:tcp://localhost:9092/~/test</code>.
If it is not, try to refresh the page, and it should appear.</p>

<p>Use <code>curl</code> to send request to the application:</p>

<markup
lang="bash"
title="Get a book from the library"
>curl -i http://localhost:8080/library/SomeBook</markup>

<markup
lang="listing"
title="HTTP response"
>HTTP/1.1 404 Not Found
Date: Tue, 12 Jan 2021 14:00:48 +0100
transfer-encoding: chunked
connection: keep-alive</markup>

<p>There is currently no book inside the library, so the application returns a 404. Yet the application created an empty
library table. Try to add a new book.</p>

<markup
lang="bash"
title="Add a book from the library"
>curl -i -X PUT -d "Fantasy" http://localhost:8080/library/HarryPotter</markup>

<markup
lang="listing"
title="HTTP response"
>HTTP/1.1 201 Created
Date: Tue, 12 Jan 2021 14:01:08 +0100
transfer-encoding: chunked
connection: keep-alive</markup>

<p>This command creates an HTTP PUT request with the genre <code>Fantasy</code> content at the address
<a target="_blank" href="http://localhost:8080/library/{book-name}" class="bare">http://localhost:8080/library/{book-name}</a>. The 201 code means that Harry Potter book was successfully added to
the library. You can now try to get it !</p>

<markup
lang="bash"
title="Get Harry Potter from the library"
>curl -i http://localhost:8080/library/HarryPotter</markup>

<markup
lang="listing"
title="HTTP response"
>HTTP/1.1 200 OK
Content-Type: text/plain
Date: Tue, 12 Jan 2021 14:01:14 +0100
connection: keep-alive
content-length: 6

Fantasy</markup>

<p>The application accepted the request and returned an HTTP 200 OK with the book genre that was added earlier.</p>

<markup
lang="bash"
title="Get Harry Potter from the library in Json"
>curl -i http://localhost:8080/library/json/HarryPotter</markup>

<markup
lang="listing"
title="HTTP response"
>HTTP/1.1 200 OK
Content-Type: text/plain
Date: Tue, 12 Jan 2021 14:01:14 +0100
connection: keep-alive
content-length: 6

{"INFO":"Fantasy"}</markup>

<p>It returns the database row in a Json format for the Harry Potter book.
Harry Potter can be removed from the library with the following:</p>

<markup
lang="bash"
title="Remove Harry Potter from the library"
>curl -i -X DELETE http://localhost:8080/library/HarryPotter</markup>

<markup
lang="listing"
title="HTTP response"
>HTTP/1.1 204 No Content
Date: Tue, 12 Jan 2021 14:01:22 +0100
connection: keep-alive</markup>

<p>The book had been removed from the library and confirmed by the 204 HTTP status. To check that the book was correctly
deleted, try to get it again.</p>

<markup
lang="bash"
title="Get Harry Potter from the library"
>curl -i http://localhost:8080/library/HarryPotter</markup>

<markup
lang="listing"
title="HTTP response"
>HTTP/1.1 404 Not Found
Date: Tue, 12 Jan 2021 14:00:48 +0100
transfer-encoding: chunked
connection: keep-alive</markup>

<p>The book is not found. We quickly checked, thanks to this suite of command, the application behavior.</p>

<markup
lang="bash"
title="Check the health of your application:"
>curl http://localhost:8080/observe/health</markup>

<markup
lang="json"
title="Response body"
>{
  "status": "UP",
  "checks": [
    {
      "name": "jdbc:h2",
      "status": "UP"
    }
  ]
}</markup>

<p>It confirms that the database is UP.</p>

<markup
lang="bash"
title="Check the metrics of your application:"
>curl -H "Accept: application/json" http://localhost:8080/observe/metrics/application</markup>

<markup
lang="json"
title="Response body"
>{
  "db.counter.select-book" : 4
}</markup>

<p>The select-book statement was invoked four times.</p>


<h3 id="_summary">Summary</h3>
<div class="section">
<p>This guide provided an introduction to the Helidon DB Client&#8217;s key features. If you want to learn more, see the
Helidon DB Client samples in <a target="_blank" href="https://github.com/oracle/helidon/tree/4.0.8/examples/dbclient">GitHub</a>.</p>

</div>

</div>

</doc-view>
