/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.webclient.http2;

import java.time.Duration;
import java.util.Objects;
import java.util.Optional;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;

/**
 * Interface generated from definition. Please add javadoc to the definition interface.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.webclient.http2.Http2ClientProtocolConfigBlueprint")
public interface Http2ClientProtocolConfig extends Http2ClientProtocolConfigBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static Http2ClientProtocolConfig.Builder builder() {
        return new Http2ClientProtocolConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static Http2ClientProtocolConfig.Builder builder(Http2ClientProtocolConfig instance) {
        return Http2ClientProtocolConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static Http2ClientProtocolConfig create(Config config) {
        return Http2ClientProtocolConfig.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static Http2ClientProtocolConfig create() {
        return Http2ClientProtocolConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link Http2ClientProtocolConfig}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends Http2ClientProtocolConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends Http2ClientProtocolConfig> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private boolean ping = false;
        private boolean priorKnowledge = false;
        private Config config;
        private Duration flowControlBlockTimeout = Duration.parse("PT0.1S");
        private Duration pingTimeout = Duration.parse("PT0.5S");
        private int initialWindowSize = 65535;
        private int maxFrameSize = 16384;
        private long maxHeaderListSize = -1;
        private String name = "h2";

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(Http2ClientProtocolConfig prototype) {
            name(prototype.name());
            priorKnowledge(prototype.priorKnowledge());
            maxFrameSize(prototype.maxFrameSize());
            maxHeaderListSize(prototype.maxHeaderListSize());
            initialWindowSize(prototype.initialWindowSize());
            flowControlBlockTimeout(prototype.flowControlBlockTimeout());
            ping(prototype.ping());
            pingTimeout(prototype.pingTimeout());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(Http2ClientProtocolConfig.BuilderBase<?, ?> builder) {
            name(builder.name());
            priorKnowledge(builder.priorKnowledge());
            maxFrameSize(builder.maxFrameSize());
            maxHeaderListSize(builder.maxHeaderListSize());
            initialWindowSize(builder.initialWindowSize());
            flowControlBlockTimeout(builder.flowControlBlockTimeout());
            ping(builder.ping());
            pingTimeout(builder.pingTimeout());
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            config.get("name").as(String.class).ifPresent(this::name);
            config.get("prior-knowledge").as(Boolean.class).ifPresent(this::priorKnowledge);
            config.get("max-frame-size").as(Integer.class).ifPresent(this::maxFrameSize);
            config.get("max-header-list-size").as(Long.class).ifPresent(this::maxHeaderListSize);
            config.get("initial-window-size").as(Integer.class).ifPresent(this::initialWindowSize);
            config.get("flow-control-block-timeout").as(Duration.class).ifPresent(this::flowControlBlockTimeout);
            config.get("ping").as(Boolean.class).ifPresent(this::ping);
            config.get("ping-timeout").as(Duration.class).ifPresent(this::pingTimeout);
            return self();
        }

        /**
         *
         *
         * @param name
         * @return updated builder instance
         * @see #name()
         */
        public BUILDER name(String name) {
            Objects.requireNonNull(name);
            this.name = name;
            return self();
        }

        /**
         * Prior knowledge of HTTP/2 capabilities of the server. If server we are connecting to does not
         * support HTTP/2 and prior knowledge is set to {@code false}, only features supported by HTTP/1 will be available
         * and attempts to use HTTP/2 specific will throw an {@link UnsupportedOperationException}.
         * <h4>Plain text connection</h4>
         * If prior knowledge is set to {@code true}, we will not attempt an upgrade of connection and use prior knowledge.
         * If prior knowledge is set to {@code false}, we will initiate an HTTP/1 connection and upgrade it to HTTP/2,
         * if supported by the server.
         * plaintext connection ({@code h2c}).
         * <h4>TLS protected connection</h4>
         * If prior knowledge is set to {@code true}, we will negotiate protocol using HTTP/2 only, failing if not supported.
         * if prior knowledge is set to {@code false}, we will negotiate protocol using both HTTP/2 and HTTP/1, using the protocol
         * supported by server.
         *
         * @param priorKnowledge whether to use prior knowledge of HTTP/2
         * @return updated builder instance
         * @see #priorKnowledge()
         */
        public BUILDER priorKnowledge(boolean priorKnowledge) {
            this.priorKnowledge = priorKnowledge;
            return self();
        }

        /**
         * Configure initial MAX_FRAME_SIZE setting for new HTTP/2 connections.
         * Maximum size of data frames in bytes the client is prepared to accept from the server.
         * Default value is 2^14(16_384).
         *
         * @param maxFrameSize data frame size in bytes between 2^14(16_384) and 2^24-1(16_777_215)
         * @return updated builder instance
         * @see #maxFrameSize()
         */
        public BUILDER maxFrameSize(int maxFrameSize) {
            this.maxFrameSize = maxFrameSize;
            return self();
        }

        /**
         * Configure initial MAX_HEADER_LIST_SIZE setting for new HTTP/2 connections.
         * Sends to the server the maximum header field section size client is prepared to accept.
         * Defaults to {@code -1}, which means "unconfigured".
         *
         * @param maxHeaderListSize units of octets
         * @return updated builder instance
         * @see #maxHeaderListSize()
         */
        public BUILDER maxHeaderListSize(long maxHeaderListSize) {
            this.maxHeaderListSize = maxHeaderListSize;
            return self();
        }

        /**
         * Configure INITIAL_WINDOW_SIZE setting for new HTTP/2 connections.
         * Sends to the server the size of the largest frame payload client is willing to receive.
         * Defaults to {@value io.helidon.http.http2.WindowSize#DEFAULT_WIN_SIZE}.
         *
         * @param initialWindowSize units of octets
         * @return updated builder instance
         * @see #initialWindowSize()
         */
        public BUILDER initialWindowSize(int initialWindowSize) {
            this.initialWindowSize = initialWindowSize;
            return self();
        }

        /**
         * Timeout for blocking between windows size check iterations.
         *
         * @param flowControlBlockTimeout timeout
         * @return updated builder instance
         * @see #flowControlBlockTimeout()
         */
        public BUILDER flowControlBlockTimeout(Duration flowControlBlockTimeout) {
            Objects.requireNonNull(flowControlBlockTimeout);
            this.flowControlBlockTimeout = flowControlBlockTimeout;
            return self();
        }

        /**
         * Check healthiness of cached connections with HTTP/2.0 ping frame.
         * Defaults to {@code false}.
         *
         * @param ping use ping if true
         * @return updated builder instance
         * @see #ping()
         */
        public BUILDER ping(boolean ping) {
            this.ping = ping;
            return self();
        }

        /**
         * Timeout for ping probe used for checking healthiness of cached connections.
         * Defaults to {@code PT0.5S}, which means 500 milliseconds.
         *
         * @param pingTimeout timeout
         * @return updated builder instance
         * @see #pingTimeout()
         */
        public BUILDER pingTimeout(Duration pingTimeout) {
            Objects.requireNonNull(pingTimeout);
            this.pingTimeout = pingTimeout;
            return self();
        }

        /**
         *
         *
         * @return the name
         */
        public String name() {
            return name;
        }

        /**
         * Prior knowledge of HTTP/2 capabilities of the server. If server we are connecting to does not
         * support HTTP/2 and prior knowledge is set to {@code false}, only features supported by HTTP/1 will be available
         * and attempts to use HTTP/2 specific will throw an {@link UnsupportedOperationException}.
         * <h4>Plain text connection</h4>
         * If prior knowledge is set to {@code true}, we will not attempt an upgrade of connection and use prior knowledge.
         * If prior knowledge is set to {@code false}, we will initiate an HTTP/1 connection and upgrade it to HTTP/2,
         * if supported by the server.
         * plaintext connection ({@code h2c}).
         * <h4>TLS protected connection</h4>
         * If prior knowledge is set to {@code true}, we will negotiate protocol using HTTP/2 only, failing if not supported.
         * if prior knowledge is set to {@code false}, we will negotiate protocol using both HTTP/2 and HTTP/1, using the protocol
         * supported by server.
         *
         * @return the prior knowledge
         */
        public boolean priorKnowledge() {
            return priorKnowledge;
        }

        /**
         * Configure initial MAX_FRAME_SIZE setting for new HTTP/2 connections.
         * Maximum size of data frames in bytes the client is prepared to accept from the server.
         * Default value is 2^14(16_384).
         *
         * @return the max frame size
         */
        public int maxFrameSize() {
            return maxFrameSize;
        }

        /**
         * Configure initial MAX_HEADER_LIST_SIZE setting for new HTTP/2 connections.
         * Sends to the server the maximum header field section size client is prepared to accept.
         * Defaults to {@code -1}, which means "unconfigured".
         *
         * @return the max header list size
         */
        public long maxHeaderListSize() {
            return maxHeaderListSize;
        }

        /**
         * Configure INITIAL_WINDOW_SIZE setting for new HTTP/2 connections.
         * Sends to the server the size of the largest frame payload client is willing to receive.
         * Defaults to {@value io.helidon.http.http2.WindowSize#DEFAULT_WIN_SIZE}.
         *
         * @return the initial window size
         */
        public int initialWindowSize() {
            return initialWindowSize;
        }

        /**
         * Timeout for blocking between windows size check iterations.
         *
         * @return the flow control block timeout
         */
        public Duration flowControlBlockTimeout() {
            return flowControlBlockTimeout;
        }

        /**
         * Check healthiness of cached connections with HTTP/2.0 ping frame.
         * Defaults to {@code false}.
         *
         * @return the ping
         */
        public boolean ping() {
            return ping;
        }

        /**
         * Timeout for ping probe used for checking healthiness of cached connections.
         * Defaults to {@code PT0.5S}, which means 500 milliseconds.
         *
         * @return the ping timeout
         */
        public Duration pingTimeout() {
            return pingTimeout;
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "Http2ClientProtocolConfigBuilder{"
                    + "name=" + name + ","
                    + "priorKnowledge=" + priorKnowledge + ","
                    + "maxFrameSize=" + maxFrameSize + ","
                    + "maxHeaderListSize=" + maxHeaderListSize + ","
                    + "initialWindowSize=" + initialWindowSize + ","
                    + "flowControlBlockTimeout=" + flowControlBlockTimeout + ","
                    + "ping=" + ping + ","
                    + "pingTimeout=" + pingTimeout
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
            new Http2ClientConfigSupport.ProtocolConfigDecorator().decorate(this);
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class Http2ClientProtocolConfigImpl implements Http2ClientProtocolConfig {

            private final boolean ping;
            private final boolean priorKnowledge;
            private final Duration flowControlBlockTimeout;
            private final Duration pingTimeout;
            private final int initialWindowSize;
            private final int maxFrameSize;
            private final long maxHeaderListSize;
            private final String name;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected Http2ClientProtocolConfigImpl(Http2ClientProtocolConfig.BuilderBase<?, ?> builder) {
                this.name = builder.name();
                this.priorKnowledge = builder.priorKnowledge();
                this.maxFrameSize = builder.maxFrameSize();
                this.maxHeaderListSize = builder.maxHeaderListSize();
                this.initialWindowSize = builder.initialWindowSize();
                this.flowControlBlockTimeout = builder.flowControlBlockTimeout();
                this.ping = builder.ping();
                this.pingTimeout = builder.pingTimeout();
            }

            @Override
            public String name() {
                return name;
            }

            @Override
            public boolean priorKnowledge() {
                return priorKnowledge;
            }

            @Override
            public int maxFrameSize() {
                return maxFrameSize;
            }

            @Override
            public long maxHeaderListSize() {
                return maxHeaderListSize;
            }

            @Override
            public int initialWindowSize() {
                return initialWindowSize;
            }

            @Override
            public Duration flowControlBlockTimeout() {
                return flowControlBlockTimeout;
            }

            @Override
            public boolean ping() {
                return ping;
            }

            @Override
            public Duration pingTimeout() {
                return pingTimeout;
            }

            @Override
            public String toString() {
                return "Http2ClientProtocolConfig{"
                        + "name=" + name + ","
                        + "priorKnowledge=" + priorKnowledge + ","
                        + "maxFrameSize=" + maxFrameSize + ","
                        + "maxHeaderListSize=" + maxHeaderListSize + ","
                        + "initialWindowSize=" + initialWindowSize + ","
                        + "flowControlBlockTimeout=" + flowControlBlockTimeout + ","
                        + "ping=" + ping + ","
                        + "pingTimeout=" + pingTimeout
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof Http2ClientProtocolConfig other)) {
                    return false;
                }
                return Objects.equals(name, other.name())
                        && priorKnowledge == other.priorKnowledge()
                        && maxFrameSize == other.maxFrameSize()
                        && maxHeaderListSize == other.maxHeaderListSize()
                        && initialWindowSize == other.initialWindowSize()
                        && Objects.equals(flowControlBlockTimeout, other.flowControlBlockTimeout())
                        && ping == other.ping()
                        && Objects.equals(pingTimeout, other.pingTimeout());
            }

            @Override
            public int hashCode() {
                return Objects.hash(name, priorKnowledge, maxFrameSize, maxHeaderListSize, initialWindowSize, flowControlBlockTimeout, ping, pingTimeout);
            }

        }

    }

    /**
     * Fluent API builder for {@link Http2ClientProtocolConfig}.
     */
    class Builder extends Http2ClientProtocolConfig.BuilderBase<Http2ClientProtocolConfig.Builder, Http2ClientProtocolConfig> implements io.helidon.common.Builder<Http2ClientProtocolConfig.Builder, Http2ClientProtocolConfig> {

        private Builder() {
        }

        @Override
        public Http2ClientProtocolConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new Http2ClientProtocolConfigImpl(this);
        }

        @Override
        public Http2ClientProtocolConfig build() {
            return buildPrototype();
        }

    }

}
