/*
 * Decompiled with CFR 0.152.
 */
package pl.decerto.hyperon.common.security;

import com.google.common.collect.ImmutableSet;
import java.util.Optional;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.ConditionContext;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.ConfigurationCondition;
import org.springframework.core.type.AnnotatedTypeMetadata;
import org.springframework.security.authentication.AccountStatusException;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.DisabledException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.session.SessionRegistry;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import pl.decerto.hyperon.common.security.AuthResult;
import pl.decerto.hyperon.common.security.LoginAttemptHolder;
import pl.decerto.hyperon.common.security.LoginConfiguration;
import pl.decerto.hyperon.common.security.LoginEventService;
import pl.decerto.hyperon.common.security.MppUserDetails;
import pl.decerto.hyperon.common.security.SimpleAuthentication;
import pl.decerto.hyperon.common.security.UserRoleChecker;
import pl.decerto.hyperon.common.security.activedirectory.exception.EmptyFirstOrLastNameException;
import pl.decerto.hyperon.common.security.activedirectory.exception.InternalUserException;
import pl.decerto.hyperon.common.security.activedirectory.exception.NoRoleAssignedException;
import pl.decerto.hyperon.common.security.activedirectory.exception.NotUniqueEmailException;
import pl.decerto.hyperon.common.security.activedirectory.exception.NotUniqueLoginException;
import pl.decerto.hyperon.common.security.domain.LoginEvent;
import pl.decerto.hyperon.common.security.domain.SystemRights;
import pl.decerto.hyperon.common.security.dto.SystemUser;
import pl.decerto.hyperon.common.utils.Messages;

@Component
@Conditional(value={SecurityCondition.class})
public class AccessGateway {
    private static final Logger log = LoggerFactory.getLogger(AccessGateway.class);
    private static final String UNAUTHORIZED_MESSAGE = Messages.message("authorization.unauthorized");
    private static final ImmutableSet<Class<? extends AuthenticationException>> AUTHENTICATION_EXCEPTIONS = ImmutableSet.of(BadCredentialsException.class, NoRoleAssignedException.class, NotUniqueEmailException.class, NotUniqueLoginException.class, InternalUserException.class, EmptyFirstOrLastNameException.class, (Object[])new Class[0]);
    private final AuthenticationManager authenticationManager;
    private final UserDetailsService userService;
    private final LoginEventService loginService;
    private final LoginAttemptHolder loginAttemptHolder;
    private final Optional<SessionRegistry> sessionRegistry;

    @Autowired
    public AccessGateway(AuthenticationManager authenticationManager, UserDetailsService userService, LoginEventService loginService, LoginConfiguration loginConfiguration, Optional<SessionRegistry> sessionRegistry) {
        this.authenticationManager = authenticationManager;
        this.userService = userService;
        this.loginService = loginService;
        this.loginAttemptHolder = new LoginAttemptHolder(loginConfiguration.getMaxLoginAttempts(), loginConfiguration.getCoolDown());
        this.sessionRegistry = sessionRegistry;
    }

    public AuthResult authenticate(String login, String password) {
        log.debug("Authenticate and role check for {}", (Object)login);
        if (StringUtils.isBlank(login) || StringUtils.isBlank(password)) {
            return new AuthResult(false, Messages.message("authorization.loginAndPasswordShouldBeFilled"));
        }
        if (this.loginAttemptHolder.reachedAttemptsLimit(login)) {
            return this.createCoolDownAuthResult();
        }
        UsernamePasswordAuthenticationToken token = new UsernamePasswordAuthenticationToken((Object)login, (Object)password);
        log.trace("Attempt to authenticate with token {} ", (Object)token);
        try {
            return this.authenticate(login, token);
        }
        catch (AuthenticationException e) {
            String message = this.getAuthExceptionMessage(login, e);
            return new AuthResult(false, message);
        }
    }

    private String getAuthExceptionMessage(String login, AuthenticationException e) {
        this.saveFailureLoginEvent(login, (Exception)((Object)e));
        if (this.isKnownException(e)) {
            log.info("Authentication failed, reason:{}", (Object)e.getMessage());
            return ((Object)((Object)e)).getClass().isAssignableFrom(BadCredentialsException.class) ? Messages.message("authorization.incorrectPassword") : e.getMessage();
        }
        log.warn("Authentication failed", e);
        return Messages.message("authorization.incorrectPassword");
    }

    private AuthResult createCoolDownAuthResult() {
        return new AuthResult(false, Messages.message("authorization.login.attempt.cooldown", this.loginAttemptHolder.getMaxAttempts(), this.loginAttemptHolder.getFailureCoolDown()));
    }

    private AuthResult authenticate(String login, UsernamePasswordAuthenticationToken token) {
        this.loginAttemptHolder.incrementAttempt(login);
        Authentication authentication = this.authenticationManager.authenticate((Authentication)token);
        SecurityContextHolder.clearContext();
        SecurityContextHolder.getContext().setAuthentication(authentication);
        String sessionId = RequestContextHolder.currentRequestAttributes().getSessionId();
        Optional<MppUserDetails> userDetails = this.getCurrentUser();
        if (this.hasAccess()) {
            this.loginAttemptHolder.reset(login);
            userDetails.ifPresent(user -> {
                this.saveSuccessLoginEvent(user.getUser());
                this.registerNewSession((MppUserDetails)user);
            });
            return new AuthResult(true, Messages.message("authorization.passwordAndLoginMatch"));
        }
        userDetails.ifPresent(user -> this.saveFailureLoginEvent(user.getUser()));
        return new AuthResult(false, UNAUTHORIZED_MESSAGE);
    }

    private Optional<MppUserDetails> getCurrentUser() {
        return UserRoleChecker.currentUser();
    }

    private void registerNewSession(MppUserDetails user) {
        String sessionId = RequestContextHolder.currentRequestAttributes().getSessionId();
        this.sessionRegistry.ifPresent(reg -> reg.registerNewSession(sessionId, (Object)user.getUser()));
    }

    private boolean isKnownException(AuthenticationException e) {
        Class<DisabledException> clazz = ((Object)((Object)e)).getClass();
        if (this.isInstanceOfAuthenticationExceptions(clazz)) {
            return true;
        }
        return clazz.isAssignableFrom(DisabledException.class);
    }

    public AuthResult authenticateOmittingPassword(String username) {
        log.debug("Only role checking for {}", (Object)username);
        try {
            UserDetails userDetails = this.userService.loadUserByUsername(username);
            SecurityContextHolder.clearContext();
            SecurityContextHolder.getContext().setAuthentication((Authentication)new SimpleAuthentication(userDetails));
            return this.getCurrentUser().map(user -> this.authorizeUser((MppUserDetails)user, username)).orElseGet(() -> new AuthResult(false, UNAUTHORIZED_MESSAGE));
        }
        catch (Exception e) {
            this.saveFailureLoginEvent(username, e);
            log.warn("Authentication fail: {}", (Object)e.getMessage());
            return new AuthResult(false, UNAUTHORIZED_MESSAGE);
        }
    }

    public AuthResult authorizeUser(MppUserDetails user, String username) {
        try {
            if (this.hasAccess()) {
                this.saveSuccessLoginEvent(user.getUser());
                return new AuthResult(true, Messages.message("authorization.passwordAndLoginMatch"));
            }
            this.saveFailureLoginEvent(user.getUser());
            return new AuthResult(false, UNAUTHORIZED_MESSAGE);
        }
        catch (Exception e) {
            this.saveFailureLoginEvent(username, e);
            log.warn("Authorization failed: {}", (Object)e.getMessage());
            return new AuthResult(false, UNAUTHORIZED_MESSAGE);
        }
    }

    private boolean hasAccess() {
        return UserRoleChecker.hasRight(SystemRights.MPP_ACCESS.name());
    }

    private void saveSuccessLoginEvent(SystemUser user) {
        this.loginService.saveLoginEvent(user, LoginEvent.Type.SUCCESS);
    }

    private void saveFailureLoginEvent(SystemUser user) {
        this.loginService.saveLoginEvent(user, LoginEvent.Type.UNAUTHORIZED_ACCESS);
    }

    private void saveFailureLoginEvent(String username, Exception e) {
        try {
            MppUserDetails mppUser = (MppUserDetails)this.userService.loadUserByUsername(username);
            if (e instanceof AccountStatusException) {
                this.loginService.saveLoginEvent(mppUser.getUser(), LoginEvent.Type.ACCOUNT_INACTIVE);
            } else if (this.isInstanceOfAuthenticationExceptions(e.getClass())) {
                this.loginService.saveLoginEvent(mppUser.getUser(), LoginEvent.Type.BAD_CREDENTIALS);
            }
        }
        catch (UsernameNotFoundException ex) {
            log.error("username:{} not found", (Object)ex.getMessage());
        }
    }

    private boolean isInstanceOfAuthenticationExceptions(Class clazz) {
        return AUTHENTICATION_EXCEPTIONS.stream().anyMatch(e -> e.isAssignableFrom(clazz));
    }

    static class SecurityCondition
    implements ConfigurationCondition {
        SecurityCondition() {
        }

        @Override
        public ConfigurationCondition.ConfigurationPhase getConfigurationPhase() {
            return ConfigurationCondition.ConfigurationPhase.PARSE_CONFIGURATION;
        }

        @Override
        public boolean matches(ConditionContext context, AnnotatedTypeMetadata metadata) {
            return context.getEnvironment().getProperty("higson.runtime-rest.security.type") != null;
        }
    }
}

