package com.hippoagent.activities;

import android.content.Intent;
import androidx.databinding.DataBindingUtil;
import android.os.Bundle;
import androidx.annotation.Nullable;
import androidx.core.widget.NestedScrollView;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;
import android.text.TextUtils;
import android.view.MenuItem;
import android.view.View;

import com.hippoagent.HippoApplication;
import com.hippoagent.R;
import com.hippoagent.adapters.BroadcastDetailAdapter;
import com.hippoagent.adapters.OnItemClickListener;
import com.hippoagent.databinding.ActivityBroadcastDetailsBinding;
import com.hippoagent.datastructure.FuguAppConstant;
import com.hippoagent.datastructure.MessageMode;
import com.hippoagent.langs.Restring;
import com.hippoagent.model.BroadCastObjectModel;
import com.hippoagent.model.BroadcastResponse;
import com.hippoagent.model.BroadcastUser;
import com.hippoagent.model.Conversation;
import com.hippoagent.retrofit.APIError;
import com.hippoagent.retrofit.CommonParams;
import com.hippoagent.retrofit.ResponseResolver;
import com.hippoagent.retrofit.RestClient;
import com.hippoagent.utils.Constants;
import com.hippoagent.utils.DateUtils;
import com.hippoagent.utils.filelogger.Logger;
import com.hippoagent.utils.Utils;
import com.google.gson.Gson;

import java.util.ArrayList;

/**
 * Created by gurmail on 12/02/19.
 *
 * @author gurmail
 */
public class BroadcastDetailsActivity extends FuguBaseActivity implements FuguAppConstant, OnItemClickListener {

    private static final String TAG = BroadcastDetailsActivity.class.getSimpleName();
    private ActivityBroadcastDetailsBinding detailsBinding;
    private BroadcastDetailAdapter detailAdapter;
    private BroadCastObjectModel objectModel;
    private ArrayList<BroadcastUser> broadcastUsers = new ArrayList<>();
    private String STANDARD_DATE_FORMAT_TZ = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'";
    private String OUTPUT_FORMAT = "MMM dd, yyyy, hh:mm a";

    private int pastVisiblesItems, visibleItemCount, totalItemCount;
    private boolean hasMorePages;
    private boolean isPagingApiInProgress;

    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        detailsBinding = DataBindingUtil.setContentView(BroadcastDetailsActivity.this, R.layout.activity_broadcast_details);

        setSupportActionBar(detailsBinding.myToolbar);
        setToolbar(detailsBinding.myToolbar, getResources().getString(R.string.broadcast_detail));

        objectModel = (BroadCastObjectModel) getIntent().getSerializableExtra("objectModel");

        detailsBinding.textViewFromValue.setText(objectModel.getFullName());
        String localDate = DateUtils.getInstance().convertToLocal(objectModel.getCreatedAt(), STANDARD_DATE_FORMAT_TZ, OUTPUT_FORMAT);
        detailsBinding.textViewDateValue.setText(localDate);
        detailsBinding.textViewTitleValue.setText(objectModel.getBroadcastTitle());
        detailsBinding.receipientTxt.setText(Restring.getString(this, R.string.recipients));

        Utils.setColor(this, detailsBinding.textViewMsgValue, objectModel.getMessage());

        if(!TextUtils.isEmpty(objectModel.getFallbackText())) {
            detailsBinding.textViewFallbackValue.setText(objectModel.getFallbackText());
            detailsBinding.rlFallback.setVisibility(View.VISIBLE);
        } else {
            detailsBinding.rlFallback.setVisibility(View.GONE);
        }

        if(!TextUtils.isEmpty(objectModel.getBroadcastType())) {
            detailsBinding.layoutType.setVisibility(View.VISIBLE);
            if(objectModel.getBroadcastType().equalsIgnoreCase("email")) {
                detailsBinding.textViewTypeValue.setText(Restring.getString(this, R.string.email));
            } else {
                detailsBinding.textViewTypeValue.setText(Restring.getString(this, R.string.hippo_in_app));
            }
        } else {
            detailsBinding.layoutType.setVisibility(View.GONE);
        }

        detailAdapter = new BroadcastDetailAdapter(detailsBinding.recyclerView, this);

        LinearLayoutManager layoutManager = new LinearLayoutManager(this);
        layoutManager.setOrientation(LinearLayoutManager.VERTICAL);
        detailsBinding.recyclerView.setLayoutManager(layoutManager);

        detailsBinding.recyclerView.setAdapter(detailAdapter);
        getBroadcastStatusList(0);

        detailsBinding.nestedScrollView.setOnScrollChangeListener((NestedScrollView.OnScrollChangeListener) (v, scrollX, scrollY, oldScrollX, oldScrollY) -> {
            if(v.getChildAt(v.getChildCount() - 1) != null) {
                if ((scrollY >= (v.getChildAt(v.getChildCount() - 1).getMeasuredHeight() - v.getMeasuredHeight())) &&
                        scrollY > oldScrollY) {
                    //code to fetch more data for endless scrolling
                    visibleItemCount = layoutManager.getChildCount();
                    totalItemCount = layoutManager.getItemCount();
                    pastVisiblesItems = layoutManager.findFirstVisibleItemPosition();

                    if (!isPagingApiInProgress && hasMorePages) {
                        if ((visibleItemCount + pastVisiblesItems) >= totalItemCount) {
                            getBroadcastStatusList(broadcastUsers.size());
                        }
                    }
                }
            }
        });

        detailsBinding.recyclerView.addOnScrollListener(new RecyclerView.OnScrollListener() {

            @Override
            public void onScrolled(RecyclerView recyclerView, int dx, int dy) {
                super.onScrolled(recyclerView, dx, dy);
                if (dy > 0) {
                    visibleItemCount = layoutManager.getChildCount();
                    totalItemCount = layoutManager.getItemCount();
                    pastVisiblesItems = layoutManager.findFirstVisibleItemPosition();

                    if (!isPagingApiInProgress && hasMorePages) {
                        if ((visibleItemCount + pastVisiblesItems) >= totalItemCount) {
                            getBroadcastStatusList(broadcastUsers.size());
                        }
                    }
                }
            }
        });
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        if (item.getItemId() == android.R.id.home) {
            onBackPressed(); // close this context and return to preview context (if there is any)
        }
        return true;//super.onOptionsItemSelected(item);
    }

    @Override
    public void onBackPressed() {
        super.onBackPressed();
    }

    @Override
    public void onClickItem(int position) {
        String channelId = broadcastUsers.get(position).getRepliedOn();
        if(!TextUtils.isEmpty(channelId)) {
            Conversation conversation = new Conversation();
            conversation.setChannelId(Integer.parseInt(channelId));

            //conversation.setLabel(label);
            conversation.setUserId(broadcastUsers.get(position).getUserId());
            conversation.setUnreadCount(broadcastUsers.get(position).getUnreadCount());
            conversation.setStatus(MessageMode.OPEN_CHAT.getOrdinal());
            conversation.setDisableReply(0);

            Intent chatIntent = new Intent(BroadcastDetailsActivity.this, FuguChatActivity.class);
            chatIntent.putExtra(Constants.CONVERSATION, new Gson().toJson(conversation, Conversation.class));
            startActivityForResult(chatIntent, 100);
        }
    }

    private void getBroadcastStatusList(int offset) {
        if(offset != 0) {
            isPagingApiInProgress= true;
            detailAdapter.showPaginationProgressBar(true, true);
        }
        CommonParams commonParams = new CommonParams.Builder()
                .add(Constants.ACCESS_TOKEN, HippoApplication.getInstance().getUserData().getAccessToken())
                .add("channel_id", objectModel.getChannelId())
                .add("page_offset", offset)
                .build();

        Logger.INSTANCE.apiRequest("/api/broadcast/broadcastStatus", new Gson().toJson(commonParams.getMap()));
        RestClient.getApiInterface().broadcastStatus(commonParams.getMap()).enqueue(new ResponseResolver<BroadcastResponse>(this, offset == 0, false) {
            @Override
            public void success(BroadcastResponse broadcastResponse) {
                Logger.INSTANCE.apiResponse("/api/broadcast/broadcastStatus", "");
                if (offset != 0) {
                    isPagingApiInProgress = false;
                    detailAdapter.showPaginationProgressBar(false, true);
                }

                if(broadcastUsers == null)
                    broadcastUsers = new ArrayList<>();

                if (offset == 0)
                    broadcastUsers.clear();

                broadcastUsers.addAll(broadcastResponse.getData().getBroadcastUsers());

                detailAdapter.setData(broadcastUsers);
                hasMorePages = broadcastResponse.getData().getBroadcastUsers().size() == broadcastResponse.getData().getPageSize();

            }

            @Override
            public void failure(APIError error) {
                Logger.INSTANCE.apiFailed("/api/broadcast/broadcastStatus", error.getMessage());
            }
        });

    }
}
