package faye;

import android.os.Handler;
import android.os.HandlerThread;
import android.os.Message;

import com.google.gson.Gson;
import com.hippoagent.utils.Log;
import com.hippoagent.utils.filelogger.Logger;

import org.java_websocket.WebSocketImpl;
import org.java_websocket.exceptions.WebsocketNotConnectedException;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.net.Socket;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.channels.NotYetConnectedException;
import java.util.HashSet;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocketFactory;

/**
 * Created by Bhavya Rattan on 01/05/17
 * Click Labs
 * bhavya.rattan@click-labs.com
 */

public class FuguAgentFayeClient {

    private static final String LOG_TAG = FuguAgentFayeClient.class.getSimpleName();

    private FuguAgentWebSocket mFuguAgentWebSocket = null;
    private FuguAgentFayeClientListener mManagerListener = null;
    private FayeServiceListener serviceListener = null;
    private HashSet<String> mChannels;
    private String mServerUrl = "";
    private boolean mFayeConnected = false;
    private boolean mIsConnectedServer = false;
    private FuguAgentMetaMessage mFuguAgentMetaMessage;
    private Handler mMessageHandler;

    public FuguAgentFayeClient(String url, FuguAgentMetaMessage meta) {
        Log.w("faye url", "faye url = " + url);
        mServerUrl = url;
        mFuguAgentMetaMessage = meta;
        mChannels = new HashSet<String>();
    }

    {
        HandlerThread thread = new HandlerThread("FayeHandler");
        thread.start();
        mMessageHandler = new Handler(thread.getLooper()) {
            @Override
            public void handleMessage(Message msg) {
                super.handleMessage(msg);
                switch (msg.what) {
                    case FuguAgentWebSocket.ON_OPEN:
                        Log.i(LOG_TAG, "onOpen() executed");
                        mIsConnectedServer = true;
                        handShake();
                        break;
                    case FuguAgentWebSocket.ON_CLOSE:
                        Log.i(LOG_TAG, "onClosed() executed");
                        mIsConnectedServer = false;
                        mFayeConnected = false;
                        if (mManagerListener != null)
                            mManagerListener.onDisconnectedServer(FuguAgentFayeClient.this);
                        if (serviceListener != null)
                            serviceListener.onDisconnectedServer(FuguAgentFayeClient.this);

                        Logger.INSTANCE.fayeDisconnected();
                        break;
                    case FuguAgentWebSocket.ON_MESSAGE:
                        try {
                            Log.i(LOG_TAG, "onMessage executed");
                            handleFayeMessage((String) msg.obj);
                        } catch (NotYetConnectedException e) {
                            // Do noting
                            e.printStackTrace();
                        }
                        break;
                    case FuguAgentWebSocket.ON_PONG:
                        if (mManagerListener != null)
                            mManagerListener.onPongReceived();
                        if (serviceListener != null)
                            serviceListener.onPongReceived();
                        break;
                    case FuguAgentWebSocket.ON_NOT_CONNECTED:
                        if (mManagerListener != null)
                            mManagerListener.onNotConnected();

                        break;
                }
            }
        };
    }

    /* Public Methods */

    public FuguAgentFayeClientListener getmManagerListener() {
        return mManagerListener;
    }

    public void setmManagerListener(FuguAgentFayeClientListener mManagerListener) {
        this.mManagerListener = mManagerListener;
    }

    public FayeServiceListener getServiceListener() {
        return serviceListener;
    }

    public void setServiceListener(FayeServiceListener serviceListener) {
        this.serviceListener = serviceListener;
    }

    public void addChannel(String channel) {
        mChannels.add(channel);
    }

    public boolean isConnectedServer() {
        return mIsConnectedServer;
    }

    public boolean isFayeConnected() {
        return mFayeConnected;
    }

    public void connectServer() {
        openWebSocketConnection();
    }

    public void disconnectServer() {
        Log.e("disconnectServer", "disconnectServer");
        for (String channel : mChannels) {
            unsubscribe(channel);
        }
        mChannels.clear();
        disconnect();
    }

    public void subscribeChannel(String channel) {
        Log.e("subscribeChannel", "subscribeChannel");
        //unsubscribe(channel);
        mChannels.add(channel);
        subscribe(channel);
        Logger.INSTANCE.fayeSubscribed(channel);
        Log.v("channel------>>>>>>>>>>--------------", channel);
    }

    public void subscribeToChannels(String... channels) {
        for (String channel : channels) {
            mChannels.add(channel);
            subscribe(channel);
        }
    }

    public void unsubscribeChannel(String channel) {
        Log.e("unsubscribeChannel", "unsubscribeChannel");
        if (mChannels.contains(channel)) {
            unsubscribe(channel);
            mChannels.remove(channel);
            Logger.INSTANCE.fayeUnsubscribed(channel);
        }
    }

    public void unsubscribeChannels(String... channels) {
        Log.e("unsubscribeChannels", "unsubscribeChannels");
        for (String channel : channels) {
            unsubscribe(channel);
        }
    }

    public void unsubscribeAll() {
        Log.e("unsubscribeAll", "unsubscribeAll");
        for (String channel : mChannels) {
            unsubscribe(channel);
            Logger.INSTANCE.fayeUnsubscribed(channel);
        }
    }

    public void publish(String channel, JSONObject data, boolean b) {
        publish(channel, data, null, null);
    }

    public void publish(String channel, JSONObject data) {
        publish(channel, data, false);
    }

    public void publish(String channel, JSONObject data, String ext, String id) {
        try {
            android.util.Log.w("data sent", channel + " <<<<    channel + data = " + new Gson().toJson(data));
            String publish = mFuguAgentMetaMessage.publish(channel, data, ext, id);
            mFuguAgentWebSocket.send(publish);
        } catch (JSONException e) {
            Log.e(LOG_TAG, "Build publish message to JSON error" + e);
        } catch (Exception e) {
            Log.e(LOG_TAG, "Build publish message to error" + e);
        }
    }

    /* Private Methods */
    private Socket getSSLWebSocket() {
        SSLContext sslContext = null;
        try {
            sslContext = SSLContext.getInstance("SSL");
            sslContext.init(null, null, null);
            SSLSocketFactory factory = sslContext.getSocketFactory();
            return factory.createSocket();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    private void openWebSocketConnection() {
        // Clean up any existing socket
        WebSocketImpl.DEBUG = false;
        if (mFuguAgentWebSocket != null) {
            mFuguAgentWebSocket.close();
        }
        try {
            URI uri = new URI(mServerUrl);
            mFuguAgentWebSocket = new FuguAgentWebSocket(uri, mMessageHandler);
            mFuguAgentWebSocket.setConnectionLostTimeout(60);
            if (uri.getScheme().equals("https") || uri.getScheme().equals("wss")) {
                mFuguAgentWebSocket.setSocket(getSSLWebSocket());
            }
            mFuguAgentWebSocket.connect();
        } catch (URISyntaxException e) {
            Log.e(LOG_TAG, "Server URL error" + e);
        }
    }

    private void closeWebSocketConnection() {
        if (mFuguAgentWebSocket != null) {
            mFuguAgentWebSocket.close();
        }
    }

    private void handShake() {
        try {
            String handshake = mFuguAgentMetaMessage.handShake();
            mFuguAgentWebSocket.send(handshake);
        } catch (JSONException e) {
            Log.e(LOG_TAG, "HandShake message error" + e);
        } catch (Exception e) {
            Log.e(LOG_TAG, "HandShake message error" + e);
            e.printStackTrace();
        }
    }

    private void subscribe(String channel) {
        try {
            String subscribe = mFuguAgentMetaMessage.subscribe(channel);
            mFuguAgentWebSocket.send(subscribe);
            Log.v("Channel subscribe", "Successfully");
        } catch (WebsocketNotConnectedException e) {
            if (mManagerListener != null) {
                mManagerListener.onWebSocketError();
            }
            if (serviceListener != null)
                serviceListener.onWebSocketError();
            mFayeConnected = false;
            e.printStackTrace();
            Logger.INSTANCE.fayeError();
            Log.e(LOG_TAG, "Subscribe message error" + e);
        } catch (Exception e) {
            Log.e(LOG_TAG, "Subscribe message error" + e);
        }
    }

    private void unsubscribe(String channel) {
        try {
            String unsubscribe = mFuguAgentMetaMessage.unsubscribe(channel);
            mFuguAgentWebSocket.send(unsubscribe);
            Log.i(LOG_TAG, "UnSubscribe:" + channel);
        } catch (Exception e) {
            e.printStackTrace();
            Log.e(LOG_TAG, "Unsubscribe message error" + e);
        }
    }

    private void connect() {
        try {
            String connect = mFuguAgentMetaMessage.connect();
            mFuguAgentWebSocket.send(connect);
            mFuguAgentWebSocket.setConnectionLostTimeout(10);
        } catch (Exception e) {
            Log.e(LOG_TAG, "Connect message error" + e);
        }
    }

    private void reconnectFaye() {

    }

    private void disconnect() {
        try {
            String disconnect = mFuguAgentMetaMessage.disconnect();
            mFuguAgentWebSocket.send(disconnect);

            //mFuguAgentWebSocket.reconnectBlocking();
        } catch (JSONException e) {
            Log.e(LOG_TAG, "Disconnect message error" + e);
        }
    }

    private void handleFayeMessage(String message) {
        JSONArray arr = null;
        try {
            arr = new JSONArray(message);
        } catch (JSONException e) {
            Log.e(LOG_TAG, "Unknown message type: " + message + e);
        }

        int length = arr.length();
        for (int i = 0; i < length; ++i) {
            JSONObject obj = arr.optJSONObject(i);
            if (obj == null) continue;

            String channel = obj.optString(FuguAgentMetaMessage.KEY_CHANNEL);
            boolean successful = obj.optBoolean("successful");
            if (channel.equals(FuguAgentMetaMessage.HANDSHAKE_CHANNEL)) {
                if (successful) {
                    mFuguAgentMetaMessage.setClient(obj.optString(FuguAgentMetaMessage.KEY_CLIENT_ID));
                    if (mManagerListener != null)
                        mManagerListener.onConnectedServer(this);
                    if (serviceListener != null)
                        serviceListener.onConnectedServer(this);
                    connect();
                } else {
                    Log.e(LOG_TAG, "Handshake Error: " + obj.toString());
                }
                return;
            }

            if (channel.equals(FuguAgentMetaMessage.CONNECT_CHANNEL)) {
                if (successful) {
                    mFayeConnected = true;
                    connect();
                } else {
                    Log.e(LOG_TAG, "Connecting Error: " + obj.toString());
                }
                return;
            }

            if (channel.equals(FuguAgentMetaMessage.DISCONNECT_CHANNEL)) {
                if (successful) {
                    if (mManagerListener != null)
                        mManagerListener.onDisconnectedServer(this);
                    if (serviceListener != null)
                        serviceListener.onDisconnectedServer(this);
                    mFayeConnected = false;
                    closeWebSocketConnection();
                    Logger.INSTANCE.fayeDisconnected();
                } else {
                    Log.e(LOG_TAG, "Disconnecting Error: " + obj.toString());
                }
                return;
            }

            if (channel.equals(FuguAgentMetaMessage.SUBSCRIBE_CHANNEL)) {
                String subscription = obj.optString(FuguAgentMetaMessage.KEY_SUBSCRIPTION);
                if (successful) {
                    mFayeConnected = true;
                    Log.i(LOG_TAG, "Subscribed channel " + subscription);
                } else {
                    Log.e(LOG_TAG, "Subscribing channel " + subscription
                            + " Error: " + obj.toString());
                }
                return;
            }

            if (channel.equals(FuguAgentMetaMessage.UNSUBSCRIBE_CHANNEL)) {
                String subscription = obj.optString(FuguAgentMetaMessage.KEY_SUBSCRIPTION);
                if (successful) {
                    Log.i(LOG_TAG, "Unsubscribed channel " + subscription);
                } else {
                    Log.e(LOG_TAG, "Unsubscribing channel " + subscription
                            + " Error: " + obj.toString());
                }
                return;
            }


            if (mChannels.contains(channel)) {
                String data = obj.optString(FuguAgentMetaMessage.KEY_DATA, null);
                if (data != null) {
                    if (mManagerListener != null)
                        mManagerListener.onReceivedMessage(this, data, channel);

                    if (serviceListener != null)
                        serviceListener.onReceivedMessage(this, data, channel);
                } else {
                    try {
                        if (obj.has("error")) {
                            if (mManagerListener != null)
                                mManagerListener.onErrorReceived(this, obj.optString("error"), channel);
                            if (serviceListener != null)
                                serviceListener.onErrorReceived(this, obj.optString("error"), channel);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            } else {
                Log.e(LOG_TAG, "Cannot handle this message: " + obj.toString());
                if (obj.has("error")) {
                    if (mManagerListener != null)
                        mManagerListener.onErrorReceived(this, obj.optString("error"), channel);
                    if (serviceListener != null)
                        serviceListener.onErrorReceived(this, obj.optString("error"), channel);
                } else {
                    String data = obj.optString(FuguAgentMetaMessage.KEY_DATA, null);
                    System.out.println("3333333 ~~~> " + data);
                    if (serviceListener != null)
                        serviceListener.onReceivedMessage(this, data, channel);
                    if (mManagerListener != null)
                        mManagerListener.onReceivedMessage(this, data, channel);
                }

            }
            return;
        }
    }


    public boolean hasChannel(String channel) {
        return mChannels.contains(channel);
    }

    public HashSet<String> getmChannels() {
        return mChannels;
    }

    public boolean isChannelSubscribed(String mChannel) {
        return getmChannels() != null && getmChannels().contains(mChannel);
    }

    public boolean hasSubscribed(String s) {
        if (mChannels.contains(s))
            return false;
        return true;
    }

    public boolean isOpened() {
        if (mFuguAgentWebSocket != null && mFuguAgentWebSocket.isOpen() && mFuguAgentWebSocket.getSocket().isConnected()) {
            return true;
        }
        return false;
    }

}
