/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate;

/*!
 *  \~chinese
 *  聊天室监听接口  
 *
 *  \~english
 *  chat room change listener
 */

import java.util.List;

public interface EMChatRoomChangeListener {
    /**
     * \~chinese
     * 聊天室被解散。
     * 
     * @param roomId
     *            聊天室id
     * @param roomName
     *            聊天室名称
     *
     * \~english
     * callback when chat room is destroyed
     * 
     * @param roomId        chatroom id
     * @param roomName      chatroom subject
     */
    void onChatRoomDestroyed(final String roomId, final String roomName);

    /**
     * \~chinese
     * 聊天室加入新成员事件
     * 
     * @param roomId
     *          聊天室id
     * @param participant
     *          新成员username
     *
     * \~english
     * Callback when a member join the chatroom.
     * 
     * @param roomId        chatroom id
     * @param participant   new member's username
     */
    void onMemberJoined(final String roomId, final String participant);

    /**
     * \~chinese
     * 聊天室成员主动退出事件
     * 
     * @param roomId
     *          聊天室id
     * @param roomName
     *          聊天室名字
     * @param participant
     *          退出的成员的username
     * 
     * \~english
     * Callback when a member exited the chatroom
     * 
     * @param roomId       chatroom id
     * @param roomName     chatroom's subject
     * @param participant  the member who exited the chatroom
      */
    void onMemberExited(final String roomId, final String roomName, final String participant);

    /**
     * \~chinese
     * 聊天室人员被移除
     * 
     * @param roomId
     *          聊天室id
     * @param roomName
     *          聊天室名字
     * @param participant
     *          被移除人员的username
     *
     * \~english
     * Callback when a member is dismissed from a chat room
     * 
     * @param roomId        chatroom id
     * @param roomName      the chatroom's subject
     * @param participant   the member is dismissed from a chat room
     */
    void onRemovedFromChatRoom(final String roomId, final String roomName, final String participant);

    /**
     * \~chinese
     * 有成员被禁言
     *
     * @param chatRoomId    聊天室id
     * @param mutes         禁言的成员
     * @param expireTime    禁言有效期，单位是毫秒。
     *
     * \~english
     * Callback when chat room member(s) is muted (added to mute list), and is not allowed to post message temporarily based on muted time duration
     *
     * @param chatRoomId    chatroom id
     * @param mutes         muted username
     * @param expireTime    mute operation expired time
     */
    void onMuteListAdded(final String chatRoomId, final List<String> mutes, final long expireTime);

    /**
     * \~chinese
     * 成员从禁言列表中移除
     *
     * @param chatRoomId    聊天室id
     * @param mutes         从禁言列表中移除的成员名单
     *
     * \~english
     * Callback when chat room member(s) is unmuted (removed from mute list), and allow to post message now
     *
     * @param chatRoomId    chatroom id
     * @param mutes         member(s) muted was removed from the mute list
     */
    void onMuteListRemoved(final String chatRoomId, final List<String> mutes);

    /**
     * \~chinese
     * 有成员提升为管理员权限
     *
     * @param chatRoomId    聊天室id
     * @param admin         提升的管理员
     *
     * \~english
     * Callback when a member has been changed to admin
     *
     * @param  chatRoomId   chatroom id
     * @param  admin        member who has been changed to admin
     */
    void onAdminAdded(final String chatRoomId, final String admin);

    /**
     * \~chinese
     * 移除管理员权限
     *
     * @param  chatRoomId   聊天室id
     * @param  admin        被移除的管理员
     *
     * \~english
     * Callback when member is removed from admin
     *
     * @param  chatRoomId   chatroom id
     * @param  admin        the member whose admin permission is removed
     */
    void onAdminRemoved(final String chatRoomId, final String admin);

    /**
     * \~chinese
     * 转移拥有者
     *
     * @param chatRoomId    聊天室id
     * @param newOwner      新所有者
     * @param oldOwner      原聊天室所有者
     *
     * \~english
     * Callback when chat room ownership has been transferred
     *
     * @param chatRoomId    chatroom id
     * @param newOwner      new owner
     * @param oldOwner      previous owner
     */
    void onOwnerChanged(final String chatRoomId, final String newOwner, final String oldOwner);


    /**
     * \~chinese
     * 聊天室公告更改事件
     * @param chatRoomId  聊天室id
     * @param announcement 更新的公告内容
     *
     * \~english
     * callback when the announcement changed
     * @param chatRoomId chatroom id
     * @param announcement changed announcement
     */
    void onAnnouncementChanged(String chatRoomId, String announcement);
}
