/************************************************************
 *  * hyphenate CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 hyphenate Technologies. All rights reserved. 
 *
 * NOTICE: All information contained herein is, and remains 
 * the property of hyphenate Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from hyphenate Technologies.
 */
package com.hyphenate.chat;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMContactListener;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMAContactListener;
import com.hyphenate.chat.adapter.EMAContactManager;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.exceptions.HyphenateException;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * \~chinese
 * EMContactManager 用来记录，查询，修改 用户的联系人/好友列表 它主要负责：
 * 登陆后从环信服务器获取联系人列表
 * 添加联系人
 * 删除联系人
 * 刷新联系人列表
 *
 * 如果应用自己管理联系人/好友列表，则不需要使用此 EMContactManager EMChatOptions.setUseRoster(false);
 *
 * \~english
 * EMContactManager is used to manage Hyphenate friends. if your app has your own friends management, you don't need to
 *  use hyphenate contact manager class and please set EMContactManager EMChatOptions.setUseRoster(false);
 * @see EMContact
 */
public class EMContactManager {

    EMAContactManager emaObject;
    EMClient mClient;
    private List<EMContactListener> contactListeners = Collections.synchronizedList(new ArrayList<EMContactListener>());
    private EMAContactListenerImpl contactImpl = new EMAContactListenerImpl();

    EMContactManager(EMClient client, EMAContactManager contactManager) {
        mClient = client;
        emaObject = new EMAContactManager(contactManager);
        emaObject.registerContactListener(contactImpl);
    }

    class EMAContactListenerImpl extends EMAContactListener {

        @Override
        public void onContactAdded(String username) {
            synchronized (contactListeners) {
                try {
                    for (EMContactListener listener : contactListeners) {
                        listener.onContactAdded(username);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        @Override
        public void onContactDeleted(String username) {
            EMClient.getInstance().chatManager().caches.remove(username);

            synchronized (contactListeners) {
                try {
                    for (EMContactListener listener : contactListeners) {
                        listener.onContactDeleted(username);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        @Override
        public void onContactInvited(String username, String reason) {
            synchronized (contactListeners) {
                try {
                    for (EMContactListener listener : contactListeners) {
                        listener.onContactInvited(username, reason);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        @Override
        public void onContactAgreed(String username) {
            synchronized (contactListeners) {
                try {
                    for (EMContactListener listener : contactListeners) {
                        listener.onFriendRequestAccepted(username);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        @Override
        public void onContactRefused(String username) {
            synchronized (contactListeners) {
                try {
                    for (EMContactListener listener : contactListeners) {
                        listener.onFriendRequestDeclined(username);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
    }

    private void handleError(EMAError error)  throws HyphenateException {
        if (error.errCode() != EMAError.EM_NO_ERROR) {
            throw new HyphenateException(error);
        }
    }

    /**
     * \~chinese
     * 添加联系人
     *
     * 同步方法，会阻塞当前线程
     *
     * @param username		用户名
     * @param reason		原因、理由（optional. use "" or null)
     * @throws HyphenateException
     *
     * \~english
     * add a new user
     *
     * Synchronization method will block the current thread
     *
     * @param username		The user to add
     * @param reason     	message for adding contact (optional. use "" or null))
     * @throws HyphenateException
     */
    public void addContact(String username, String reason) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.inviteContact(username, reason, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 添加联系人
     *
     * @param username  用户名
     * @param reason    原因、理由（optional)
     * @param callback
     *
     * \~english
     * add a new user
     *
     * @param username  user id
     * @param reason    message for adding contact (optional)
     * @param callback
     */
    public void aysncAddContact(final String username, final String reason, final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    addContact(username, reason);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 删除好友
     *
     * @param username
     * @throws HyphenateException
     *
     * \~english
     * Delete a contact and all the conversations associated
     *
     * @param username
     * @throws HyphenateException
     */
    public void deleteContact(String username) throws HyphenateException {
        deleteContact(username, false);
    }

    /**
     * \~chinese
     * 删除好友
     *
     * 同步方法，会阻塞当前线程
     *
     * @param username
     * @param keepConversation  是否保留会话和消息
     * @throws HyphenateException
     *
     * \~english
     * Delete a contact
     *
     * Synchronization method will block the current thread
     *
     * @param username
     * @param keepConversation  If to keep the assoicated conversation and messages
     * @throws HyphenateException
     */
    public void deleteContact(String username, boolean keepConversation) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.deleteContact(username, error, keepConversation);
        EMClient.getInstance().chatManager().caches.remove(username);
        handleError(error);
    }

    /**
     * \~chinese
     * 删除好友
     *
     * @param username
     * @param callback
     *
     * \~english
     * Delete a contact
     *
     * @param username      The user to delete
     * @param callback
     */
    public void aysncDeleteContact(final String username,
                                   final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    deleteContact(username);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 从服务器获取所有的好友
     *
     * 同步方法，会阻塞当前线程
     *
     * @return 联系人列表
     * @throws HyphenateException
     *
     * \~english
     * Get all contacts from the server
     *
     * Synchronization method will block the current thread
     *
     * @return list of contacts
     * @throws HyphenateException
     */
    public List<String> getAllContactsFromServer() throws HyphenateException {
        EMAError error = new EMAError();
        List<String> contacts = emaObject.getContactsFromServer(error);
        handleError(error);
        return contacts;
    }

    /**
     * \~chinese
     * 从服务器获取所有的好友
     *
     * @param callback 包含联系人列表
     *
     * \~english
     * Get all contacts from the server
     *
     * @param callback contains list of contacts
     */
    public void aysncGetAllContactsFromServer(final EMValueCallBack<List<String>> callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    List<String> allContacts = getAllContactsFromServer();
                    callback.onSuccess(allContacts);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    //region contactListener
    /**
     * \~chinese
     * 应用通过此接口设置回调，获得联系人变化
     *
     * @param contactListener
     *
     * \~english
     * register new contact listener
     *
     * @param contactListener
     */
    public void setContactListener(EMContactListener contactListener) {
        if (!contactListeners.contains(contactListener)) {
            contactListeners.add(contactListener);
        }
    }

    /**
     * \~chinese
     * 移除contact listener
     *
     * \~english
     * remove contact listener
     */
    public void removeContactListener(EMContactListener contactListener) {
        contactListeners.remove(contactListener);
    }
    //endregion


    /**
     * \~chinese
     * 把指定用户加入到黑名单中
     *
     * 同步方法，会阻塞当前线程
     *
     * @param username  此用户的username
     * @param both      把用户加入黑民单时，如果是both双方发消息时对方都收不到；如果不是，
     * 			        则我能给黑名单的中用户发消息，但是对方发给我时我是收不到的
     * @throws HyphenateException
     *
     * \~english
     * add a user to blacklist
     *
     * Synchronization method will block the current thread
     *
     * @param username  user to be blocked
     * @param both      if true, block the message sending from both side; if false, block receiving message
     * @throws HyphenateException
     */
    public void addUserToBlackList(String username, boolean both) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.addToBlackList(username, both, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 把指定用户加入到黑名单中
     *
     * @param username  此用户的username
     * @param both      把用户加入黑民单时，如果是both双方发消息时对方都收不到；如果不是，
     *                  则我能给黑名单的中用户发消息，但是对方发给我时我是收不到的
     * @param callback
     *
     * \~english
     * add the user to blacklist
     *
     * @param username  user to be blocked
     * @param both      if true, block the message exchange from both side; if false, block receiving message
     * @param callback
     */
    public void aysncAddUserToBlackList(final String username,
                                        final boolean both,
                                        final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    addUserToBlackList(username, both);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 把用户从黑名单中移除
     *
     * 同步方法，会阻塞当前线程
     *
     * @param username
     * @throws HyphenateException
     *
     * \~english
     * remove the contact from blacklist
     *
     * Synchronization method will block the current thread
     *
     * @param username
     * @throws HyphenateException
     */
    public void removeUserFromBlackList(String username) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.removeFromBlackList(username, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 把用户从黑名单中移除
     *
     * @param username
     * @param callback
     *
     * \~english
     * remove the contact from blacklist
     *
     * @param username
     * @param callback
     */
    public void aysncRemoveUserFromBlackList(final String username,
                                             final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    removeUserFromBlackList(username);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 从本地获取黑名单中的用户的usernames
     *
     * @return List
     *
     * \~english
     * get local blacklist
     *
     * @return List
     */
    public List<String> getBlackListUsernames() {
        EMAError error = new EMAError();
        return emaObject.getBlackListFromDB(error);
    }

    /**
     * \~chinese
     * 从服务器获取黑名单中的用户的usernames
     *
     * @return
     * @throws HyphenateException
     *
     * \~english
     * get local blacklist
     *
     * @return
     * @throws HyphenateException
     */
    public List<String> getBlackListFromServer() throws HyphenateException {
        EMAError error = new EMAError();
        List<String> blacklist = emaObject.getBlackListFromServer(error);
        handleError(error);
        return blacklist;
    }

    /**
     * \~chinese
     * 从服务器获取黑名单中的用户的usernames
     *
     * @param callback 包含黑名单列表
     *
     * \~english
     * get blacklist from server
     *
     * @param callback contains black list
     */
    public void aysncGetBlackListFromServer(final EMValueCallBack<List<String>> callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    List<String> blackList = getBlackListFromServer();
                    callback.onSuccess(blackList);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 保存黑名单列表
     *
     * 同步方法，会阻塞当前线程
     *
     * @param blackList
     *
     * \~english
     * save blacklist to local database
     *
     * Synchronization method will block the current thread
     *
     * @param blackList
     */
    public void saveBlackList(List<String> blackList) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.saveBlackList(blackList, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 保存黑名单列表
     *
     * @param blackList
     * @param callback
     *
     * \~english
     * save blacklist to local database
     *
     * @param blackList
     * @param callback
     */
    public void asyncSaveBlackList(final List<String> blackList,
                                   final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    saveBlackList(blackList);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 接受加好友的邀请
     *
     * @param username
     *
     * \~english
     *
     * accept a friend invitation
     * @param username
     */
    public void acceptInvitation(String username) throws HyphenateException
    {
        EMAError error = new EMAError();
        emaObject.acceptInvitation(username, error);
        handleError(error);
    }


    /**
     * \~chinese
     * 接受加好友的邀请
     *
     * @param username
     * @param callback
     *
     * \~english
     *
     * accept a friend invitation
     * @param username
     * @param callback
     */
    public void asyncAcceptInvitation(final String username,
                                      final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    acceptInvitation(username);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 拒绝加好友的邀请
     *
     * @param username
     *
     * \~english
     * decline a friend invitation
     *
     * @param username
     */
    public void declineInvitation(String username) throws HyphenateException  {
        EMAError error = new EMAError();
        emaObject.declineInvitation(username, error);
        handleError(error);
    }

    /**
     * \~chinese
     * 拒绝加好友的邀请
     *
     * @param username
     * @param callback
     *
     * \~english
     * decline a friend invitation
     *
     * @param username
     * @param callback
     */
    public void asyncDeclineInvitation(final String username,
                                       final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    declineInvitation(username);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 从数据库获取好友列表
     *
     * @param error
     * @return 好友列表
     *
     * \~english
     * get contacts from local database
     *
     * @param error
     * @return contact list
     */
    List<String> getContactsFromDB(EMAError error) {
        return emaObject.getContactsFromDB(error);
    }

    void onLogout() {
    }
}
