/**
 * Copyright 2019 Pramati Prism, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.hyscale.generator.services.plugins;

import com.fasterxml.jackson.core.JsonProcessingException;
import io.hyscale.generator.services.utils.ConfigMapDataUtil;
import io.hyscale.plugin.framework.annotation.ManifestPlugin;
import io.hyscale.commons.exception.HyscaleException;
import io.hyscale.commons.models.ManifestContext;
import io.hyscale.generator.services.predicates.ManifestPredicates;
import io.hyscale.generator.services.provider.PropsProvider;
import io.hyscale.plugin.framework.handler.ManifestHandler;
import io.hyscale.plugin.framework.models.ManifestSnippet;
import io.hyscale.servicespec.commons.fields.HyscaleSpecFields;
import io.hyscale.servicespec.commons.model.service.Props;
import io.hyscale.servicespec.commons.model.service.ServiceSpec;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.*;

@Component
@ManifestPlugin(name = "ConfigMapDataHandler")
public class ConfigMapDataHandler implements ManifestHandler {

    private static final Logger logger = LoggerFactory.getLogger(ConfigMapDataHandler.class);

    @Override
    public List<ManifestSnippet> handle(ServiceSpec serviceSpec, ManifestContext manifestContext) throws HyscaleException {
        if (!ManifestPredicates.getPropsPredicate().test(serviceSpec)) {
            logger.debug("Props found to be empty while processing ConfigMap data.");
            return Collections.emptyList();
        }
        Props props = PropsProvider.getProps(serviceSpec);
        String serviceName = serviceSpec.get(HyscaleSpecFields.name, String.class);

        String propsVolumePath = serviceSpec.get(HyscaleSpecFields.propsVolumePath, String.class);

        List<ManifestSnippet> manifestSnippetList = new ArrayList<>();
        try {
            List<ManifestSnippet> configMapSnippets = ConfigMapDataUtil.build(props,propsVolumePath);
            ConfigMapDataUtil.updatePodChecksum(configMapSnippets, manifestContext, null);
            manifestSnippetList.addAll(configMapSnippets);
            logger.debug("Added ConfigMap map data to the manifest snippet list");
        } catch (JsonProcessingException e) {
            logger.error("Error while generating manifest for props of service {}", serviceName, e);
        }
        return manifestSnippetList;
    }

}
