/*
 * Decompiled with CFR 0.152.
 */
package io.inbot.datemath;

import java.time.DayOfWeek;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.Month;
import java.time.OffsetDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.DateTimeParseException;
import java.time.format.TextStyle;
import java.time.temporal.ChronoField;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalAdjusters;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class DateMath {
    private static final Pattern DURATION_PATTERN = Pattern.compile("-?\\s*([0-9]+)\\s*([ms|s|h|d|w|m|y])");
    private static final Pattern SUM_PATTERN = Pattern.compile("(.+)\\s*([\\+-])\\s*(.+)");
    private static final DateTimeFormatter CONSISTENT_ISO_INSTANT = new DateTimeFormatterBuilder().parseCaseInsensitive().appendInstant(3).toFormatter();

    public static Instant now() {
        return Instant.now();
    }

    public static String formatIsoDate(OffsetDateTime date) {
        return DateMath.formatIsoDate(date.toInstant());
    }

    public static String formatIsoDateNow() {
        return DateMath.formatIsoDate(DateMath.now());
    }

    public static String formatIsoDate(LocalDate date) {
        LocalDateTime time = LocalDateTime.of(date.getYear(), date.getMonth(), date.getDayOfMonth(), 0, 0, 0);
        return DateMath.formatIsoDate(time.toInstant(ZoneOffset.UTC));
    }

    public static String formatIsoDate(LocalDateTime time) {
        return DateMath.formatIsoDate(time.toInstant(ZoneOffset.UTC));
    }

    public static String formatIsoDate(Instant date) {
        return CONSISTENT_ISO_INSTANT.format(date);
    }

    public static String formatIsoDate(long timeInMillisSinceEpoch) {
        return DateMath.formatIsoDate(Instant.ofEpochMilli(timeInMillisSinceEpoch));
    }

    private static Instant flexibleInstantParse(String text, ZoneId zoneId) throws DateTimeParseException {
        try {
            return Instant.parse(text);
        }
        catch (DateTimeParseException e) {
            if (zoneId == null) {
                zoneId = ZoneId.of("Z");
            }
            try {
                LocalDate localDate = LocalDate.parse(text);
                LocalDateTime localDateTime = LocalDateTime.of(localDate, LocalTime.MIDNIGHT);
                return localDateTime.toInstant(ZoneOffset.of(zoneId.getId()));
            }
            catch (DateTimeParseException e1) {
                LocalTime localTime = LocalTime.parse(text);
                LocalDate today = LocalDate.now(zoneId);
                LocalDateTime localDateTime = LocalDateTime.of(today, localTime);
                return localDateTime.toInstant(ZoneOffset.of(zoneId.getId()));
            }
        }
    }

    public static Instant parse(String text) {
        return DateMath.parse(text, ZoneOffset.UTC);
    }

    public static Instant parse(String text, String zoneId) {
        ZoneId zone = ZoneId.of(zoneId, ZoneId.SHORT_IDS);
        return DateMath.parse(text, zone);
    }

    private static Instant parse(String text, ZoneId zone) {
        if (text == null) {
            throw new IllegalArgumentException("cannot parse empty string");
        }
        text = text.trim();
        try {
            return DateMath.flexibleInstantParse(text, zone);
        }
        catch (DateTimeParseException e) {
            return DateMath.toInstant(DateMath.parseRelativeTime(text, zone));
        }
    }

    private static LocalDateTime parseRelativeTime(String text, ZoneId zoneId) {
        if (zoneId == null) {
            zoneId = ZoneOffset.UTC;
        }
        LocalDateTime now = LocalDateTime.ofInstant(Instant.now(), zoneId);
        switch (text.replace('_', ' ').toLowerCase(Locale.ENGLISH)) {
            case "min": {
                return LocalDateTime.of(0, 1, 1, 0, 0);
            }
            case "max": {
                return LocalDateTime.of(9999, 1, 1, 0, 0);
            }
            case "distant past": {
                return LocalDateTime.ofInstant(Instant.MIN, zoneId);
            }
            case "distant future": {
                return LocalDateTime.ofInstant(Instant.MAX, zoneId);
            }
            case "morning": {
                return DateMath.parseRelativeTime("09:00", zoneId);
            }
            case "midnight": {
                return DateMath.parseRelativeTime("00:00", zoneId);
            }
            case "noon": {
                return DateMath.parseRelativeTime("12:00", zoneId);
            }
            case "now": {
                return now;
            }
            case "beginning month": {
                return now.truncatedTo(ChronoUnit.DAYS).with(TemporalAdjusters.firstDayOfMonth());
            }
            case "end month": {
                return now.truncatedTo(ChronoUnit.DAYS).with(TemporalAdjusters.firstDayOfNextMonth());
            }
            case "beginning year": {
                return now.truncatedTo(ChronoUnit.DAYS).with(TemporalAdjusters.firstDayOfYear());
            }
            case "end year": {
                return now.truncatedTo(ChronoUnit.DAYS).with(TemporalAdjusters.firstDayOfNextYear());
            }
            case "beginning week": {
                return now.truncatedTo(ChronoUnit.DAYS).with(TemporalAdjusters.previous(DayOfWeek.SUNDAY));
            }
            case "end week": {
                return now.truncatedTo(ChronoUnit.DAYS).with(TemporalAdjusters.next(DayOfWeek.SUNDAY));
            }
            case "tomorrow": {
                return now.truncatedTo(ChronoUnit.DAYS).truncatedTo(ChronoUnit.DAYS).plus(1L, ChronoUnit.DAYS);
            }
            case "day after tomorrow": {
                return now.truncatedTo(ChronoUnit.DAYS).truncatedTo(ChronoUnit.DAYS).plus(2L, ChronoUnit.DAYS);
            }
            case "yesterday": {
                return now.truncatedTo(ChronoUnit.DAYS).truncatedTo(ChronoUnit.DAYS).minus(1L, ChronoUnit.DAYS);
            }
            case "day before yesterday": {
                return now.truncatedTo(ChronoUnit.DAYS).truncatedTo(ChronoUnit.DAYS).minus(2L, ChronoUnit.DAYS);
            }
            case "next month": {
                return DateMath.parseRelativeTime("1m", zoneId);
            }
            case "last month": {
                return DateMath.parseRelativeTime("-1m", zoneId);
            }
            case "next year": {
                return DateMath.parseRelativeTime("1y", zoneId);
            }
            case "last year": {
                return DateMath.parseRelativeTime("-1y", zoneId);
            }
        }
        Matcher durationMatcher = DURATION_PATTERN.matcher(text);
        if (durationMatcher.matches()) {
            boolean minus = text.startsWith("-");
            int amount = Integer.valueOf(durationMatcher.group(1));
            String unit = durationMatcher.group(2);
            return DateMath.adjust(now, minus, amount, unit);
        }
        Matcher sumMatcher = SUM_PATTERN.matcher(text);
        if (sumMatcher.matches()) {
            String left = sumMatcher.group(1);
            String operator = sumMatcher.group(2);
            String right = sumMatcher.group(3);
            Instant offset = DateMath.parse(left);
            boolean minus = operator.equals("-");
            Matcher rightHandSideMatcher = DURATION_PATTERN.matcher(right);
            if (rightHandSideMatcher.matches()) {
                int amount = Integer.valueOf(rightHandSideMatcher.group(1));
                String unit = rightHandSideMatcher.group(2);
                return DateMath.adjust(LocalDateTime.ofInstant(offset, zoneId), minus, amount, unit);
            }
            throw new IllegalArgumentException("illegal duration. Should match ([0-9]+)([s|h|d|w|m|y]): " + right);
        }
        throw new IllegalArgumentException("illegal time expression " + text);
    }

    private static LocalDateTime adjust(LocalDateTime now, boolean minus, int amount, String unit) {
        ChronoUnit chronoUnit;
        switch (unit) {
            case "ms": {
                chronoUnit = ChronoUnit.MILLIS;
                break;
            }
            case "s": {
                chronoUnit = ChronoUnit.SECONDS;
                break;
            }
            case "h": {
                chronoUnit = ChronoUnit.HOURS;
                break;
            }
            case "d": {
                chronoUnit = ChronoUnit.DAYS;
                break;
            }
            case "w": {
                chronoUnit = ChronoUnit.DAYS;
                amount *= 7;
                break;
            }
            case "m": {
                if (minus) {
                    return now.minusMonths(amount);
                }
                return now.plusMonths(amount);
            }
            case "y": {
                if (minus) {
                    return now.minusYears(amount);
                }
                return now.plusYears(amount);
            }
            default: {
                throw new IllegalArgumentException("illegal time unit. Should be [ms|s|h|d|w|m|y]: ");
            }
        }
        if (minus) {
            return now.minus(amount, chronoUnit);
        }
        return now.plus(amount, chronoUnit);
    }

    public static Instant toInstant(LocalDate date) {
        return DateMath.toInstant(LocalDateTime.of(date, LocalTime.MIDNIGHT));
    }

    public static Instant toInstant(LocalDateTime dateTime) {
        return dateTime.toInstant(ZoneOffset.UTC);
    }

    public static String renderWeekYear(Instant t, ZoneId zoneId, Locale locale) {
        LocalDateTime ld = LocalDateTime.ofInstant(t, zoneId);
        int week = ld.get(ChronoField.ALIGNED_WEEK_OF_YEAR);
        int year = ld.get(ChronoField.YEAR);
        return week + ", " + year;
    }

    public static String renderMonthYear(Instant t, ZoneId zoneId, Locale locale) {
        LocalDateTime ld = LocalDateTime.ofInstant(t, zoneId);
        int month = ld.get(ChronoField.MONTH_OF_YEAR);
        int year = ld.get(ChronoField.YEAR);
        return Month.of(month).getDisplayName(TextStyle.FULL, locale) + ", " + year;
    }
}

