/*
 * Copyright © 2021 CodeOnce Software (https://www.codeonce.fr/)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.ioevent.starter.service;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.kafka.common.header.Header;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.kafka.support.KafkaHeaders;
import org.springframework.kafka.support.KafkaNull;
import org.springframework.messaging.Message;
import org.springframework.messaging.support.MessageBuilder;
import org.springframework.stereotype.Service;

import com.ioevent.starter.annotations.ConditionalIOResponse;
import com.ioevent.starter.annotations.IOEvent;
import com.ioevent.starter.annotations.IOFlow;
import com.ioevent.starter.annotations.IOPayload;
import com.ioevent.starter.annotations.IOResponse;
import com.ioevent.starter.annotations.InputEvent;
import com.ioevent.starter.annotations.OutputEvent;
import com.ioevent.starter.configuration.properties.IOEventProperties;
import com.ioevent.starter.domain.IOEventHeaders;
import com.ioevent.starter.domain.IOEventParallelEventInformation;
import com.ioevent.starter.domain.IOEventType;
import com.ioevent.starter.enums.EventTypesEnum;
import com.ioevent.starter.enums.MessageTypesEnum;

import lombok.extern.slf4j.Slf4j;

/**
 * class service for IOEvent,
 */
@Slf4j
@Service
public class IOEventService {

	@Autowired
	private KafkaTemplate<String, Object> kafkaTemplate;

	/**
	 * This is a kafka producer which send parallel events info to
	 * ioevent-parallel-gateway-events topic
	 *
	 * @param parallelEventInfo for the parallel event information,
	 */
	public void sendParallelEventInfo(IOEventParallelEventInformation parallelEventInfo) {
		Message<IOEventParallelEventInformation> message = MessageBuilder.withPayload(parallelEventInfo)
				.setHeader(KafkaHeaders.TOPIC, "ioevent-parallel-gateway-events").setHeader(KafkaHeaders.KEY,
						parallelEventInfo.getHeaders().get(IOEventHeaders.CORRELATION_ID.toString()))
				.build();

		kafkaTemplate.send(message);
	}

	/**
	 * method returns all Inputs names of @IOEvent definition,
	 *
	 * @param ioEvent for the IOEvent annotation,
	 * @return list of Inputs names,
	 */
	public List<String> getInputNames(IOEvent ioEvent) {
		List<String> result = new ArrayList<>();

		for (InputEvent inputEvent : ioEvent.input()) {
			if (!StringUtils.isBlank(inputEvent.key() + inputEvent.value())) {
				if (!StringUtils.isBlank(inputEvent.value())) {
					result.add(inputEvent.value());
				} else {
					result.add(inputEvent.key());
				}
			}
		}

		for (InputEvent inputEvent : ioEvent.gatewayInput().input()) {
			if (!StringUtils.isBlank(inputEvent.key() + inputEvent.value())) {
				if (!StringUtils.isBlank(inputEvent.value())) {
					result.add(inputEvent.value());
				} else {
					result.add(inputEvent.key());
				}
			}
		}
		return result;
	}

	/**
	 * method returns all parallel Input names of @IOEvent definition,
	 *
	 * @param ioEvent for the IOEvent annotation,
	 * @return list of Inputs names,
	 */
	public List<String> getParalleListInput(IOEvent ioEvent) {
		List<String> result = new ArrayList<>();
		for (InputEvent inputEvent : ioEvent.gatewayInput().input()) {
			if (!StringUtils.isBlank(inputEvent.key() + inputEvent.value())) {
				if (!StringUtils.isBlank(inputEvent.value())) {
					result.add(inputEvent.value());
				} else {
					result.add(inputEvent.key());
				}
			}
		}
		return result;
	}

	/**
	 * method returns all outputs names of @IOEvent definition,
	 *
	 * @param ioEvent for the IOEvent annotation,
	 * @return list of outputs names,
	 */
	public List<String> getOutputNames(IOEvent ioEvent) {
		List<String> result = new ArrayList<>();

		for (OutputEvent outputEvent : ioEvent.output()) {
			if (!StringUtils.isBlank(outputEvent.key() + outputEvent.value())) {
				if (!StringUtils.isBlank(outputEvent.value())) {
					result.add(outputEvent.value());
				} else {
					result.add(outputEvent.key());
				}
			}
		}

		for (OutputEvent outputEvent : ioEvent.gatewayOutput().output()) {
			if (!StringUtils.isBlank(outputEvent.key() + outputEvent.value())) {
				if (!StringUtils.isBlank(outputEvent.value())) {
					result.add(outputEvent.value());
				} else {
					result.add(outputEvent.key());
				}
			}
		}
		return result;
	}

	/**
	 * method returns exclusive Gateway outputs names of @IOEvent definition,
	 *
	 * @param ioEvent for the IOEvent annotation,
	 * @return list of outputs names,
	 */
	public List<String> getExclusiveGatewayOutputNames(IOEvent ioEvent) {
		List<String> result = new ArrayList<>();

		for (OutputEvent outputEvent : ioEvent.gatewayOutput().output()) {
			if (!StringUtils.isBlank(outputEvent.key() + outputEvent.value())) {
				if (!StringUtils.isBlank(outputEvent.value())) {
					result.add(outputEvent.value());
				} else {
					result.add(outputEvent.key());
				}
			}
		}
		return result;
	}

	/**
	 * method returns all output Event of @IOEvent definition,
	 *
	 * @param ioEvent for the IOEvent annotation,
	 * @return list of OutputEvent Object ,
	 */
	public List<OutputEvent> getOutputs(IOEvent ioEvent) {
		List<OutputEvent> result = new ArrayList<>();

		for (OutputEvent outputEvent : ioEvent.output()) {
			if (!StringUtils.isBlank(outputEvent.key() + outputEvent.value() + outputEvent.suffix())) {
				result.add(outputEvent);
			}
		}

		for (OutputEvent outputEvent : ioEvent.gatewayOutput().output()) {
			if (!StringUtils.isBlank(outputEvent.key() + outputEvent.value() + outputEvent.suffix())) {
				result.add(outputEvent);
			}
		}
		return result;
	}

	/**
	 * method returns all inputs of @IOEvent definition,
	 *
	 * @param ioEvent for the IOEvent annotation,
	 * @return list of InputEvent Object ,
	 */
	public List<InputEvent> getInputs(IOEvent ioEvent) {
		List<InputEvent> result = new ArrayList<>();

		for (InputEvent inputEvent : ioEvent.input()) {
			if (!StringUtils.isBlank(inputEvent.key() + inputEvent.value())) {
				result.add(inputEvent);
			}
		}

		for (InputEvent inputEvent : ioEvent.gatewayInput().input()) {
			if (!StringUtils.isBlank(inputEvent.key() + inputEvent.value())) {
				result.add(inputEvent);
			}
		}
		return result;
	}

	/**
	 * method returns all topics of @IOEvent annotation,
	 *
	 * @param ioEvent for the IOEvent annotation,
	 * @return list of Topics names ,
	 */

	public List<String> getTopics(IOEvent ioEvent) {
		List<String> result = new ArrayList<>();
		if (!ioEvent.topic().equals("")) {
			result.add(ioEvent.topic());
		}
		for (InputEvent inputEvent : ioEvent.input()) {
			if (!inputEvent.topic().equals("")) {
				result.add(inputEvent.topic());
			}
		}
		for (OutputEvent outputEvent : ioEvent.output()) {
			if (!outputEvent.topic().equals("")) {
				result.add(outputEvent.topic());
			}
		}
		for (InputEvent inputEvent : ioEvent.gatewayInput().input()) {
			if (!inputEvent.topic().equals("")) {
				result.add(inputEvent.topic());
			}
		}
		for (OutputEvent outputEvent : ioEvent.gatewayOutput().output()) {
			if (!outputEvent.topic().equals("")) {
				result.add(outputEvent.topic());
			}
		}
		return result;
	}

	/**
	 * method returns all Input topics of @IOEvent definition,
	 *
	 * @param ioEvent for the IOEvent annotation,
	 * @return list of Topics names ,
	 */

	public List<String> getInputTopic(IOEvent ioEvent, IOFlow ioFlow) {
		List<String> result = new ArrayList<>();
		if ((ioFlow != null) && !StringUtils.isBlank(ioFlow.topic())) {
			result.add(ioFlow.topic());
		}
		if (!StringUtils.isBlank(ioEvent.topic())) {
			result.add(ioEvent.topic());
		}
		for (InputEvent inputEvent : ioEvent.input()) {
			if (!StringUtils.isBlank(inputEvent.topic())) {
				result.add(inputEvent.topic());
			}
		}

		for (InputEvent inputEvent : ioEvent.gatewayInput().input()) {
			if (!StringUtils.isBlank(inputEvent.topic())) {
				result.add(inputEvent.topic());
			}
		}

		return result;
	}

	/**
	 * method returns all Input topics of @IOEvent definition,
	 *
	 * @param ioEvent for the IOEvent annotation,
	 * @return list of Topics names ,
	 */

	public List<String> getOutputEventTopics(IOEvent ioEvent, IOFlow ioFlow) {
		List<String> result = new ArrayList<>();
		if ((ioFlow != null) && !StringUtils.isBlank(ioFlow.topic())) {
			result.add(ioFlow.topic());
		}
		if (!StringUtils.isBlank(ioEvent.topic())) {
			result.add(ioEvent.topic());
		}
		for (OutputEvent outputEvent : ioEvent.output()) {
			if (!StringUtils.isBlank(outputEvent.topic())) {
				result.add(outputEvent.topic());
			}
		}

		for (OutputEvent outputEvent : ioEvent.gatewayOutput().output()) {
			if (!StringUtils.isBlank(outputEvent.topic())) {
				result.add(outputEvent.topic());
			}
		}

		return result;
	}

	/**
	 * method returns if two lists are equal
	 *
	 * @param firstList  list of String,
	 * @param secondList list of String,
	 * @return boolean ,
	 */
	public boolean sameList(List<String> firstList, List<String> secondList) {
		return (firstList.size() == secondList.size() && firstList.containsAll(secondList)
				&& secondList.containsAll(firstList));
	}

	/**
	 * method returns event type from the IOEvent annotation
	 *
	 * @param ioEvent for the IOEvent annotation,
	 * @return IOEventType ,
	 */
	public IOEventType getIOEventType(IOEvent ioEvent) {
		if (isStartConditional(ioEvent)) {
			return IOEventType.START_CONDITIONAL;
		}
		if (!StringUtils.isBlank(ioEvent.startEvent().key() + ioEvent.startEvent().value())) {
			if (isStartTimer(ioEvent)) {
				return IOEventType.START_TIMER;
			} else {
				return IOEventType.START;
			}
		} else if (!StringUtils.isBlank(ioEvent.endEvent().key() + ioEvent.endEvent().value())) {
			return IOEventType.END;
		} else {
			if (ioEvent.timer().delay() > 0) {
				return IOEventType.INTERMEDIATE_TIMER;
			} else if (ioEvent.timer().limit() > 0) {
				return IOEventType.BOUNDRY_TIMER;
			} else if (isMessage(ioEvent)) {
				if (isMessageThrow(ioEvent)) {
					return IOEventType.MESSAGE_THROW;
				} else if (isMessageCatch(ioEvent)) {
					return IOEventType.MESSAGE_CATCH;
				}
			}

			return IOEventType.TASK;
		}
	}

	public boolean isStartConditional(IOEvent ioEvent) {
		return (ioEvent.EventType().equals(EventTypesEnum.START_CONDITIONAL_EVENT));
	}
	
	public boolean isMessage(IOEvent ioEvent) {
		return (!StringUtils.isBlank(ioEvent.message().key()));
		// An iomessage must have property key not blank (not empty string)
	}

	public boolean isMessageThrow(IOEvent ioEvent) {
		return (ioEvent.message().messageType().equals(MessageTypesEnum.THROW));
	}

	public boolean isMessageCatch(IOEvent ioEvent) {
		return (ioEvent.message().messageType().equals(MessageTypesEnum.CATCH));
	}

	/**
	 * method returns if the IOEvent annotation is of a Start Event
	 *
	 * @param ioEvent for the IOEvent annotation,
	 * @return boolean ,
	 */
	public boolean isStart(IOEvent ioEvent) {
		return (!StringUtils.isBlank(ioEvent.startEvent().key() + ioEvent.startEvent().value())
				&& (!getOutputs(ioEvent).isEmpty())
		);

	}

	public boolean isConditionalStart(IOEvent ioEvent) {
		return ((ioEvent.EventType().equals(EventTypesEnum.START_CONDITIONAL_EVENT))
		);
	}

	/**
	 * method returns if the IOEvent annotation is a start timer Event
	 *
	 * @param ioEvent for the IOEvent annotation,
	 * @return boolean ,
	 */
	public boolean isStartTimer(IOEvent ioEvent) {
		return (!StringUtils.isBlank(ioEvent.startEvent().timer().cron()));
	}

	/**
	 * method returns if the IOEvent annotation is an intermediate timer Event
	 *
	 * @param ioEvent for the IOEvent annotation,
	 * @return boolean ,
	 */
	public boolean isIntermediateTimer(IOEvent ioEvent) {
		return (ioEvent.timer().delay() > 0);
	}

	/**
	 * method returns if the IOEvent annotation is a boundry timer Event
	 *
	 * @param ioEvent for the IOEvent annotation,
	 * @return boolean ,
	 */
	public boolean isBoundryTimer(IOEvent ioEvent) {
		return (ioEvent.timer().limit() > 0);
	}

	/**
	 * method returns if the IOEvent annotation is of a End Event
	 *
	 * @param ioEvent for the IOEvent annotation,
	 * @return boolean ,
	 */
	public boolean isEnd(IOEvent ioEvent) {
		return (!StringUtils.isBlank(ioEvent.endEvent().key() + ioEvent.endEvent().value())
				&& (!getInputs(ioEvent).isEmpty()));
	}

	/**
	 * method returns if the IOEvent annotation is of a Implicit Task Event
	 *
	 * @param ioEvent for the IOEvent annotation,
	 * @return boolean ,
	 */
	public boolean isImplicitTask(IOEvent ioEvent) {
		return ((getInputs(ioEvent).isEmpty() && (!ioEvent.EventType().equals(EventTypesEnum.START_CONDITIONAL_EVENT))
				|| getOutputs(ioEvent).isEmpty())
				&& (StringUtils.isBlank(ioEvent.startEvent().key() + ioEvent.startEvent().value())
				&& StringUtils.isBlank(ioEvent.endEvent().key() + ioEvent.endEvent().value())));

	}

	/**
	 * method returns if the IOEvent annotation is of a Task Event
	 *
	 * @param ioEvent for the IOEvent annotation,
	 * @return boolean ,
	 */
	public boolean isTransition(IOEvent ioEvent) {
		return (StringUtils.isBlank(ioEvent.startEvent().key() + ioEvent.startEvent().value())
				&& StringUtils.isBlank(ioEvent.endEvent().key() + ioEvent.endEvent().value())
				&& !getInputs(ioEvent).isEmpty() && !getOutputs(ioEvent).isEmpty());
	}

	/**
	 * method returns input event of @IOEvent by name,
	 *
	 * @param ioEvent   for the IOEvent annotation,
	 * @param inputName for the Input event name
	 * @return InputEvent ,
	 */
	public InputEvent getInputEventByName(IOEvent ioEvent, String inputName) {
		for (InputEvent inputEvent : getInputs(ioEvent)) {
			if (inputName.equals(inputEvent.key())) {
				return inputEvent;
			}
		}
		return null;
	}

	/**
	 * method returns Task specific type from the IOEvent annotation
	 *
	 * @param ioEvent for the IOEvent annotation,
	 * @return IOEventType ,
	 */
	public IOEventType checkTaskType(IOEvent ioEvent) {
		IOEventType type = IOEventType.TASK;
		if ((ioEvent.gatewayOutput().output().length != 0) || (ioEvent.gatewayInput().input().length != 0)) {

			if (ioEvent.gatewayOutput().parallel() || ioEvent.gatewayInput().parallel()) {
				type = IOEventType.GATEWAY_PARALLEL;
			} else if (ioEvent.gatewayOutput().exclusive() || ioEvent.gatewayInput().exclusive()) {
				type = IOEventType.GATEWAY_EXCLUSIVE;
			}
		} else if (isMessage(ioEvent)) {
			if (isMessageThrow(ioEvent)) {
				return IOEventType.MESSAGE_THROW;
			} else if (isMessageCatch(ioEvent)) {
				return IOEventType.MESSAGE_CATCH;
			}
		}

		return type;
	}

	/**
	 * method returns ID generated from @IOEvent elements ,
	 *
	 * @param ioEvent for the IOEvent annotation,
	 * @return String of ID generated ,
	 */
	public String generateID(IOEvent ioEvent) {

		return ioEvent.key().replaceAll("[^a-zA-Z ]", "").toLowerCase().replace(" ", "") + "-"
				+ getInputNames(ioEvent).hashCode() + "-" + getOutputNames(ioEvent).hashCode();
	}

	/**
	 * method returns ProcessName from @IOEvent ,@IOFlow and recordProcessName ,
	 *
	 * @param ioEvent           for the IOEvent annotation,
	 * @param ioFlow            for the IOFlow annotation,
	 * @param recordProcessName for the process name consumed from record ,
	 * @return String of ProcessName ,
	 */
	public String getProcessName(IOEvent ioEvent, IOFlow ioFlow, String recordProcessName) {
		if (!StringUtils.isBlank(recordProcessName)) {
			return recordProcessName;

		} else if (!StringUtils.isBlank(ioEvent.startEvent().key() + ioEvent.startEvent().value())) {
			if (!StringUtils.isBlank(ioEvent.startEvent().value())) {
				return ioEvent.startEvent().value();
			}
			return ioEvent.startEvent().key();
		} else if (!StringUtils.isBlank(ioEvent.endEvent().key() + ioEvent.endEvent().value())) {
			if (!StringUtils.isBlank(ioEvent.endEvent().value())) {
				return ioEvent.endEvent().value();
			}
			return ioEvent.endEvent().key();

		} else if (!Objects.isNull(ioFlow)) {

			return ioFlow.name();
		}
		return "";
	}

	/**
	 * method returns output topic from @IOEvent ,@IOFlow and outputEventTopic ,
	 *
	 * @param ioEvent          for the IOEvent annotation,
	 * @param ioFlow           for the IOFlow annotation,
	 * @param outputEventTopic for the output Event Topic name,
	 * @return String of TopicName ,
	 */
	public String getOutputTopicName(IOEvent ioEvent, IOFlow ioFlow, String outputEventTopic) {
		if (!StringUtils.isBlank(outputEventTopic)) {
			return outputEventTopic;
		} else if (!StringUtils.isBlank(ioEvent.topic())) {
			return ioEvent.topic();
		} else if ((ioFlow != null) && !StringUtils.isBlank(ioFlow.topic())) {
			return ioFlow.topic();

		} else {
			return "";
		}
	}

	/**
	 * method returns ApiKey from @IOFlow and IOEventProperties ,
	 *
	 * @param ioFlow            for the IOFlow annotation,
	 * @param iOEventProperties for the IOEvent custom properties value ,
	 * @return String of ApiKey ,
	 */
	public String getApiKey(IOEventProperties iOEventProperties, IOFlow ioFlow) {
		if ((!Objects.isNull(ioFlow)) && (StringUtils.isNotBlank(ioFlow.apiKey()))) {
			return ioFlow.apiKey();
		} else if (StringUtils.isNotBlank(iOEventProperties.getApikey())) {
			return iOEventProperties.getApikey();
		}
		return "";
	}

	/**
	 * method returns Output Key from OutputEvent ,
	 *
	 * @param outputEvent for the OutputEvent annotation,
	 * @return String of Output Key ,
	 */
	public String getOutputKey(OutputEvent outputEvent) {
		if (!StringUtils.isBlank(outputEvent.value())) {
			return outputEvent.value();
		} else {
			return outputEvent.key();
		}
	}

	/**
	 * method returns payload of the method ,
	 *
	 * @param joinPoint    for the JoinPoint where the method have been called ,
	 * @param returnObject for the object returned by the method
	 * @return IOResponse ,
	 */
	public IOResponse<Object> getpayload(JoinPoint joinPoint, Object returnObject) {
		try {
			if (returnObject != null) {
				IOResponse<Object> ioEventResponse = IOResponse.class.cast(returnObject);
				return ioEventResponse;
			}
			throw new NullPointerException();

		} catch (Exception e) {

			if (returnObject == null) {
				MethodSignature signature = (MethodSignature) joinPoint.getSignature();
				int ioPayloadIndex = getIOPayloadIndex(signature.getMethod());
				if (ioPayloadIndex >= 0) {
					return new IOResponse<>(null, isNullpayload(joinPoint.getArgs()[ioPayloadIndex]));
				}
				try {
					return new IOResponse<>(null, isNullpayload(joinPoint.getArgs()[0]));
				} catch (Exception argException) {
					return new IOResponse<>(null, KafkaNull.INSTANCE);

				}

			}
			return new IOResponse<>(null, returnObject);

		}
	}

	public ConditionalIOResponse<Object> getConditionalPayload(JoinPoint joinPoint, Object returnObject) {
		ConditionalIOResponse<Object> conditional = ConditionalIOResponse.class.cast(returnObject);
		return new ConditionalIOResponse<>(getpayload(joinPoint, returnObject).getKey(),
				getpayload(joinPoint, returnObject).getBody(), conditional.isCondition());
	}


	private Object isNullpayload(Object object) {
		if (object == null) {
			return KafkaNull.INSTANCE;
		}
		return object;
	}

	/**
	 * method returns IOPayload Annotation index in method parameters,
	 *
	 * @param method for the method object ,
	 * @return int ,
	 */
	public int getIOPayloadIndex(Method method) {
		Annotation[][] parameterAnnotations = method.getParameterAnnotations();
		int parameterIndex = 0;
		for (Annotation[] annotations : parameterAnnotations) {
			if (Arrays.asList(annotations).stream().filter(IOPayload.class::isInstance).count() != 0) {
				return parameterIndex;
			}

			parameterIndex++;
		}
		return -1;
	}

	/**
	 * method returns map of headers by merging the consumed headers with the new
	 * headers created in method,
	 *
	 * @param headersConsumed for the List of Header consumed from the event ,
	 * @param newHeaders      a Map of String,Object for the new headers declared in
	 *                        method
	 * @return Map of String,Object
	 */
	public Map<String, Object> prepareHeaders(List<Header> headersConsumed, Map<String, Object> newHeaders) {
		Map<String, Object> result = new HashMap<>();
		if (headersConsumed != null) {
			result = headersConsumed.stream().collect(Collectors.toMap(Header::key, h -> new String(h.value())));
		}
		if (newHeaders != null) {
			result.putAll(newHeaders);
		}
		return result;
	}

	public boolean validExclusiveOutput(IOEvent ioEvent, IOResponse<Object> ioEventResponse) {
		return this.getExclusiveGatewayOutputNames(ioEvent).contains(ioEventResponse.getKey());
	}

	/**
	 * Check if IOFlow is null or the IOFlow name is blank so it throws an
	 * IllegalArgumentException
	 *
	 * @param ioFlow
	 */
	public void ioflowExistValidation(IOFlow ioFlow) {

		if (ioFlow == null) {
			throw new IllegalArgumentException(
					"IOFlow must be declared on the class, please be sure you add  @IOFlow annotation to the service class");

		} else {
			if (StringUtils.isBlank(ioFlow.name())) {
				throw new IllegalArgumentException(
						"IOFlow name can't be empty/null or whitespace, please be sure you add a name field to @IOFlow annotation declared on the service class");

			}
		}
	}

	/**
	 * Check if IOEvent key is blank so it throws an IllegalArgumentException
	 *
	 * @param ioEvent
	 */
	public void ioeventKeyValidation(IOEvent ioEvent) {
		if (StringUtils.isBlank(ioEvent.key())) {
			throw new IllegalArgumentException(
					"IOEvent key can't be empty/null or whitespace , please be sure you add a key to your @IOEvent annotation");
		}
	}

	/**
	 * Check if IOEvent is an exclusive gateway ,in case it's an exclusive gateway
	 * it check if the method return type is IOResponse otherwise throw an
	 * IllegalArgumentException
	 *
	 * @param ioEvent
	 * @param method
	 */
	public void gatewayValidation(IOEvent ioEvent, Method method) {
		if ((ioEvent.gatewayOutput().output().length != 0) && ioEvent.gatewayOutput().exclusive()
				&& !ioEvent.gatewayOutput().parallel()) {
			if (!method.getReturnType().equals(IOResponse.class)) {
				throw new IllegalArgumentException(
						"IOEvent Method with Exclusive Gateway must return IOResponse Object , please be sure you return IOResponce in your exclusive gateway method");
			}
			if (ioEvent.exception().exception().length != 0) {
				throw new IllegalArgumentException(
						"IOEvent Method with Exclusive Gateway can not be declared with @ExceptionEvent ");
			}
		}
		if ((ioEvent.gatewayOutput().output().length != 0 || ioEvent.gatewayInput().input().length != 0)
				&& (ioEvent.gatewayOutput().parallel() || ioEvent.gatewayInput().parallel())) {
			if (ioEvent.exception().exception().length != 0) {
				throw new IllegalArgumentException(
						"IOEvent Method with Parallel Gateway can not be declared with @ExceptionEvent");
			}
		}
	}

	public String getMethodReturnType(Method method) {
		if (method.getGenericReturnType() != null) {
			return method.getGenericReturnType().getTypeName();
		}
		int ioPayloadIndex = getIOPayloadIndex(method);
		if (ioPayloadIndex >= 0) {
			return method.getGenericParameterTypes()[ioPayloadIndex].getTypeName();
		}
		return null;
	}

	public void startAndEndvalidation(IOEvent ioEvent, Method method) {
		if (isStart(ioEvent) || isEnd(ioEvent)) {
			if (ioEvent.exception().exception().length != 0) {
				throw new IllegalArgumentException(
						"IOEvent Method with Start/End Event can not be declared with @ExceptionEvent");
			}
		}
	}


	public void startTimervalidation(IOEvent ioEvent, Method method) {
		if (isStartTimer(ioEvent)) {
			if (method.getParameterCount() != 0) {
				throw new IllegalArgumentException(
						"IOEvent Method with Start Timer Event can not have parameters");
			}
		}
	}
	
//	public void topicExistValidation(IOFlow ioFlow, IOEvent ioEvent) {
//	String errorMsg = "Topic not specified, verify that you have specified the topic in @IOFlow, @IOEvent, @InputEvent or @OutputEvent annotations ";
//	if (StringUtils.isBlank(ioFlow.topic()) && (StringUtils.isBlank(ioEvent.topic()) )) {
//
//		checkInputTopic(getInputs(ioEvent), errorMsg);
//		checkOutputTopic(getOutputs(ioEvent), errorMsg);
//
//	}
//}

//void checkInputTopic(List<InputEvent> inputs, String msg) {
//	if (inputs.isEmpty()) {
//		throw new IllegalArgumentException(msg);
//	}
//	for (InputEvent input : inputs) {
//		if ( StringUtils.isBlank(input.topic())) {
//			throw new IllegalArgumentException(msg);
//		}
//	}
//}

//void checkOutputTopic(List<OutputEvent> outputs, String msg) {
//	if (outputs.isEmpty()) {
//		throw new IllegalArgumentException(msg);
//	}
//	for (OutputEvent output : outputs) {
//		if (StringUtils.isBlank(output.topic()) ) {
//			throw new IllegalArgumentException(msg);
//		}
//	}
//}

}
