/*
 * Copyright 2013 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.jdev.miniprofiler.sql;

import io.jdev.miniprofiler.ProfilerProvider;
import io.jdev.miniprofiler.StaticProfilerProvider;
import io.jdev.miniprofiler.sql.log4jdbc.ConnectionSpy;
import io.jdev.miniprofiler.sql.log4jdbc.SpyLogFactory;

import javax.sql.DataSource;
import java.io.Closeable;
import java.io.IOException;
import java.io.PrintWriter;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.util.logging.Logger;

public class ProfilingDataSource implements DataSource, Closeable {

    private final DataSource targetDataSource;

    public ProfilingDataSource(DataSource targetDataSource) {
        this(targetDataSource, new StaticProfilerProvider());
    }

    public ProfilingDataSource(DataSource targetDataSource, ProfilerProvider profilerProvider) {
        this.targetDataSource = targetDataSource;
        SpyLogFactory.setSpyLogDelegator(new ProfilingSpyLogDelegator(profilerProvider));
    }

    @Override
    public Connection getConnection() throws SQLException {
        Connection conn = targetDataSource.getConnection();
        return new ConnectionSpy(conn);
    }

    @Override
    public Connection getConnection(String username, String password) throws SQLException {
        Connection conn = targetDataSource.getConnection(username, password);
        return new ConnectionSpy(conn);
    }

    @Override
    public PrintWriter getLogWriter() throws SQLException {
        return targetDataSource.getLogWriter();
    }

    @Override
    public void setLogWriter(PrintWriter out) throws SQLException {
        targetDataSource.setLogWriter(out);
    }

    @Override
    public void setLoginTimeout(int seconds) throws SQLException {
        targetDataSource.setLoginTimeout(seconds);
    }

    @Override
    public int getLoginTimeout() throws SQLException {
        return targetDataSource.getLoginTimeout();
    }

    @Override
    public Logger getParentLogger() throws SQLFeatureNotSupportedException {
        return targetDataSource.getParentLogger();
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> T unwrap(Class<T> iface) throws SQLException {
        if (iface.isAssignableFrom(getClass())) {
            return (T) this;
        } else if(iface.isAssignableFrom(targetDataSource.getClass())) {
            return (T) targetDataSource;
        } else {
            return targetDataSource.unwrap(iface);
        }
    }

    @Override
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        return iface.isAssignableFrom(getClass())
            || iface.isAssignableFrom(targetDataSource.getClass())
            || targetDataSource.isWrapperFor(iface);
    }

    @Override
    public void close() throws IOException {
        if(targetDataSource instanceof Closeable) {
            ((Closeable) targetDataSource).close();
        } else {
            try {
                Method closeMethod = targetDataSource.getClass().getMethod("close");
                closeMethod.invoke(targetDataSource);
            } catch (NoSuchMethodException e) {
                throw new UnsupportedOperationException("Data source has no close method", e);
            } catch (InvocationTargetException e) {
                Throwable cause = e.getCause();
                if(cause instanceof RuntimeException) {
                    throw (RuntimeException) cause;
                } else {
                    throw new RuntimeException("Error closing data source", cause);
                }
            } catch (IllegalAccessException e) {
                throw new UnsupportedOperationException("Error closing data source", e);
            }
        }
    }
}
