package io.joynr.capabilities.directory;

/*
 * #%L
 * %%
 * Copyright (C) 2011 - 2014 BMW Car IT GmbH
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import io.joynr.arbitration.DiscoveryQos;
import io.joynr.capabilities.CapabilitiesStore;
import io.joynr.capabilities.CapabilityEntry;

import java.util.Collection;
import java.util.List;

import joynr.infrastructure.GlobalCapabilitiesDirectoryAbstractProvider;
import joynr.types.CapabilityInformation;
import joynr.types.ProviderQos;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;
import com.google.inject.Inject;
import com.google.inject.Singleton;

/**
 * The capabilities directory implementation for server-side capabilities querying.
 * 
 * 
 * Capability informations are stored in a concurrentHashMap. Using a in memory database could be possible optimization.
 */

// TODO Evaluate pro /cons of a in memory database
// TODO Using the interfaceAddress as the key may increase performance in most
// requests.

@Singleton
public class CapabilitiesDirectoryImpl extends GlobalCapabilitiesDirectoryAbstractProvider {
    private static final Logger logger = LoggerFactory.getLogger(CapabilitiesDirectoryImpl.class);
    private CapabilitiesStore capabiltiesStore;

    @Inject
    public CapabilitiesDirectoryImpl(CapabilitiesStore capabiltiesStore) {
        this.capabiltiesStore = capabiltiesStore;
    }

    @Override
    public void add(CapabilityInformation capabilityInformation) {

        CapabilityEntry capabilityEntry = capabilityInformation2Entry(capabilityInformation);
        logger.debug("registered capability: {}", capabilityEntry);
        capabiltiesStore.add(capabilityEntry);
    }

    private CapabilityEntry capabilityInformation2Entry(CapabilityInformation capabilityInformation) {
        return CapabilityEntry.fromCapabilityInformation(capabilityInformation);
    }

    @Override
    public void add(List<CapabilityInformation> capabilitiesInformation) {
        // TODO check interfaces before adding them
        List<CapabilityEntry> capabilityEntries = Lists.newArrayList();
        for (CapabilityInformation capInfo : capabilitiesInformation) {
            capabilityEntries.add(capabilityInformation2Entry(capInfo));
        }

        logger.debug("registered capabilities: interface {}", capabilityEntries.toString());

        capabiltiesStore.add(capabilityEntries);
    }

    @Override
    public void remove(String participantId) {
        logger.debug("removed capability with participantId: {}", participantId);
        capabiltiesStore.remove(participantId);
    }

    @Override
    public void remove(List<String> capabilities) {
        // TODO who is allowed to remove capabilities
        List<CapabilityEntry> capabilityEntries = Lists.newArrayList();
        logger.debug("Removing capabilities: Capabilities {}", capabilityEntries);
        capabiltiesStore.remove(capabilities);
    }

    @Override
    public List<CapabilityInformation> lookup(final String domain, final String interfaceName) {
        logger.debug("Searching channels for domain: " + domain + " interfaceName: " + interfaceName + " {}");
        List<CapabilityInformation> capabilityInformationList = Lists.newArrayList();
        Collection<CapabilityEntry> entryCollection = capabiltiesStore.lookup(domain,
                                                                              interfaceName,
                                                                              DiscoveryQos.NO_FILTER);
        for (CapabilityEntry entry : entryCollection) {
            capabilityInformationList.add(entry.toCapabilityInformation());
        }
        return capabilityInformationList;
    }

    @Override
    public CapabilityInformation lookup(String forParticipantId) {
        logger.debug("Searching capabilities for participantId: {}", forParticipantId);
        CapabilityEntry capEntry = capabiltiesStore.lookup(forParticipantId, DiscoveryQos.NO_FILTER);
        if (capEntry == null) {
            return null;
        } else {
            return capEntry.toCapabilityInformation();
        }
    }

    @Override
    public ProviderQos getProviderQos() {
        // TODO set capDirImpl provider QoS
        return providerQos;
    }

}
