package io.kestros.commons.validation.services.impl;

import io.kestros.commons.osgiserviceutils.exceptions.CacheRetrievalException;
import io.kestros.commons.structuredslingmodels.BaseResource;
import io.kestros.commons.validation.ModelValidationMessageType;
import io.kestros.commons.validation.models.ModelValidator;
import io.kestros.commons.validation.models.RegisteredModelValidator;
import io.kestros.commons.validation.services.ModelValidationCacheService;
import io.kestros.commons.validation.services.ModelValidationService;
import io.kestros.commons.validation.services.ModelValidatorProviderService;
import java.util.ArrayList;
import java.util.List;
import javax.annotation.Nonnull;
import org.apache.felix.hc.api.FormattingResultLog;
import org.osgi.service.component.ComponentContext;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ReferencePolicyOption;

@Component(immediate = true,
           service = ModelValidationService.class)
public class ModelValidationServiceImpl implements ModelValidationService {

  @Reference(cardinality = ReferenceCardinality.OPTIONAL,
             policyOption = ReferencePolicyOption.GREEDY)
  private ModelValidatorProviderService validatorProviderService;

  @Reference(cardinality = ReferenceCardinality.OPTIONAL,
             policyOption = ReferencePolicyOption.GREEDY)
  private ModelValidationCacheService validationCacheService;

  @Override
  public String getDisplayName() {
    return "Model Validation Service";
  }

  @Override
  public void activate(ComponentContext componentContext) {

  }

  @Override
  public void deactivate(ComponentContext componentContext) {

  }

  @Override
  public void runAdditionalHealthChecks(FormattingResultLog log) {

  }

  @Nonnull
  @Override
  public <T extends BaseResource> List<ModelValidator> getProcessedValidators(@Nonnull T model) {
    List<ModelValidator> processedValidators = new ArrayList<>();

    if (validationCacheService != null && validationCacheService.isLive()) {
      // TODO get cached.

    } else {
      // TODO message.
    }
    if (validatorProviderService != null) {

      for (RegisteredModelValidator registeredModelValidator :
          validatorProviderService.getActiveValidators(
          model.getClass())) {
        ModelValidator modelValidator = registeredModelValidator.getModelValidator();
        if (modelValidator.getModel() != null) {
          modelValidator.setModel(null);
        }
        modelValidator.setModel(model);
        modelValidator.validate();

        processedValidators.add(modelValidator);
      }
    }

    return processedValidators;
  }

  @Nonnull
  @Override
  public <T extends BaseResource> Integer getErrorsCount(@Nonnull T model) {
    return this.getErrorMessages(model).size();
  }

  @Nonnull
  @Override
  public <T extends BaseResource> Integer getWarningsCount(@Nonnull T model) {
    return this.getWarningMessages(model).size();
  }

  @Nonnull
  @Override
  public <T extends BaseResource> List<String> getErrorMessages(@Nonnull T model) {
    return getMessagesOfLevel(model, ModelValidationMessageType.ERROR, true);
  }

  @Nonnull
  @Override
  public <T extends BaseResource> List<String> getWarningMessages(@Nonnull T model) {
    return getMessagesOfLevel(model, ModelValidationMessageType.WARNING, true);
  }

  @Nonnull
  public <T extends BaseResource> List<String> getMessagesOfLevel(@Nonnull T model,
      ModelValidationMessageType type, boolean cache) {
    if (validationCacheService != null && validationCacheService.isLive()) {
      try {
        if (type.equals(ModelValidationMessageType.ERROR)) {
          return validationCacheService.getCachedErrorMessages(model.getResource(),
              model.getClass());
        } else if (type.equals(ModelValidationMessageType.WARNING)) {
          return validationCacheService.getCachedWarningMessages(model.getResource(),
              model.getClass());
        }
      } catch (CacheRetrievalException e) {
        // todo log.
      }
    }
    List<String> messages = new ArrayList<>();
    for (ModelValidator modelValidator : this.getProcessedValidators(model)) {
      if (type.equals(modelValidator.getType())) {
        if (!modelValidator.isValid()) {
          messages.add(modelValidator.getMessage());
        }
      }
    }
    List<String> errorMessages = new ArrayList<>();
    List<String> warningMessages = new ArrayList<>();
    if (validationCacheService != null && cache) {
      if (type.equals(ModelValidationMessageType.ERROR)) {
        errorMessages = messages;
        warningMessages = getMessagesOfLevel(model, ModelValidationMessageType.WARNING, false);
      } else if (type.equals(ModelValidationMessageType.WARNING)) {
        warningMessages = messages;
        errorMessages = getMessagesOfLevel(model, ModelValidationMessageType.ERROR, false);
      }
      validationCacheService.cacheValidationResults(model, errorMessages, warningMessages);
    }

    return messages;


  }


}
