package io.kestros.commons.validation.services.impl;

import io.kestros.commons.validation.models.ModelValidator;
import io.kestros.commons.validation.models.RegisteredModelValidator;
import io.kestros.commons.validation.services.ModelValidatorProviderService;
import io.kestros.commons.validation.services.ModelValidatorRegistrationHandlerService;
import java.util.ArrayList;
import java.util.List;
import org.apache.felix.hc.api.FormattingResultLog;
import org.osgi.service.component.ComponentContext;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ReferencePolicyOption;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Component(immediate = true,
           service = ModelValidatorProviderService.class,
           property = "service.ranking:Integer=100")
public class ModelValidatorProviderServiceImpl implements ModelValidatorProviderService {

  private static final Logger LOG = LoggerFactory.getLogger(
      ModelValidatorProviderServiceImpl.class);

  @Reference(cardinality = ReferenceCardinality.OPTIONAL,
             policyOption = ReferencePolicyOption.GREEDY)
  private ModelValidatorRegistrationHandlerService registrationHandlerService;

  @Override
  public String getDisplayName() {
    return "Model Validator Provider Service";
  }

  @Override
  public void activate(ComponentContext componentContext) {

  }

  @Override
  public void deactivate(ComponentContext componentContext) {

  }

  @Override
  public void runAdditionalHealthChecks(FormattingResultLog log) {

  }

  @Override
  public void activateValidator(ModelValidator validator, Class type) {
    if (registrationHandlerService != null) {
      List<RegisteredModelValidator> registeredModelValidators
          = registrationHandlerService.getRegisteredModelValidatorMap().get(type);

      for (RegisteredModelValidator registeredModelValidator : registeredModelValidators) {
        if (registeredModelValidator.getModelValidator().getMessage().equals(
            validator.getMessage())) {
          registeredModelValidator.activate();
        }
      }
    } else {
      LOG.error("Unable to activate validator {}. RegistrationHandlerService was null.",
          validator.getMessage());
    }
  }

  @Override
  public void deactivateValidator(ModelValidator validator, Class type) {
    if (registrationHandlerService != null) {
      List<RegisteredModelValidator> registeredModelValidators
          = registrationHandlerService.getRegisteredModelValidatorMap().get(type);

      for (RegisteredModelValidator registeredModelValidator : registeredModelValidators) {
        if (registeredModelValidator.getModelValidator().getMessage().equals(
            validator.getMessage())) {
          registeredModelValidator.deactivate();
        }
      }
    } else {
      LOG.error("Unable to deactivate validator {}. RegistrationHandlerService was null.",
          validator.getMessage());
    }
  }

  @Override
  public List<RegisteredModelValidator> getAllValidators(Class type) {
    return registrationHandlerService.getRegisteredModelValidatorMap().get(type);
  }

  @Override
  public List<RegisteredModelValidator> getActiveValidators(Class type) {
    List<RegisteredModelValidator> activateValidators = new ArrayList<>();
    if (registrationHandlerService != null) {
      if (registrationHandlerService.getRegisteredModelValidatorMap().containsKey(type)) {
        for (RegisteredModelValidator modelValidator :
            registrationHandlerService.getRegisteredModelValidatorMap().get(
            type)) {
          if (modelValidator.isActive()) {
            activateValidators.add(modelValidator);
          }
        }
      }
    } else {
      LOG.error("Unable to retrieve active validators for {}. RegistrationHandlerService was null.",
          type.getName());
    }
    return activateValidators;
  }

  @Override
  public List<RegisteredModelValidator> getInactiveValidators(Class type) {
    List<RegisteredModelValidator> inactivateValidators = new ArrayList<>();
    if (registrationHandlerService != null) {
      if (registrationHandlerService.getRegisteredModelValidatorMap().containsKey(type)) {
        for (RegisteredModelValidator modelValidator :
            registrationHandlerService.getRegisteredModelValidatorMap().get(
            type)) {
          if (!modelValidator.isActive()) {
            inactivateValidators.add(modelValidator);
          }
        }
      }
    } else {
      LOG.error(
          "Unable to retrieve inactive validators for {}. RegistrationHandlerService was null.",
          type.getName());
    }
    return inactivateValidators;
  }

}