package io.kestros.commons.validation.services.impl;

import static io.kestros.commons.osgiserviceutils.utils.OsgiServiceUtils.getAllOsgiServicesOfType;

import io.kestros.commons.osgiserviceutils.utils.OsgiServiceUtils;
import io.kestros.commons.validation.models.ModelValidator;
import io.kestros.commons.validation.models.RegisteredModelValidator;
import io.kestros.commons.validation.services.ModelValidationActivateStatusService;
import io.kestros.commons.validation.services.ModelValidatorRegistrationHandlerService;
import io.kestros.commons.validation.services.ModelValidatorRegistrationService;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.felix.hc.api.FormattingResultLog;
import org.osgi.service.component.ComponentContext;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ReferencePolicyOption;

@Component(immediate = true,
           service = ModelValidatorRegistrationHandlerService.class,
           property = "service.ranking:Integer=100")
public class ModelValidatorRegistrationHandlerServiceImpl
    implements ModelValidatorRegistrationHandlerService {

  @Reference(cardinality = ReferenceCardinality.OPTIONAL,
             policyOption = ReferencePolicyOption.GREEDY)
  private ModelValidationActivateStatusService modelValidationActivateStatusService;

  private ComponentContext componentContext;
  Map<Class, List<RegisteredModelValidator>> registeredModelValidatorMap = new HashMap<>();

  @Override
  public String getDisplayName() {
    return "Model Validator Registration Handler Service";
  }

  @Activate
  public void activate(ComponentContext componentContext) {
    this.componentContext = componentContext;
  }

  @Override
  public void deactivate(ComponentContext componentContext) {

  }

  @Override
  public void runAdditionalHealthChecks(FormattingResultLog log) {
    List<ModelValidatorRegistrationService> registrationServiceList
        = OsgiServiceUtils.getAllOsgiServicesOfType(componentContext,
        ModelValidatorRegistrationService.class);

    for (ModelValidatorRegistrationService modelValidatorRegistrationService :
        registrationServiceList) {
      if (modelValidatorRegistrationService.getModelValidatorRegistrationHandlerService() == null) {
        log.warn(String.format("No referenced ModelValidatorRegistrationService for %s",
            modelValidatorRegistrationService.getClass().getName()));
      }
    }

    if (getRegisteredModelValidatorMap().isEmpty()) {
      log.info("No registered validators detected.");
    }
  }

  @Override
  public Map<Class, List<RegisteredModelValidator>> getRegisteredModelValidatorMap() {
    return this.registeredModelValidatorMap;
  }

  @Override
  public void registerAllValidatorsFromAllServices() {
    List<ModelValidatorRegistrationService> services = getAllOsgiServicesOfType(
        this.componentContext, ModelValidatorRegistrationService.class);
    for (ModelValidatorRegistrationService service : services) {
      this.registerAllValidatorsFromService(service);
    }
  }

  @Override
  public void registerAllValidatorsFromService(
      ModelValidatorRegistrationService registrationService) {
    this.registerValidators(registrationService.getModelValidators(),
        registrationService.getModelType());
  }

  @Override
  public void unregisterAllValidatorsFromService(
      ModelValidatorRegistrationService registrationService) {
    this.removeValidators(registrationService.getModelValidators(),
        registrationService.getModelType());
  }

  @Override
  public void registerValidators(List<ModelValidator> modelValidators, Class type) {
    if (!registeredModelValidatorMap.containsKey(type)) {

      registeredModelValidatorMap.put(type,
          getRegisteredModelValidatorsFromModelValidators(modelValidators, type));
    } else {
      List<RegisteredModelValidator> newModelValidators = new ArrayList<>();
      List<RegisteredModelValidator> validatorsToAdd
          = getRegisteredModelValidatorsFromModelValidators(modelValidators, type);
      List<RegisteredModelValidator> existingRegisteredModelValidators
          = registeredModelValidatorMap.get(type);

      if (existingRegisteredModelValidators != null) {

        for (RegisteredModelValidator existingRegisteredValidator :
            existingRegisteredModelValidators) {
          RegisteredModelValidator overriddingModelValidator = null;
          for (RegisteredModelValidator newRegisteredModelValidator : validatorsToAdd) {
            if (existingRegisteredValidator.getModelValidator().getMessage().equals(
                newRegisteredModelValidator.getModelValidator().getMessage())) {
              overriddingModelValidator = newRegisteredModelValidator;
            }
          }
          if (overriddingModelValidator != null) {
            if (!existingRegisteredValidator.isActive()) {
              overriddingModelValidator.deactivate();
            }
            newModelValidators.add(overriddingModelValidator);
          }
        }

      }

      for (RegisteredModelValidator newRegisteredModelValidator : validatorsToAdd) {
        boolean isNewValidator = true;
        for (RegisteredModelValidator existingRegisteredValidator :
            existingRegisteredModelValidators) {

          if (existingRegisteredValidator.getModelValidator().getMessage().equals(
              newRegisteredModelValidator.getModelValidator().getMessage())) {
            isNewValidator = false;
          }
        }
        if (isNewValidator) {
          newModelValidators.add(newRegisteredModelValidator);
        }
      }
      registeredModelValidatorMap.remove(type);

      registeredModelValidatorMap.put(type, newModelValidators);
    }

  }

  @Override
  public void removeValidators(List<ModelValidator> modelValidators, Class type) {
    // todo may not be needed.
    //    if (registeredModelValidatorMap.containsKey(type)) {
    //      List<RegisteredModelValidator> newRegisteredModelValidatorList = new ArrayList<>();
    //
    //      registeredModelValidatorMap.remove(type);
    //      registeredModelValidatorMap.put(type, newRegisteredModelValidatorList);
    //    }
  }

  @Override
  public ModelValidationActivateStatusService getModelValidationActivateStatusService() {
    return this.modelValidationActivateStatusService;
  }

  private List<RegisteredModelValidator> getRegisteredModelValidatorsFromModelValidators(
      List<ModelValidator> validators, Class type) {
    List<RegisteredModelValidator> registeredModelValidatorList = new ArrayList<>();
    if (validators != null && type != null) {
      for (ModelValidator modelValidator : validators) {
        boolean active = true;
        if (modelValidationActivateStatusService != null) {
          active = modelValidationActivateStatusService.isModelValidatorActiveForClass(
              modelValidator, type);
        }
        RegisteredModelValidator registeredModelValidator = new RegisteredModelValidator(
            modelValidator, active);
        registeredModelValidatorList.add(registeredModelValidator);
      }
    }
    return registeredModelValidatorList;
  }

}
