package io.kotest.core.test

import io.kotest.mpp.bestName
import kotlin.time.Duration
import kotlin.time.ExperimentalTime

@OptIn(ExperimentalTime::class)
data class TestResult(
   val status: TestStatus,
   val error: Throwable?,
   val reason: String?,
   val duration: Duration
) {
   companion object {
      fun success(duration: Duration) = TestResult(
         TestStatus.Success,
         null,
         null,
         duration
      )

      /**
       * Returns a [TestResult] derived from a throwable.
       * If the throwable is either an [AssertionError] or one of the library specific assertion types,
       * then a [TestStatus.Failure] will be returned, otherwise a [TestStatus.Error] will be returned.
       */
      fun throwable(e: Throwable, duration: Duration): TestResult {
         return when (e) {
            is AssertionError -> failure(e, duration)
            else -> when (e::class.bestName()) {
               "org.opentest4j.AssertionFailedError", "AssertionFailedError" -> failure(e, duration)
               "org.junit.ComparisonFailure", "ComparisonFailure" -> failure(e, duration)
               else -> error(e, duration)
            }
         }
      }

      val Ignored = TestResult(
         TestStatus.Ignored,
         null,
         null,
         Duration.ZERO
      )

      private fun failure(e: Throwable, duration: Duration) = TestResult(
         TestStatus.Failure,
         e,
         null,
         duration
      )

      private fun error(t: Throwable, duration: Duration) = TestResult(
         TestStatus.Error,
         t,
         null,
         duration
      )

      fun ignored(reason: String?) = TestResult(
         TestStatus.Ignored,
         null,
         reason,
         Duration.ZERO
      )
   }
}

enum class TestStatus {
   // the test was skipped completely
   Ignored,

   // the test was successful
   Success,

   // the test failed because of some exception that was not an assertion error
   Error,

   // the test ran but an assertion failed
   Failure
}
