package io.kotest.property.arbitrary

import io.kotest.property.Arb
import io.kotest.property.Exhaustive
import io.kotest.property.Gen
import io.kotest.property.RandomSource
import io.kotest.property.Sample
import io.kotest.property.map

/**
 * Returns a sequence of size [count] from values generated from this arb.
 * Edgecases will be ignored.
 */
fun <A> Arb<A>.take(count: Int, rs: RandomSource = RandomSource.default()): Sequence<A> =
   samples(rs).map { it.value }.take(count)

/**
 * Returns a single value generated from this arb ignoring edgecases.
 * Alias for next.
 */
fun <A> Arb<A>.single(rs: RandomSource = RandomSource.default()): A = this.samples(rs).map { it.value }.first()

/**
 * Returns a single value generated from this arb ignoring edgecases.
 * Alias for single.
 */
fun <A> Arb<A>.next(rs: RandomSource = RandomSource.default()): A = single(rs)

/**
 * Returns a new [Arb] which ensures only unique values are generated by keeping track of previously
 * generated values.
 *
 * Note: This arb can result in an infinite loop if more elements are requested than can be generated uniquely.
 */
@Deprecated("distinct will be removed in 4.6")
fun <A> Arb<A>.distinct() = distinctBy { it }

/**
 * Returns a new [Arb] which ensures only distinct keys returned by the given [selector] function are generated by
 * keeping track of previously generated values.
 *
 * Note: This arb can result in an infinite loop if more elements are requested than can be generated uniquely.
 */
@Deprecated("distinctBy will be removed in 4.6")
fun <A, B> Arb<A>.distinctBy(selector: (A) -> B) = object : Arb<A>() {

   override fun edgecase(rs: RandomSource): A? = this@distinctBy.edgecase(rs)

   override fun values(rs: RandomSource): Sequence<Sample<A>> {
      return this@distinctBy.values(rs).distinctBy { selector(it.value) }
   }

   override fun sample(rs: RandomSource): Sample<A> =
      this@distinctBy.samples(rs).distinctBy { selector(it.value) }.first()
}

/**
 * Wraps a [Arb] lazily. The given [f] is only evaluated once,
 * and not until the wrapper [Arb] is evaluated.
 **/
fun <A> Arb.Companion.lazy(f: () -> Arb<A>): Arb<A> {
   val arb by kotlin.lazy { f() }

   return object : Arb<A>() {
      override fun edgecase(rs: RandomSource): A? = arb.edgecase(rs)
      override fun values(rs: RandomSource): Sequence<Sample<A>> = arb.values(rs)
      override fun sample(rs: RandomSource): Sample<A> = arb.sample(rs)
   }
}
