/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.kroxylicious.proxy.filter;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.OptionalInt;

import org.apache.kafka.common.message.ApiMessageType;
import org.apache.kafka.common.message.RequestHeaderData;
import org.apache.kafka.common.message.ResponseHeaderData;
import org.apache.kafka.common.protocol.ApiKeys;
import org.apache.kafka.common.protocol.ApiMessage;

/**
 * Invoker for KrpcFilters that implement any number of Specific Message interfaces (for
 * example {@link io.kroxylicious.proxy.filter.AlterConfigsResponseFilter}.
 */
class SpecificFilterArrayInvoker implements FilterInvoker {

    private static final FilterInvoker[] HANDLE_NOTHING = createHandleNothing();

    private final FilterInvoker[] requestInvokers;
    private final FilterInvoker[] responseInvokers;

    SpecificFilterArrayInvoker(KrpcFilter filter) {
        Map<Integer, FilterInvoker> requestInvokers = new HashMap<>();
        Map<Integer, FilterInvoker> responseInvokers = new HashMap<>();
        if (filter instanceof AddOffsetsToTxnRequestFilter) {
            requestInvokers.put(25, new AddOffsetsToTxnRequestFilterInvoker((AddOffsetsToTxnRequestFilter) filter));
        }
        if (filter instanceof AddOffsetsToTxnResponseFilter) {
            responseInvokers.put(25, new AddOffsetsToTxnResponseFilterInvoker((AddOffsetsToTxnResponseFilter) filter));
        }
        if (filter instanceof AddPartitionsToTxnRequestFilter) {
            requestInvokers.put(24, new AddPartitionsToTxnRequestFilterInvoker((AddPartitionsToTxnRequestFilter) filter));
        }
        if (filter instanceof AddPartitionsToTxnResponseFilter) {
            responseInvokers.put(24, new AddPartitionsToTxnResponseFilterInvoker((AddPartitionsToTxnResponseFilter) filter));
        }
        if (filter instanceof AllocateProducerIdsRequestFilter) {
            requestInvokers.put(67, new AllocateProducerIdsRequestFilterInvoker((AllocateProducerIdsRequestFilter) filter));
        }
        if (filter instanceof AllocateProducerIdsResponseFilter) {
            responseInvokers.put(67, new AllocateProducerIdsResponseFilterInvoker((AllocateProducerIdsResponseFilter) filter));
        }
        if (filter instanceof AlterClientQuotasRequestFilter) {
            requestInvokers.put(49, new AlterClientQuotasRequestFilterInvoker((AlterClientQuotasRequestFilter) filter));
        }
        if (filter instanceof AlterClientQuotasResponseFilter) {
            responseInvokers.put(49, new AlterClientQuotasResponseFilterInvoker((AlterClientQuotasResponseFilter) filter));
        }
        if (filter instanceof AlterConfigsRequestFilter) {
            requestInvokers.put(33, new AlterConfigsRequestFilterInvoker((AlterConfigsRequestFilter) filter));
        }
        if (filter instanceof AlterConfigsResponseFilter) {
            responseInvokers.put(33, new AlterConfigsResponseFilterInvoker((AlterConfigsResponseFilter) filter));
        }
        if (filter instanceof AlterPartitionReassignmentsRequestFilter) {
            requestInvokers.put(45, new AlterPartitionReassignmentsRequestFilterInvoker((AlterPartitionReassignmentsRequestFilter) filter));
        }
        if (filter instanceof AlterPartitionReassignmentsResponseFilter) {
            responseInvokers.put(45, new AlterPartitionReassignmentsResponseFilterInvoker((AlterPartitionReassignmentsResponseFilter) filter));
        }
        if (filter instanceof AlterPartitionRequestFilter) {
            requestInvokers.put(56, new AlterPartitionRequestFilterInvoker((AlterPartitionRequestFilter) filter));
        }
        if (filter instanceof AlterPartitionResponseFilter) {
            responseInvokers.put(56, new AlterPartitionResponseFilterInvoker((AlterPartitionResponseFilter) filter));
        }
        if (filter instanceof AlterReplicaLogDirsRequestFilter) {
            requestInvokers.put(34, new AlterReplicaLogDirsRequestFilterInvoker((AlterReplicaLogDirsRequestFilter) filter));
        }
        if (filter instanceof AlterReplicaLogDirsResponseFilter) {
            responseInvokers.put(34, new AlterReplicaLogDirsResponseFilterInvoker((AlterReplicaLogDirsResponseFilter) filter));
        }
        if (filter instanceof AlterUserScramCredentialsRequestFilter) {
            requestInvokers.put(51, new AlterUserScramCredentialsRequestFilterInvoker((AlterUserScramCredentialsRequestFilter) filter));
        }
        if (filter instanceof AlterUserScramCredentialsResponseFilter) {
            responseInvokers.put(51, new AlterUserScramCredentialsResponseFilterInvoker((AlterUserScramCredentialsResponseFilter) filter));
        }
        if (filter instanceof ApiVersionsRequestFilter) {
            requestInvokers.put(18, new ApiVersionsRequestFilterInvoker((ApiVersionsRequestFilter) filter));
        }
        if (filter instanceof ApiVersionsResponseFilter) {
            responseInvokers.put(18, new ApiVersionsResponseFilterInvoker((ApiVersionsResponseFilter) filter));
        }
        if (filter instanceof BeginQuorumEpochRequestFilter) {
            requestInvokers.put(53, new BeginQuorumEpochRequestFilterInvoker((BeginQuorumEpochRequestFilter) filter));
        }
        if (filter instanceof BeginQuorumEpochResponseFilter) {
            responseInvokers.put(53, new BeginQuorumEpochResponseFilterInvoker((BeginQuorumEpochResponseFilter) filter));
        }
        if (filter instanceof BrokerHeartbeatRequestFilter) {
            requestInvokers.put(63, new BrokerHeartbeatRequestFilterInvoker((BrokerHeartbeatRequestFilter) filter));
        }
        if (filter instanceof BrokerHeartbeatResponseFilter) {
            responseInvokers.put(63, new BrokerHeartbeatResponseFilterInvoker((BrokerHeartbeatResponseFilter) filter));
        }
        if (filter instanceof BrokerRegistrationRequestFilter) {
            requestInvokers.put(62, new BrokerRegistrationRequestFilterInvoker((BrokerRegistrationRequestFilter) filter));
        }
        if (filter instanceof BrokerRegistrationResponseFilter) {
            responseInvokers.put(62, new BrokerRegistrationResponseFilterInvoker((BrokerRegistrationResponseFilter) filter));
        }
        if (filter instanceof ControlledShutdownRequestFilter) {
            requestInvokers.put(7, new ControlledShutdownRequestFilterInvoker((ControlledShutdownRequestFilter) filter));
        }
        if (filter instanceof ControlledShutdownResponseFilter) {
            responseInvokers.put(7, new ControlledShutdownResponseFilterInvoker((ControlledShutdownResponseFilter) filter));
        }
        if (filter instanceof CreateAclsRequestFilter) {
            requestInvokers.put(30, new CreateAclsRequestFilterInvoker((CreateAclsRequestFilter) filter));
        }
        if (filter instanceof CreateAclsResponseFilter) {
            responseInvokers.put(30, new CreateAclsResponseFilterInvoker((CreateAclsResponseFilter) filter));
        }
        if (filter instanceof CreateDelegationTokenRequestFilter) {
            requestInvokers.put(38, new CreateDelegationTokenRequestFilterInvoker((CreateDelegationTokenRequestFilter) filter));
        }
        if (filter instanceof CreateDelegationTokenResponseFilter) {
            responseInvokers.put(38, new CreateDelegationTokenResponseFilterInvoker((CreateDelegationTokenResponseFilter) filter));
        }
        if (filter instanceof CreatePartitionsRequestFilter) {
            requestInvokers.put(37, new CreatePartitionsRequestFilterInvoker((CreatePartitionsRequestFilter) filter));
        }
        if (filter instanceof CreatePartitionsResponseFilter) {
            responseInvokers.put(37, new CreatePartitionsResponseFilterInvoker((CreatePartitionsResponseFilter) filter));
        }
        if (filter instanceof CreateTopicsRequestFilter) {
            requestInvokers.put(19, new CreateTopicsRequestFilterInvoker((CreateTopicsRequestFilter) filter));
        }
        if (filter instanceof CreateTopicsResponseFilter) {
            responseInvokers.put(19, new CreateTopicsResponseFilterInvoker((CreateTopicsResponseFilter) filter));
        }
        if (filter instanceof DeleteAclsRequestFilter) {
            requestInvokers.put(31, new DeleteAclsRequestFilterInvoker((DeleteAclsRequestFilter) filter));
        }
        if (filter instanceof DeleteAclsResponseFilter) {
            responseInvokers.put(31, new DeleteAclsResponseFilterInvoker((DeleteAclsResponseFilter) filter));
        }
        if (filter instanceof DeleteGroupsRequestFilter) {
            requestInvokers.put(42, new DeleteGroupsRequestFilterInvoker((DeleteGroupsRequestFilter) filter));
        }
        if (filter instanceof DeleteGroupsResponseFilter) {
            responseInvokers.put(42, new DeleteGroupsResponseFilterInvoker((DeleteGroupsResponseFilter) filter));
        }
        if (filter instanceof DeleteRecordsRequestFilter) {
            requestInvokers.put(21, new DeleteRecordsRequestFilterInvoker((DeleteRecordsRequestFilter) filter));
        }
        if (filter instanceof DeleteRecordsResponseFilter) {
            responseInvokers.put(21, new DeleteRecordsResponseFilterInvoker((DeleteRecordsResponseFilter) filter));
        }
        if (filter instanceof DeleteTopicsRequestFilter) {
            requestInvokers.put(20, new DeleteTopicsRequestFilterInvoker((DeleteTopicsRequestFilter) filter));
        }
        if (filter instanceof DeleteTopicsResponseFilter) {
            responseInvokers.put(20, new DeleteTopicsResponseFilterInvoker((DeleteTopicsResponseFilter) filter));
        }
        if (filter instanceof DescribeAclsRequestFilter) {
            requestInvokers.put(29, new DescribeAclsRequestFilterInvoker((DescribeAclsRequestFilter) filter));
        }
        if (filter instanceof DescribeAclsResponseFilter) {
            responseInvokers.put(29, new DescribeAclsResponseFilterInvoker((DescribeAclsResponseFilter) filter));
        }
        if (filter instanceof DescribeClientQuotasRequestFilter) {
            requestInvokers.put(48, new DescribeClientQuotasRequestFilterInvoker((DescribeClientQuotasRequestFilter) filter));
        }
        if (filter instanceof DescribeClientQuotasResponseFilter) {
            responseInvokers.put(48, new DescribeClientQuotasResponseFilterInvoker((DescribeClientQuotasResponseFilter) filter));
        }
        if (filter instanceof DescribeClusterRequestFilter) {
            requestInvokers.put(60, new DescribeClusterRequestFilterInvoker((DescribeClusterRequestFilter) filter));
        }
        if (filter instanceof DescribeClusterResponseFilter) {
            responseInvokers.put(60, new DescribeClusterResponseFilterInvoker((DescribeClusterResponseFilter) filter));
        }
        if (filter instanceof DescribeConfigsRequestFilter) {
            requestInvokers.put(32, new DescribeConfigsRequestFilterInvoker((DescribeConfigsRequestFilter) filter));
        }
        if (filter instanceof DescribeConfigsResponseFilter) {
            responseInvokers.put(32, new DescribeConfigsResponseFilterInvoker((DescribeConfigsResponseFilter) filter));
        }
        if (filter instanceof DescribeDelegationTokenRequestFilter) {
            requestInvokers.put(41, new DescribeDelegationTokenRequestFilterInvoker((DescribeDelegationTokenRequestFilter) filter));
        }
        if (filter instanceof DescribeDelegationTokenResponseFilter) {
            responseInvokers.put(41, new DescribeDelegationTokenResponseFilterInvoker((DescribeDelegationTokenResponseFilter) filter));
        }
        if (filter instanceof DescribeGroupsRequestFilter) {
            requestInvokers.put(15, new DescribeGroupsRequestFilterInvoker((DescribeGroupsRequestFilter) filter));
        }
        if (filter instanceof DescribeGroupsResponseFilter) {
            responseInvokers.put(15, new DescribeGroupsResponseFilterInvoker((DescribeGroupsResponseFilter) filter));
        }
        if (filter instanceof DescribeLogDirsRequestFilter) {
            requestInvokers.put(35, new DescribeLogDirsRequestFilterInvoker((DescribeLogDirsRequestFilter) filter));
        }
        if (filter instanceof DescribeLogDirsResponseFilter) {
            responseInvokers.put(35, new DescribeLogDirsResponseFilterInvoker((DescribeLogDirsResponseFilter) filter));
        }
        if (filter instanceof DescribeProducersRequestFilter) {
            requestInvokers.put(61, new DescribeProducersRequestFilterInvoker((DescribeProducersRequestFilter) filter));
        }
        if (filter instanceof DescribeProducersResponseFilter) {
            responseInvokers.put(61, new DescribeProducersResponseFilterInvoker((DescribeProducersResponseFilter) filter));
        }
        if (filter instanceof DescribeQuorumRequestFilter) {
            requestInvokers.put(55, new DescribeQuorumRequestFilterInvoker((DescribeQuorumRequestFilter) filter));
        }
        if (filter instanceof DescribeQuorumResponseFilter) {
            responseInvokers.put(55, new DescribeQuorumResponseFilterInvoker((DescribeQuorumResponseFilter) filter));
        }
        if (filter instanceof DescribeTransactionsRequestFilter) {
            requestInvokers.put(65, new DescribeTransactionsRequestFilterInvoker((DescribeTransactionsRequestFilter) filter));
        }
        if (filter instanceof DescribeTransactionsResponseFilter) {
            responseInvokers.put(65, new DescribeTransactionsResponseFilterInvoker((DescribeTransactionsResponseFilter) filter));
        }
        if (filter instanceof DescribeUserScramCredentialsRequestFilter) {
            requestInvokers.put(50, new DescribeUserScramCredentialsRequestFilterInvoker((DescribeUserScramCredentialsRequestFilter) filter));
        }
        if (filter instanceof DescribeUserScramCredentialsResponseFilter) {
            responseInvokers.put(50, new DescribeUserScramCredentialsResponseFilterInvoker((DescribeUserScramCredentialsResponseFilter) filter));
        }
        if (filter instanceof ElectLeadersRequestFilter) {
            requestInvokers.put(43, new ElectLeadersRequestFilterInvoker((ElectLeadersRequestFilter) filter));
        }
        if (filter instanceof ElectLeadersResponseFilter) {
            responseInvokers.put(43, new ElectLeadersResponseFilterInvoker((ElectLeadersResponseFilter) filter));
        }
        if (filter instanceof EndQuorumEpochRequestFilter) {
            requestInvokers.put(54, new EndQuorumEpochRequestFilterInvoker((EndQuorumEpochRequestFilter) filter));
        }
        if (filter instanceof EndQuorumEpochResponseFilter) {
            responseInvokers.put(54, new EndQuorumEpochResponseFilterInvoker((EndQuorumEpochResponseFilter) filter));
        }
        if (filter instanceof EndTxnRequestFilter) {
            requestInvokers.put(26, new EndTxnRequestFilterInvoker((EndTxnRequestFilter) filter));
        }
        if (filter instanceof EndTxnResponseFilter) {
            responseInvokers.put(26, new EndTxnResponseFilterInvoker((EndTxnResponseFilter) filter));
        }
        if (filter instanceof EnvelopeRequestFilter) {
            requestInvokers.put(58, new EnvelopeRequestFilterInvoker((EnvelopeRequestFilter) filter));
        }
        if (filter instanceof EnvelopeResponseFilter) {
            responseInvokers.put(58, new EnvelopeResponseFilterInvoker((EnvelopeResponseFilter) filter));
        }
        if (filter instanceof ExpireDelegationTokenRequestFilter) {
            requestInvokers.put(40, new ExpireDelegationTokenRequestFilterInvoker((ExpireDelegationTokenRequestFilter) filter));
        }
        if (filter instanceof ExpireDelegationTokenResponseFilter) {
            responseInvokers.put(40, new ExpireDelegationTokenResponseFilterInvoker((ExpireDelegationTokenResponseFilter) filter));
        }
        if (filter instanceof FetchRequestFilter) {
            requestInvokers.put(1, new FetchRequestFilterInvoker((FetchRequestFilter) filter));
        }
        if (filter instanceof FetchResponseFilter) {
            responseInvokers.put(1, new FetchResponseFilterInvoker((FetchResponseFilter) filter));
        }
        if (filter instanceof FetchSnapshotRequestFilter) {
            requestInvokers.put(59, new FetchSnapshotRequestFilterInvoker((FetchSnapshotRequestFilter) filter));
        }
        if (filter instanceof FetchSnapshotResponseFilter) {
            responseInvokers.put(59, new FetchSnapshotResponseFilterInvoker((FetchSnapshotResponseFilter) filter));
        }
        if (filter instanceof FindCoordinatorRequestFilter) {
            requestInvokers.put(10, new FindCoordinatorRequestFilterInvoker((FindCoordinatorRequestFilter) filter));
        }
        if (filter instanceof FindCoordinatorResponseFilter) {
            responseInvokers.put(10, new FindCoordinatorResponseFilterInvoker((FindCoordinatorResponseFilter) filter));
        }
        if (filter instanceof HeartbeatRequestFilter) {
            requestInvokers.put(12, new HeartbeatRequestFilterInvoker((HeartbeatRequestFilter) filter));
        }
        if (filter instanceof HeartbeatResponseFilter) {
            responseInvokers.put(12, new HeartbeatResponseFilterInvoker((HeartbeatResponseFilter) filter));
        }
        if (filter instanceof IncrementalAlterConfigsRequestFilter) {
            requestInvokers.put(44, new IncrementalAlterConfigsRequestFilterInvoker((IncrementalAlterConfigsRequestFilter) filter));
        }
        if (filter instanceof IncrementalAlterConfigsResponseFilter) {
            responseInvokers.put(44, new IncrementalAlterConfigsResponseFilterInvoker((IncrementalAlterConfigsResponseFilter) filter));
        }
        if (filter instanceof InitProducerIdRequestFilter) {
            requestInvokers.put(22, new InitProducerIdRequestFilterInvoker((InitProducerIdRequestFilter) filter));
        }
        if (filter instanceof InitProducerIdResponseFilter) {
            responseInvokers.put(22, new InitProducerIdResponseFilterInvoker((InitProducerIdResponseFilter) filter));
        }
        if (filter instanceof JoinGroupRequestFilter) {
            requestInvokers.put(11, new JoinGroupRequestFilterInvoker((JoinGroupRequestFilter) filter));
        }
        if (filter instanceof JoinGroupResponseFilter) {
            responseInvokers.put(11, new JoinGroupResponseFilterInvoker((JoinGroupResponseFilter) filter));
        }
        if (filter instanceof LeaderAndIsrRequestFilter) {
            requestInvokers.put(4, new LeaderAndIsrRequestFilterInvoker((LeaderAndIsrRequestFilter) filter));
        }
        if (filter instanceof LeaderAndIsrResponseFilter) {
            responseInvokers.put(4, new LeaderAndIsrResponseFilterInvoker((LeaderAndIsrResponseFilter) filter));
        }
        if (filter instanceof LeaveGroupRequestFilter) {
            requestInvokers.put(13, new LeaveGroupRequestFilterInvoker((LeaveGroupRequestFilter) filter));
        }
        if (filter instanceof LeaveGroupResponseFilter) {
            responseInvokers.put(13, new LeaveGroupResponseFilterInvoker((LeaveGroupResponseFilter) filter));
        }
        if (filter instanceof ListGroupsRequestFilter) {
            requestInvokers.put(16, new ListGroupsRequestFilterInvoker((ListGroupsRequestFilter) filter));
        }
        if (filter instanceof ListGroupsResponseFilter) {
            responseInvokers.put(16, new ListGroupsResponseFilterInvoker((ListGroupsResponseFilter) filter));
        }
        if (filter instanceof ListOffsetsRequestFilter) {
            requestInvokers.put(2, new ListOffsetsRequestFilterInvoker((ListOffsetsRequestFilter) filter));
        }
        if (filter instanceof ListOffsetsResponseFilter) {
            responseInvokers.put(2, new ListOffsetsResponseFilterInvoker((ListOffsetsResponseFilter) filter));
        }
        if (filter instanceof ListPartitionReassignmentsRequestFilter) {
            requestInvokers.put(46, new ListPartitionReassignmentsRequestFilterInvoker((ListPartitionReassignmentsRequestFilter) filter));
        }
        if (filter instanceof ListPartitionReassignmentsResponseFilter) {
            responseInvokers.put(46, new ListPartitionReassignmentsResponseFilterInvoker((ListPartitionReassignmentsResponseFilter) filter));
        }
        if (filter instanceof ListTransactionsRequestFilter) {
            requestInvokers.put(66, new ListTransactionsRequestFilterInvoker((ListTransactionsRequestFilter) filter));
        }
        if (filter instanceof ListTransactionsResponseFilter) {
            responseInvokers.put(66, new ListTransactionsResponseFilterInvoker((ListTransactionsResponseFilter) filter));
        }
        if (filter instanceof MetadataRequestFilter) {
            requestInvokers.put(3, new MetadataRequestFilterInvoker((MetadataRequestFilter) filter));
        }
        if (filter instanceof MetadataResponseFilter) {
            responseInvokers.put(3, new MetadataResponseFilterInvoker((MetadataResponseFilter) filter));
        }
        if (filter instanceof OffsetCommitRequestFilter) {
            requestInvokers.put(8, new OffsetCommitRequestFilterInvoker((OffsetCommitRequestFilter) filter));
        }
        if (filter instanceof OffsetCommitResponseFilter) {
            responseInvokers.put(8, new OffsetCommitResponseFilterInvoker((OffsetCommitResponseFilter) filter));
        }
        if (filter instanceof OffsetDeleteRequestFilter) {
            requestInvokers.put(47, new OffsetDeleteRequestFilterInvoker((OffsetDeleteRequestFilter) filter));
        }
        if (filter instanceof OffsetDeleteResponseFilter) {
            responseInvokers.put(47, new OffsetDeleteResponseFilterInvoker((OffsetDeleteResponseFilter) filter));
        }
        if (filter instanceof OffsetFetchRequestFilter) {
            requestInvokers.put(9, new OffsetFetchRequestFilterInvoker((OffsetFetchRequestFilter) filter));
        }
        if (filter instanceof OffsetFetchResponseFilter) {
            responseInvokers.put(9, new OffsetFetchResponseFilterInvoker((OffsetFetchResponseFilter) filter));
        }
        if (filter instanceof OffsetForLeaderEpochRequestFilter) {
            requestInvokers.put(23, new OffsetForLeaderEpochRequestFilterInvoker((OffsetForLeaderEpochRequestFilter) filter));
        }
        if (filter instanceof OffsetForLeaderEpochResponseFilter) {
            responseInvokers.put(23, new OffsetForLeaderEpochResponseFilterInvoker((OffsetForLeaderEpochResponseFilter) filter));
        }
        if (filter instanceof ProduceRequestFilter) {
            requestInvokers.put(0, new ProduceRequestFilterInvoker((ProduceRequestFilter) filter));
        }
        if (filter instanceof ProduceResponseFilter) {
            responseInvokers.put(0, new ProduceResponseFilterInvoker((ProduceResponseFilter) filter));
        }
        if (filter instanceof RenewDelegationTokenRequestFilter) {
            requestInvokers.put(39, new RenewDelegationTokenRequestFilterInvoker((RenewDelegationTokenRequestFilter) filter));
        }
        if (filter instanceof RenewDelegationTokenResponseFilter) {
            responseInvokers.put(39, new RenewDelegationTokenResponseFilterInvoker((RenewDelegationTokenResponseFilter) filter));
        }
        if (filter instanceof SaslAuthenticateRequestFilter) {
            requestInvokers.put(36, new SaslAuthenticateRequestFilterInvoker((SaslAuthenticateRequestFilter) filter));
        }
        if (filter instanceof SaslAuthenticateResponseFilter) {
            responseInvokers.put(36, new SaslAuthenticateResponseFilterInvoker((SaslAuthenticateResponseFilter) filter));
        }
        if (filter instanceof SaslHandshakeRequestFilter) {
            requestInvokers.put(17, new SaslHandshakeRequestFilterInvoker((SaslHandshakeRequestFilter) filter));
        }
        if (filter instanceof SaslHandshakeResponseFilter) {
            responseInvokers.put(17, new SaslHandshakeResponseFilterInvoker((SaslHandshakeResponseFilter) filter));
        }
        if (filter instanceof StopReplicaRequestFilter) {
            requestInvokers.put(5, new StopReplicaRequestFilterInvoker((StopReplicaRequestFilter) filter));
        }
        if (filter instanceof StopReplicaResponseFilter) {
            responseInvokers.put(5, new StopReplicaResponseFilterInvoker((StopReplicaResponseFilter) filter));
        }
        if (filter instanceof SyncGroupRequestFilter) {
            requestInvokers.put(14, new SyncGroupRequestFilterInvoker((SyncGroupRequestFilter) filter));
        }
        if (filter instanceof SyncGroupResponseFilter) {
            responseInvokers.put(14, new SyncGroupResponseFilterInvoker((SyncGroupResponseFilter) filter));
        }
        if (filter instanceof TxnOffsetCommitRequestFilter) {
            requestInvokers.put(28, new TxnOffsetCommitRequestFilterInvoker((TxnOffsetCommitRequestFilter) filter));
        }
        if (filter instanceof TxnOffsetCommitResponseFilter) {
            responseInvokers.put(28, new TxnOffsetCommitResponseFilterInvoker((TxnOffsetCommitResponseFilter) filter));
        }
        if (filter instanceof UnregisterBrokerRequestFilter) {
            requestInvokers.put(64, new UnregisterBrokerRequestFilterInvoker((UnregisterBrokerRequestFilter) filter));
        }
        if (filter instanceof UnregisterBrokerResponseFilter) {
            responseInvokers.put(64, new UnregisterBrokerResponseFilterInvoker((UnregisterBrokerResponseFilter) filter));
        }
        if (filter instanceof UpdateFeaturesRequestFilter) {
            requestInvokers.put(57, new UpdateFeaturesRequestFilterInvoker((UpdateFeaturesRequestFilter) filter));
        }
        if (filter instanceof UpdateFeaturesResponseFilter) {
            responseInvokers.put(57, new UpdateFeaturesResponseFilterInvoker((UpdateFeaturesResponseFilter) filter));
        }
        if (filter instanceof UpdateMetadataRequestFilter) {
            requestInvokers.put(6, new UpdateMetadataRequestFilterInvoker((UpdateMetadataRequestFilter) filter));
        }
        if (filter instanceof UpdateMetadataResponseFilter) {
            responseInvokers.put(6, new UpdateMetadataResponseFilterInvoker((UpdateMetadataResponseFilter) filter));
        }
        if (filter instanceof VoteRequestFilter) {
            requestInvokers.put(52, new VoteRequestFilterInvoker((VoteRequestFilter) filter));
        }
        if (filter instanceof VoteResponseFilter) {
            responseInvokers.put(52, new VoteResponseFilterInvoker((VoteResponseFilter) filter));
        }
        if (filter instanceof WriteTxnMarkersRequestFilter) {
            requestInvokers.put(27, new WriteTxnMarkersRequestFilterInvoker((WriteTxnMarkersRequestFilter) filter));
        }
        if (filter instanceof WriteTxnMarkersResponseFilter) {
            responseInvokers.put(27, new WriteTxnMarkersResponseFilterInvoker((WriteTxnMarkersResponseFilter) filter));
        }
        this.requestInvokers = createFrom(requestInvokers);
        this.responseInvokers = createFrom(responseInvokers);
    }

    /**
     * Apply the filter to the given {@code header} and {@code body} using the given {@code filterContext}.
     * @param apiKey The request api key.
     * @param apiVersion The request api version.
     * @param header The request header.
     * @param body The request body.
     * @param filterContext The filter context.
     */
    @Override
    @SuppressWarnings("SwitchStatementWithTooFewBranches")
    public void onRequest(ApiKeys apiKey,
                           short apiVersion,
                           RequestHeaderData header,
                           ApiMessage body,
                           KrpcFilterContext filterContext) {
        // We wrap the array lookup in a switch based on the API Key as it supports JIT optimisations around method dispatch.
        // See the InvokerDispatchBenchmark micro benchmark for a comparison
        switch (apiKey) {
            case ADD_OFFSETS_TO_TXN ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case ADD_PARTITIONS_TO_TXN ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case ALLOCATE_PRODUCER_IDS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case ALTER_CLIENT_QUOTAS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case ALTER_CONFIGS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case ALTER_PARTITION_REASSIGNMENTS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case ALTER_PARTITION ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case ALTER_REPLICA_LOG_DIRS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case ALTER_USER_SCRAM_CREDENTIALS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case API_VERSIONS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case BEGIN_QUORUM_EPOCH ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case BROKER_HEARTBEAT ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case BROKER_REGISTRATION ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case CONTROLLED_SHUTDOWN ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case CREATE_ACLS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case CREATE_DELEGATION_TOKEN ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case CREATE_PARTITIONS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case CREATE_TOPICS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case DELETE_ACLS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case DELETE_GROUPS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case DELETE_RECORDS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case DELETE_TOPICS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case DESCRIBE_ACLS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case DESCRIBE_CLIENT_QUOTAS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case DESCRIBE_CLUSTER ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case DESCRIBE_CONFIGS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case DESCRIBE_DELEGATION_TOKEN ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case DESCRIBE_GROUPS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case DESCRIBE_LOG_DIRS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case DESCRIBE_PRODUCERS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case DESCRIBE_QUORUM ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case DESCRIBE_TRANSACTIONS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case DESCRIBE_USER_SCRAM_CREDENTIALS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case ELECT_LEADERS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case END_QUORUM_EPOCH ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case END_TXN ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case ENVELOPE ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case EXPIRE_DELEGATION_TOKEN ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case FETCH ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case FETCH_SNAPSHOT ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case FIND_COORDINATOR ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case HEARTBEAT ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case INCREMENTAL_ALTER_CONFIGS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case INIT_PRODUCER_ID ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case JOIN_GROUP ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case LEADER_AND_ISR ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case LEAVE_GROUP ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case LIST_GROUPS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case LIST_OFFSETS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case LIST_PARTITION_REASSIGNMENTS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case LIST_TRANSACTIONS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case METADATA ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case OFFSET_COMMIT ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case OFFSET_DELETE ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case OFFSET_FETCH ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case OFFSET_FOR_LEADER_EPOCH ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case PRODUCE ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case RENEW_DELEGATION_TOKEN ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case SASL_AUTHENTICATE ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case SASL_HANDSHAKE ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case STOP_REPLICA ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case SYNC_GROUP ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case TXN_OFFSET_COMMIT ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case UNREGISTER_BROKER ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case UPDATE_FEATURES ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case UPDATE_METADATA ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case VOTE ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            case WRITE_TXN_MARKERS ->
                requestInvokers[apiKey.id].onRequest(apiKey, apiVersion, header, body, filterContext);
            default -> throw new IllegalStateException("Unsupported RPC " + apiKey);
        }

    }

    /**
     * Apply the filter to the given {@code header} and {@code body} using the given {@code filterContext}.
     * @param apiKey The request api key.
     * @param apiVersion The api version.
     * @param header The request header.
     * @param body The request body.
     * @param filterContext The filter context.
     */
    @Override
    @SuppressWarnings("SwitchStatementWithTooFewBranches")
    public void onResponse(ApiKeys apiKey,
                            short apiVersion,
                            ResponseHeaderData header,
                            ApiMessage body,
                            KrpcFilterContext filterContext) {
        // We wrap the array lookup in a switch based on the API Key as it supports JIT optimisations around method dispatch.
        // See the InvokerDispatchBenchmark micro benchmark for a comparison
        switch (apiKey) {
            case ADD_OFFSETS_TO_TXN ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case ADD_PARTITIONS_TO_TXN ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case ALLOCATE_PRODUCER_IDS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case ALTER_CLIENT_QUOTAS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case ALTER_CONFIGS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case ALTER_PARTITION_REASSIGNMENTS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case ALTER_PARTITION ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case ALTER_REPLICA_LOG_DIRS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case ALTER_USER_SCRAM_CREDENTIALS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case API_VERSIONS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case BEGIN_QUORUM_EPOCH ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case BROKER_HEARTBEAT ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case BROKER_REGISTRATION ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case CONTROLLED_SHUTDOWN ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case CREATE_ACLS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case CREATE_DELEGATION_TOKEN ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case CREATE_PARTITIONS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case CREATE_TOPICS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case DELETE_ACLS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case DELETE_GROUPS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case DELETE_RECORDS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case DELETE_TOPICS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case DESCRIBE_ACLS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case DESCRIBE_CLIENT_QUOTAS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case DESCRIBE_CLUSTER ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case DESCRIBE_CONFIGS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case DESCRIBE_DELEGATION_TOKEN ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case DESCRIBE_GROUPS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case DESCRIBE_LOG_DIRS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case DESCRIBE_PRODUCERS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case DESCRIBE_QUORUM ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case DESCRIBE_TRANSACTIONS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case DESCRIBE_USER_SCRAM_CREDENTIALS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case ELECT_LEADERS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case END_QUORUM_EPOCH ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case END_TXN ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case ENVELOPE ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case EXPIRE_DELEGATION_TOKEN ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case FETCH ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case FETCH_SNAPSHOT ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case FIND_COORDINATOR ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case HEARTBEAT ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case INCREMENTAL_ALTER_CONFIGS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case INIT_PRODUCER_ID ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case JOIN_GROUP ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case LEADER_AND_ISR ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case LEAVE_GROUP ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case LIST_GROUPS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case LIST_OFFSETS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case LIST_PARTITION_REASSIGNMENTS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case LIST_TRANSACTIONS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case METADATA ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case OFFSET_COMMIT ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case OFFSET_DELETE ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case OFFSET_FETCH ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case OFFSET_FOR_LEADER_EPOCH ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case PRODUCE ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case RENEW_DELEGATION_TOKEN ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case SASL_AUTHENTICATE ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case SASL_HANDSHAKE ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case STOP_REPLICA ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case SYNC_GROUP ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case TXN_OFFSET_COMMIT ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case UNREGISTER_BROKER ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case UPDATE_FEATURES ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case UPDATE_METADATA ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case VOTE ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            case WRITE_TXN_MARKERS ->
                    responseInvokers[apiKey.id].onResponse(apiKey, apiVersion, header, body, filterContext);
            default -> throw new IllegalStateException("Unsupported RPC " + apiKey);
        }

    }

    /**
     * <p>Determines whether a request with the given {@code apiKey} and {@code apiVersion} should be deserialized.
     * Note that it is not guaranteed that this method will be called once per request,
     * or that two consecutive calls refer to the same request.
     * That is, the sequences of invocations like the following are allowed:</p>
     * <ol>
     *     <li>{@code shouldHandleRequest} on request A</li>
     *     <li>{@code shouldHandleRequest} on request B</li>
     *     <li>{@code shouldHandleRequest} on request A</li>
     *     <li>{@code onRequest} on request A</li>
     *     <li>{@code onRequest} on request B</li>
     * </ol>
     * @param apiKey The API key
     * @param apiVersion The API version
     * @return true if request should be deserialized
     */
    @Override
    @SuppressWarnings("SwitchStatementWithTooFewBranches")
    public boolean shouldHandleRequest(ApiKeys apiKey, short apiVersion) {
        // We wrap the array lookup in a switch based on the API Key as it supports JIT optimisations around method dispatch.
        // See the InvokerDispatchBenchmark micro benchmark for a comparison
        return switch (apiKey) {
            case ADD_OFFSETS_TO_TXN ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case ADD_PARTITIONS_TO_TXN ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case ALLOCATE_PRODUCER_IDS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case ALTER_CLIENT_QUOTAS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case ALTER_CONFIGS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case ALTER_PARTITION_REASSIGNMENTS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case ALTER_PARTITION ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case ALTER_REPLICA_LOG_DIRS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case ALTER_USER_SCRAM_CREDENTIALS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case API_VERSIONS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case BEGIN_QUORUM_EPOCH ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case BROKER_HEARTBEAT ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case BROKER_REGISTRATION ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case CONTROLLED_SHUTDOWN ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case CREATE_ACLS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case CREATE_DELEGATION_TOKEN ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case CREATE_PARTITIONS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case CREATE_TOPICS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case DELETE_ACLS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case DELETE_GROUPS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case DELETE_RECORDS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case DELETE_TOPICS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case DESCRIBE_ACLS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case DESCRIBE_CLIENT_QUOTAS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case DESCRIBE_CLUSTER ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case DESCRIBE_CONFIGS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case DESCRIBE_DELEGATION_TOKEN ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case DESCRIBE_GROUPS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case DESCRIBE_LOG_DIRS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case DESCRIBE_PRODUCERS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case DESCRIBE_QUORUM ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case DESCRIBE_TRANSACTIONS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case DESCRIBE_USER_SCRAM_CREDENTIALS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case ELECT_LEADERS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case END_QUORUM_EPOCH ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case END_TXN ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case ENVELOPE ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case EXPIRE_DELEGATION_TOKEN ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case FETCH ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case FETCH_SNAPSHOT ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case FIND_COORDINATOR ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case HEARTBEAT ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case INCREMENTAL_ALTER_CONFIGS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case INIT_PRODUCER_ID ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case JOIN_GROUP ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case LEADER_AND_ISR ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case LEAVE_GROUP ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case LIST_GROUPS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case LIST_OFFSETS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case LIST_PARTITION_REASSIGNMENTS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case LIST_TRANSACTIONS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case METADATA ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case OFFSET_COMMIT ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case OFFSET_DELETE ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case OFFSET_FETCH ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case OFFSET_FOR_LEADER_EPOCH ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case PRODUCE ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case RENEW_DELEGATION_TOKEN ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case SASL_AUTHENTICATE ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case SASL_HANDSHAKE ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case STOP_REPLICA ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case SYNC_GROUP ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case TXN_OFFSET_COMMIT ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case UNREGISTER_BROKER ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case UPDATE_FEATURES ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case UPDATE_METADATA ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case VOTE ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            case WRITE_TXN_MARKERS ->
                    requestInvokers[apiKey.id].shouldHandleRequest(apiKey, apiVersion);
            default -> throw new IllegalStateException("Unsupported RPC " + apiKey);
        };

    }

    /**
     * <p>Determines whether a response with the given {@code apiKey} and {@code apiVersion} should be deserialized.
     * Note that it is not guaranteed that this method will be called once per response,
     * or that two consecutive calls refer to the same response.
     * That is, the sequences of invocations like the following are allowed:</p>
     * <ol>
     *     <li>{@code shouldHandleResponse} on response A</li>
     *     <li>{@code shouldHandleResponse} on response B</li>
     *     <li>{@code shouldHandleResponse} on response A</li>
     *     <li>{@code apply} on response A</li>
     *     <li>{@code apply} on response B</li>
     * </ol>
     * @param apiKey The API key
     * @param apiVersion The API version
     * @return true if response should be deserialized
     */
    @Override
    @SuppressWarnings("SwitchStatementWithTooFewBranches")
    public boolean shouldHandleResponse(ApiKeys apiKey, short apiVersion) {
        return switch (apiKey) {
            // We wrap the array lookup in a switch based on the API Key as it supports JIT optimisations around method dispatch.
            // See the InvokerDispatchBenchmark micro benchmark for a comparison
            case ADD_OFFSETS_TO_TXN ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case ADD_PARTITIONS_TO_TXN ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case ALLOCATE_PRODUCER_IDS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case ALTER_CLIENT_QUOTAS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case ALTER_CONFIGS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case ALTER_PARTITION_REASSIGNMENTS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case ALTER_PARTITION ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case ALTER_REPLICA_LOG_DIRS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case ALTER_USER_SCRAM_CREDENTIALS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case API_VERSIONS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case BEGIN_QUORUM_EPOCH ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case BROKER_HEARTBEAT ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case BROKER_REGISTRATION ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case CONTROLLED_SHUTDOWN ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case CREATE_ACLS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case CREATE_DELEGATION_TOKEN ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case CREATE_PARTITIONS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case CREATE_TOPICS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case DELETE_ACLS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case DELETE_GROUPS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case DELETE_RECORDS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case DELETE_TOPICS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case DESCRIBE_ACLS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case DESCRIBE_CLIENT_QUOTAS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case DESCRIBE_CLUSTER ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case DESCRIBE_CONFIGS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case DESCRIBE_DELEGATION_TOKEN ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case DESCRIBE_GROUPS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case DESCRIBE_LOG_DIRS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case DESCRIBE_PRODUCERS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case DESCRIBE_QUORUM ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case DESCRIBE_TRANSACTIONS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case DESCRIBE_USER_SCRAM_CREDENTIALS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case ELECT_LEADERS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case END_QUORUM_EPOCH ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case END_TXN ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case ENVELOPE ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case EXPIRE_DELEGATION_TOKEN ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case FETCH ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case FETCH_SNAPSHOT ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case FIND_COORDINATOR ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case HEARTBEAT ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case INCREMENTAL_ALTER_CONFIGS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case INIT_PRODUCER_ID ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case JOIN_GROUP ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case LEADER_AND_ISR ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case LEAVE_GROUP ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case LIST_GROUPS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case LIST_OFFSETS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case LIST_PARTITION_REASSIGNMENTS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case LIST_TRANSACTIONS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case METADATA ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case OFFSET_COMMIT ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case OFFSET_DELETE ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case OFFSET_FETCH ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case OFFSET_FOR_LEADER_EPOCH ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case PRODUCE ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case RENEW_DELEGATION_TOKEN ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case SASL_AUTHENTICATE ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case SASL_HANDSHAKE ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case STOP_REPLICA ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case SYNC_GROUP ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case TXN_OFFSET_COMMIT ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case UNREGISTER_BROKER ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case UPDATE_FEATURES ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case UPDATE_METADATA ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case VOTE ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            case WRITE_TXN_MARKERS ->
                    responseInvokers[apiKey.id].shouldHandleResponse(apiKey, apiVersion);
            default -> throw new IllegalStateException("Unsupported RPC " + apiKey);
        };

    }

    /**
    * Check if a KrpcFilter implements any of the Specific Message Filter interfaces
    * @param filter the filter
    * @return true if the filter implements any Specific Message Filter interfaces
    */
    public static boolean implementsAnySpecificFilterInterface(KrpcFilter filter) {
        return filter instanceof AddOffsetsToTxnRequestFilter ||
        filter instanceof AddOffsetsToTxnResponseFilter ||
        filter instanceof AddPartitionsToTxnRequestFilter ||
        filter instanceof AddPartitionsToTxnResponseFilter ||
        filter instanceof AllocateProducerIdsRequestFilter ||
        filter instanceof AllocateProducerIdsResponseFilter ||
        filter instanceof AlterClientQuotasRequestFilter ||
        filter instanceof AlterClientQuotasResponseFilter ||
        filter instanceof AlterConfigsRequestFilter ||
        filter instanceof AlterConfigsResponseFilter ||
        filter instanceof AlterPartitionReassignmentsRequestFilter ||
        filter instanceof AlterPartitionReassignmentsResponseFilter ||
        filter instanceof AlterPartitionRequestFilter ||
        filter instanceof AlterPartitionResponseFilter ||
        filter instanceof AlterReplicaLogDirsRequestFilter ||
        filter instanceof AlterReplicaLogDirsResponseFilter ||
        filter instanceof AlterUserScramCredentialsRequestFilter ||
        filter instanceof AlterUserScramCredentialsResponseFilter ||
        filter instanceof ApiVersionsRequestFilter ||
        filter instanceof ApiVersionsResponseFilter ||
        filter instanceof BeginQuorumEpochRequestFilter ||
        filter instanceof BeginQuorumEpochResponseFilter ||
        filter instanceof BrokerHeartbeatRequestFilter ||
        filter instanceof BrokerHeartbeatResponseFilter ||
        filter instanceof BrokerRegistrationRequestFilter ||
        filter instanceof BrokerRegistrationResponseFilter ||
        filter instanceof ControlledShutdownRequestFilter ||
        filter instanceof ControlledShutdownResponseFilter ||
        filter instanceof CreateAclsRequestFilter ||
        filter instanceof CreateAclsResponseFilter ||
        filter instanceof CreateDelegationTokenRequestFilter ||
        filter instanceof CreateDelegationTokenResponseFilter ||
        filter instanceof CreatePartitionsRequestFilter ||
        filter instanceof CreatePartitionsResponseFilter ||
        filter instanceof CreateTopicsRequestFilter ||
        filter instanceof CreateTopicsResponseFilter ||
        filter instanceof DeleteAclsRequestFilter ||
        filter instanceof DeleteAclsResponseFilter ||
        filter instanceof DeleteGroupsRequestFilter ||
        filter instanceof DeleteGroupsResponseFilter ||
        filter instanceof DeleteRecordsRequestFilter ||
        filter instanceof DeleteRecordsResponseFilter ||
        filter instanceof DeleteTopicsRequestFilter ||
        filter instanceof DeleteTopicsResponseFilter ||
        filter instanceof DescribeAclsRequestFilter ||
        filter instanceof DescribeAclsResponseFilter ||
        filter instanceof DescribeClientQuotasRequestFilter ||
        filter instanceof DescribeClientQuotasResponseFilter ||
        filter instanceof DescribeClusterRequestFilter ||
        filter instanceof DescribeClusterResponseFilter ||
        filter instanceof DescribeConfigsRequestFilter ||
        filter instanceof DescribeConfigsResponseFilter ||
        filter instanceof DescribeDelegationTokenRequestFilter ||
        filter instanceof DescribeDelegationTokenResponseFilter ||
        filter instanceof DescribeGroupsRequestFilter ||
        filter instanceof DescribeGroupsResponseFilter ||
        filter instanceof DescribeLogDirsRequestFilter ||
        filter instanceof DescribeLogDirsResponseFilter ||
        filter instanceof DescribeProducersRequestFilter ||
        filter instanceof DescribeProducersResponseFilter ||
        filter instanceof DescribeQuorumRequestFilter ||
        filter instanceof DescribeQuorumResponseFilter ||
        filter instanceof DescribeTransactionsRequestFilter ||
        filter instanceof DescribeTransactionsResponseFilter ||
        filter instanceof DescribeUserScramCredentialsRequestFilter ||
        filter instanceof DescribeUserScramCredentialsResponseFilter ||
        filter instanceof ElectLeadersRequestFilter ||
        filter instanceof ElectLeadersResponseFilter ||
        filter instanceof EndQuorumEpochRequestFilter ||
        filter instanceof EndQuorumEpochResponseFilter ||
        filter instanceof EndTxnRequestFilter ||
        filter instanceof EndTxnResponseFilter ||
        filter instanceof EnvelopeRequestFilter ||
        filter instanceof EnvelopeResponseFilter ||
        filter instanceof ExpireDelegationTokenRequestFilter ||
        filter instanceof ExpireDelegationTokenResponseFilter ||
        filter instanceof FetchRequestFilter ||
        filter instanceof FetchResponseFilter ||
        filter instanceof FetchSnapshotRequestFilter ||
        filter instanceof FetchSnapshotResponseFilter ||
        filter instanceof FindCoordinatorRequestFilter ||
        filter instanceof FindCoordinatorResponseFilter ||
        filter instanceof HeartbeatRequestFilter ||
        filter instanceof HeartbeatResponseFilter ||
        filter instanceof IncrementalAlterConfigsRequestFilter ||
        filter instanceof IncrementalAlterConfigsResponseFilter ||
        filter instanceof InitProducerIdRequestFilter ||
        filter instanceof InitProducerIdResponseFilter ||
        filter instanceof JoinGroupRequestFilter ||
        filter instanceof JoinGroupResponseFilter ||
        filter instanceof LeaderAndIsrRequestFilter ||
        filter instanceof LeaderAndIsrResponseFilter ||
        filter instanceof LeaveGroupRequestFilter ||
        filter instanceof LeaveGroupResponseFilter ||
        filter instanceof ListGroupsRequestFilter ||
        filter instanceof ListGroupsResponseFilter ||
        filter instanceof ListOffsetsRequestFilter ||
        filter instanceof ListOffsetsResponseFilter ||
        filter instanceof ListPartitionReassignmentsRequestFilter ||
        filter instanceof ListPartitionReassignmentsResponseFilter ||
        filter instanceof ListTransactionsRequestFilter ||
        filter instanceof ListTransactionsResponseFilter ||
        filter instanceof MetadataRequestFilter ||
        filter instanceof MetadataResponseFilter ||
        filter instanceof OffsetCommitRequestFilter ||
        filter instanceof OffsetCommitResponseFilter ||
        filter instanceof OffsetDeleteRequestFilter ||
        filter instanceof OffsetDeleteResponseFilter ||
        filter instanceof OffsetFetchRequestFilter ||
        filter instanceof OffsetFetchResponseFilter ||
        filter instanceof OffsetForLeaderEpochRequestFilter ||
        filter instanceof OffsetForLeaderEpochResponseFilter ||
        filter instanceof ProduceRequestFilter ||
        filter instanceof ProduceResponseFilter ||
        filter instanceof RenewDelegationTokenRequestFilter ||
        filter instanceof RenewDelegationTokenResponseFilter ||
        filter instanceof SaslAuthenticateRequestFilter ||
        filter instanceof SaslAuthenticateResponseFilter ||
        filter instanceof SaslHandshakeRequestFilter ||
        filter instanceof SaslHandshakeResponseFilter ||
        filter instanceof StopReplicaRequestFilter ||
        filter instanceof StopReplicaResponseFilter ||
        filter instanceof SyncGroupRequestFilter ||
        filter instanceof SyncGroupResponseFilter ||
        filter instanceof TxnOffsetCommitRequestFilter ||
        filter instanceof TxnOffsetCommitResponseFilter ||
        filter instanceof UnregisterBrokerRequestFilter ||
        filter instanceof UnregisterBrokerResponseFilter ||
        filter instanceof UpdateFeaturesRequestFilter ||
        filter instanceof UpdateFeaturesResponseFilter ||
        filter instanceof UpdateMetadataRequestFilter ||
        filter instanceof UpdateMetadataResponseFilter ||
        filter instanceof VoteRequestFilter ||
        filter instanceof VoteResponseFilter ||
        filter instanceof WriteTxnMarkersRequestFilter ||
        filter instanceof WriteTxnMarkersResponseFilter;
    }

    private static FilterInvoker[] createHandleNothing() {
        FilterInvoker[] filterInvokers = emptyInvokerArraySizedForMessageTypes();
        Arrays.stream(ApiMessageType.values()).mapToInt(ApiMessageType::apiKey).forEach(value -> filterInvokers[value] = FilterInvokers.handleNothingInvoker());
        return filterInvokers;
    }

    private static FilterInvoker[] createFrom(Map<Integer, FilterInvoker> filterInvokersByApiMessageId) {
        if (filterInvokersByApiMessageId.isEmpty()) {
            return HANDLE_NOTHING;
        }
        FilterInvoker[] filterInvokers = emptyInvokerArraySizedForMessageTypes();
        Arrays.stream(ApiMessageType.values()).mapToInt(ApiMessageType::apiKey).forEach(value -> filterInvokers[value] = filterInvokersByApiMessageId.getOrDefault(value, FilterInvokers.handleNothingInvoker()));
        return filterInvokers;
    }

    private static FilterInvoker[] emptyInvokerArraySizedForMessageTypes() {
        OptionalInt maybeMaxId = Arrays.stream(ApiMessageType.values()).mapToInt(ApiMessageType::apiKey).max();
        if (maybeMaxId.isEmpty()) {
            throw new IllegalStateException("no maximum id found");
        }
        int arraySize = maybeMaxId.getAsInt() + 1;
        return new FilterInvoker[arraySize];
    }

}
