package io.mangoo.utils;

import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Deque;
import java.util.HashMap;
import java.util.Map;

import com.google.common.base.Preconditions;

import io.mangoo.enums.ContentType;
import io.mangoo.enums.Default;
import io.undertow.server.HttpServerExchange;
import io.undertow.util.HeaderMap;
import io.undertow.util.Headers;
import io.undertow.util.Methods;

/**
 *
 * @author svenkubiak
 *
 */
public final class RequestUtils {
    private static Map<String, ZonedDateTime> modified = new HashMap<String, ZonedDateTime>();
    private static final DateTimeFormatter FORMATTER = DateTimeFormatter.ofPattern(Default.FORMATTER.toString());

    private RequestUtils() {
    }

    /**
     * Converts request and query parameter into a single map
     *
     * @param exchange The Undertow HttpServerExchange
     * @return A single map contain both request and query parameter
     */
    public static Map<String, String> getRequestParameters(HttpServerExchange exchange) {
        Map<String, String> requestParamater = new HashMap<String, String>();

        Map<String, Deque<String>> queryParameters = exchange.getQueryParameters();
        queryParameters.putAll(exchange.getPathParameters());
        queryParameters.entrySet().forEach(entry -> requestParamater.put(entry.getKey(), entry.getValue().element())); //NOSONAR

        return requestParamater;
    }

    /**
     * Checks if a given template name has the current suffix and sets is
     * if it does not exist
     *
     * @param templateName The name of the template file
     * @return The template name with correct suffix
     */
    public static String getTemplateName(String templateName) {
        return templateName.endsWith(Default.TEMPLATE_SUFFIX.toString()) ? templateName : templateName + Default.TEMPLATE_SUFFIX.toString();
    }

    /**
     * Checks if the request is a POST or a PUT request
     *
     * @param exchange The Undertow HttpServerExchange
     * @return True if the request is a POST or a PUT request, false otherwise
     */
    public static boolean isPostOrPut(HttpServerExchange exchange) {
        return (Methods.POST).equals(exchange.getRequestMethod()) || (Methods.PUT).equals(exchange.getRequestMethod());
    }

    /**
     * Checks if the requests content-type contains application/json
     *
     * @param exchange The Undertow HttpServerExchange
     * @return True if the request content-type contains application/json, false otherwise
     */
    public static boolean isJSONRequest(HttpServerExchange exchange) {
        HeaderMap requestHeaders = exchange.getRequestHeaders();

        return requestHeaders != null && requestHeaders.get(Headers.CONTENT_TYPE) != null &&
                requestHeaders.get(Headers.CONTENT_TYPE).element().toLowerCase().contains(ContentType.APPLICATION_JSON.toString().toLowerCase());
    }

    /**
     * Checks if a modified timestamp is stored and compares the two dates
     *
     * @param key The key the modified value is stored
     * @param value The LocalDateTime value to compare with
     * @return True if a stored value exists and is equal to the given value, false otherwise
     */
    public static boolean isNotModifiedContent(String key, String value) {
        Preconditions.checkNotNull(key, "A key is required for checking last modified");
        Preconditions.checkNotNull(value, "A value is required for checking last modified");

        ZonedDateTime zonedDateTime = modified.get(key);

        if (zonedDateTime == null) {
            return false;
        }

        return ZonedDateTime.parse(value, FORMATTER).equals(modified.get(key));
    }

    /**
     * Adds an additional value to the modified map
     *
     * @param key The key to store the value
     * @param value The LocalDateTime timestamp
     */
    public static void addModifiedContent(String key, ZonedDateTime value) {
        Preconditions.checkNotNull(key, "A key is required for adding last modified");
        Preconditions.checkNotNull(value, "A value is required for adding last modified");

        modified.put(key, value);
    }
}