/*
 * Decompiled with CFR 0.152.
 */
package io.mantisrx.server.master.persistence;

import io.mantisrx.master.jobcluster.IJobClusterMetadata;
import io.mantisrx.master.jobcluster.job.IMantisJobMetadata;
import io.mantisrx.master.jobcluster.job.IMantisStageMetadata;
import io.mantisrx.master.jobcluster.job.MantisStageMetadataImpl;
import io.mantisrx.master.jobcluster.job.worker.IMantisWorkerMetadata;
import io.mantisrx.master.jobcluster.job.worker.JobWorker;
import io.mantisrx.master.resourcecluster.DisableTaskExecutorsRequest;
import io.mantisrx.server.core.domain.ArtifactID;
import io.mantisrx.server.core.domain.JobArtifact;
import io.mantisrx.server.master.config.ConfigurationProvider;
import io.mantisrx.server.master.domain.JobClusterDefinitionImpl;
import io.mantisrx.server.master.domain.JobId;
import io.mantisrx.server.master.persistence.IMantisPersistenceProvider;
import io.mantisrx.server.master.persistence.exceptions.InvalidJobException;
import io.mantisrx.server.master.resourcecluster.ClusterID;
import io.mantisrx.server.master.resourcecluster.TaskExecutorID;
import io.mantisrx.server.master.resourcecluster.TaskExecutorRegistration;
import io.mantisrx.shaded.com.google.common.cache.Cache;
import io.mantisrx.shaded.com.google.common.cache.CacheBuilder;
import io.mantisrx.shaded.com.google.common.collect.ImmutableList;
import io.mantisrx.shaded.com.google.common.collect.Lists;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.PriorityBlockingQueue;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import rx.schedulers.Schedulers;

public class MantisJobStore {
    private static final Logger logger = LoggerFactory.getLogger(MantisJobStore.class);
    private final IMantisPersistenceProvider storageProvider;
    private final ConcurrentMap<String, String> archivedJobIds;
    private final ArchivedJobsMetadataCache archivedJobsMetadataCache;
    private final ArchivedWorkersCache archivedWorkersCache;
    private final PriorityBlockingQueue<TerminatedJob> terminatedJobsToDelete;

    public MantisJobStore(IMantisPersistenceProvider storageProvider) {
        this.storageProvider = storageProvider;
        this.archivedJobIds = new ConcurrentHashMap<String, String>();
        this.archivedWorkersCache = new ArchivedWorkersCache(ConfigurationProvider.getConfig().getMaxArchivedJobsToCache());
        this.archivedJobsMetadataCache = new ArchivedJobsMetadataCache(ConfigurationProvider.getConfig().getMaxArchivedJobsToCache());
        this.terminatedJobsToDelete = new PriorityBlockingQueue();
    }

    public void loadAllArchivedJobsAsync() {
        logger.info("Beginning load of Archived Jobs");
        this.storageProvider.loadAllArchivedJobs().subscribeOn(Schedulers.io()).subscribe(job -> {
            this.archivedJobsMetadataCache.add((IMantisJobMetadata)job);
            this.archivedJobIds.put(job.getJobId().getId(), job.getJobId().getId());
            this.terminatedJobsToDelete.add(new TerminatedJob(job.getJobId().getId(), this.getTerminatedAt((IMantisJobMetadata)job)));
        }, e -> logger.warn("Exception loading archived Jobs", e), () -> logger.info("Finished Loading all archived Jobs!"));
    }

    private long getTerminatedAt(IMantisJobMetadata mjmd) {
        long terminatedAt = mjmd.getSubmittedAtInstant().toEpochMilli();
        for (IMantisStageMetadata iMantisStageMetadata : mjmd.getStageMetadata().values()) {
            for (JobWorker mwmd : iMantisStageMetadata.getAllWorkers()) {
                terminatedAt = Math.max(terminatedAt, mwmd.getMetadata().getCompletedAt());
            }
        }
        return terminatedAt;
    }

    public List<IJobClusterMetadata> loadAllJobClusters() throws IOException {
        List<IJobClusterMetadata> iJobClusterMetadataList = this.storageProvider.loadAllJobClusters();
        logger.info("Loaded {} job clusters", (Object)iJobClusterMetadataList.size());
        return iJobClusterMetadataList;
    }

    public List<IMantisJobMetadata> loadAllActiveJobs() throws IOException {
        List<IMantisJobMetadata> mantisJobMetadataList = this.storageProvider.loadAllJobs();
        logger.info("Loaded {} active jobs", (Object)mantisJobMetadataList.size());
        return mantisJobMetadataList;
    }

    public List<JobClusterDefinitionImpl.CompletedJob> loadCompletedJobsForCluster(String clusterName, int limit, @Nullable JobId startJobIdExclusive) throws IOException {
        return this.storageProvider.loadLatestCompletedJobsForCluster(clusterName, limit, startJobIdExclusive);
    }

    public void deleteCompletedJobsForCluster(String clusterName) throws IOException {
        this.storageProvider.deleteCompletedJobsForCluster(clusterName);
    }

    public void createJobCluster(IJobClusterMetadata jobCluster) throws Exception {
        this.storageProvider.createJobCluster(jobCluster);
    }

    public void updateJobCluster(IJobClusterMetadata jobCluster) throws Exception {
        this.storageProvider.updateJobCluster(jobCluster);
    }

    public void deleteJobCluster(String name) throws Exception {
        this.storageProvider.deleteJobCluster(name);
    }

    public void deleteJob(String jobId) throws IOException {
        this.archivedJobsMetadataCache.remove(jobId);
        this.archivedWorkersCache.remove(jobId);
        this.storageProvider.deleteJob(jobId);
    }

    public void storeCompletedJobForCluster(String name, JobClusterDefinitionImpl.CompletedJob completedJob) throws IOException {
        this.storageProvider.storeCompletedJobForCluster(name, completedJob);
    }

    public void storeNewJob(IMantisJobMetadata jobMetadata) throws Exception {
        this.storageProvider.storeNewJob(jobMetadata);
    }

    public TaskExecutorRegistration getTaskExecutor(TaskExecutorID taskExecutorID) throws IOException {
        return this.storageProvider.getTaskExecutorFor(taskExecutorID);
    }

    public void storeNewTaskExecutor(TaskExecutorRegistration registration) throws IOException {
        this.storageProvider.storeNewTaskExecutor(registration);
    }

    public void storeNewDisabledTaskExecutorsRequest(DisableTaskExecutorsRequest request) throws IOException {
        this.storageProvider.storeNewDisableTaskExecutorRequest(request);
    }

    public void deleteExpiredDisableTaskExecutorsRequest(DisableTaskExecutorsRequest request) throws IOException {
        this.storageProvider.deleteExpiredDisableTaskExecutorRequest(request);
    }

    public List<DisableTaskExecutorsRequest> loadAllDisableTaskExecutorsRequests(ClusterID clusterID) throws IOException {
        return this.storageProvider.loadAllDisableTaskExecutorsRequests(clusterID);
    }

    public void replaceTerminatedWorker(IMantisWorkerMetadata oldWorker, IMantisWorkerMetadata replacement) throws Exception {
        this.storageProvider.storeAndUpdateWorkers(oldWorker, replacement);
    }

    public void updateJob(IMantisJobMetadata jobMetadata) throws Exception {
        this.storageProvider.updateJob(jobMetadata);
    }

    public void updateStage(IMantisStageMetadata stageMeta) throws IOException {
        this.storageProvider.updateMantisStage(stageMeta);
    }

    public List<? extends IMantisWorkerMetadata> storeNewWorkers(IMantisJobMetadata job, List<IMantisWorkerMetadata> workerRequests) throws IOException, InvalidJobException {
        if (workerRequests == null || workerRequests.isEmpty()) {
            return null;
        }
        String jobId = workerRequests.get(0).getJobId();
        logger.debug("Adding {} workers for job {}", (Object)workerRequests.size(), (Object)jobId);
        ArrayList<IMantisWorkerMetadata> addedWorkers = new ArrayList<IMantisWorkerMetadata>();
        ArrayList savedStageList = Lists.newArrayList();
        for (IMantisWorkerMetadata workerRequest : workerRequests) {
            if (!savedStageList.contains(workerRequest.getStageNum())) {
                Optional<IMantisStageMetadata> stageMetadata = job.getStageMetadata(workerRequest.getStageNum());
                if (!stageMetadata.isPresent()) {
                    throw new RuntimeException(String.format("No such stage %d", workerRequest.getStageNum()));
                }
                this.storageProvider.storeMantisStage(stageMetadata.get());
                savedStageList.add(workerRequest.getStageNum());
            }
            addedWorkers.add(workerRequest);
        }
        this.storageProvider.storeWorkers(jobId, addedWorkers);
        return addedWorkers;
    }

    public void storeNewWorker(IMantisWorkerMetadata workerRequest) throws IOException, InvalidJobException {
        this.storageProvider.storeWorker(workerRequest);
    }

    public void updateWorker(IMantisWorkerMetadata worker) throws IOException {
        this.storageProvider.updateWorker(worker);
    }

    private void archiveWorkersIfAny(IMantisJobMetadata mjmd) throws IOException {
        for (IMantisStageMetadata iMantisStageMetadata : mjmd.getStageMetadata().values()) {
            for (JobWorker removedWorker : ((MantisStageMetadataImpl)iMantisStageMetadata).removeArchiveableWorkers()) {
                this.archiveWorker(removedWorker.getMetadata());
            }
        }
    }

    public void archiveWorker(IMantisWorkerMetadata worker) throws IOException {
        this.storageProvider.archiveWorker(worker);
        try {
            ConcurrentMap<Integer, IMantisWorkerMetadata> workersMap = this.archivedWorkersCache.getArchivedWorkerMap(worker.getJobId());
            workersMap.putIfAbsent(worker.getWorkerNumber(), worker);
        }
        catch (ExecutionException e) {
            logger.warn("Error adding worker to archived cache", (Throwable)e);
        }
    }

    public Optional<IMantisJobMetadata> getArchivedJob(String jobId) {
        Optional<IMantisJobMetadata> jobOp = Optional.ofNullable(this.archivedJobsMetadataCache.getJob(jobId));
        if (!jobOp.isPresent()) {
            logger.debug("archivedJobsMetadataCache found no job for job ID {}", (Object)jobId);
        }
        return jobOp;
    }

    public void archiveJob(IMantisJobMetadata job) throws IOException {
        this.archivedJobsMetadataCache.add(job);
        this.storageProvider.archiveJob(job.getJobId().getId());
    }

    public Optional<IMantisWorkerMetadata> getArchivedWorker(String jobId, int workerNumber) {
        try {
            ConcurrentMap<Integer, IMantisWorkerMetadata> workersMap = this.archivedWorkersCache.getArchivedWorkerMap(jobId);
            if (workersMap != null) {
                return Optional.ofNullable(workersMap.get(workerNumber));
            }
        }
        catch (ExecutionException e) {
            logger.warn("Exception getting archived worker", (Throwable)e);
        }
        return Optional.empty();
    }

    public List<IMantisWorkerMetadata> getArchivedWorkers(String jobId) throws Exception {
        return ImmutableList.copyOf(this.archivedWorkersCache.getArchivedWorkerMap(jobId).values());
    }

    public void addNewJobArtifactsToCache(ClusterID clusterID, List<ArtifactID> artifacts) throws IOException {
        this.storageProvider.addNewJobArtifactsToCache(clusterID, artifacts);
    }

    public void removeJobArtifactsToCache(ClusterID clusterID, List<ArtifactID> artifacts) throws IOException {
        this.storageProvider.removeJobArtifactsToCache(clusterID, artifacts);
    }

    public List<String> getJobArtifactsToCache(ClusterID clusterID) throws IOException {
        return this.storageProvider.listJobArtifactsToCache(clusterID);
    }

    public JobArtifact getJobArtifact(ArtifactID artifactID) throws IOException {
        return this.storageProvider.getArtifactById(artifactID.getResourceID());
    }

    private class ArchivedJobsMetadataCache {
        private final Cache<String, Optional<IMantisJobMetadata>> cache;

        ArchivedJobsMetadataCache(int cacheSize) {
            this.cache = CacheBuilder.newBuilder().maximumSize((long)cacheSize).build();
        }

        IMantisJobMetadata getJob(String jobId) {
            try {
                Optional jobMetadata = (Optional)this.cache.get((Object)jobId, () -> this.loadArchivedJobImpl(jobId));
                return jobMetadata.orElse(null);
            }
            catch (Exception e) {
                return null;
            }
        }

        private Optional<IMantisJobMetadata> loadArchivedJobImpl(String jobId) throws IOException, ExecutionException {
            Optional<IMantisJobMetadata> jobMetadata = MantisJobStore.this.storageProvider.loadArchivedJob(jobId);
            if (!jobMetadata.isPresent()) {
                logger.warn("Failed to load archived job {}. No job found!", (Object)jobId);
            }
            return jobMetadata;
        }

        void add(IMantisJobMetadata job) {
            this.cache.put((Object)job.getJobId().getId(), Optional.ofNullable(job));
        }

        void remove(String jobId) {
            this.cache.invalidate((Object)jobId);
        }
    }

    private class ArchivedWorkersCache {
        private final Cache<String, ConcurrentMap<Integer, IMantisWorkerMetadata>> cache;

        ArchivedWorkersCache(int cacheSize) {
            this.cache = CacheBuilder.newBuilder().maximumSize((long)cacheSize).build();
        }

        ConcurrentMap<Integer, IMantisWorkerMetadata> getArchivedWorkerMap(String jobId) throws ExecutionException {
            return (ConcurrentMap)this.cache.get((Object)jobId, () -> {
                List<IMantisWorkerMetadata> workers = MantisJobStore.this.storageProvider.getArchivedWorkers(jobId);
                ConcurrentHashMap<Integer, IMantisWorkerMetadata> theMap = new ConcurrentHashMap<Integer, IMantisWorkerMetadata>();
                if (workers != null) {
                    for (IMantisWorkerMetadata mwmd : workers) {
                        theMap.putIfAbsent(mwmd.getWorkerNumber(), mwmd);
                    }
                }
                return theMap;
            });
        }

        void remove(String jobId) {
            this.cache.invalidate((Object)jobId);
        }
    }

    private static class TerminatedJob
    implements Comparable<TerminatedJob> {
        private final String jobId;
        private final long terminatedTime;

        private TerminatedJob(String jobId, long terminatedTime) {
            this.jobId = jobId;
            this.terminatedTime = terminatedTime;
        }

        @Override
        public int compareTo(TerminatedJob o) {
            return Long.compare(this.terminatedTime, o.terminatedTime);
        }
    }
}

