/*
 * Copyright [ 2020 - 2023 ] [Matthew Buckton]
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.mapsmessaging.security.jaas;


import com.sun.security.auth.UserPrincipal;
import io.mapsmessaging.security.identity.principals.AuthHandlerPrincipal;

import javax.security.auth.Subject;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.auth.login.LoginException;
import java.io.IOException;
import java.util.Map;

public class SSLCertificateLoginModule extends BaseLoginModule {

  @Override
  public void initialize(Subject subject, CallbackHandler callbackHandler, Map<String, ?> sharedState, Map<String, ?> options) {
    super.initialize(subject, callbackHandler, sharedState, options);
    userPrincipal = null;
  }

  @Override
  protected String getDomain() {
    return "cert";
  }

  @Override
  public boolean login() throws LoginException {
    // prompt for a username and password
    if (callbackHandler == null) {
      throw new LoginException("Error: no CallbackHandler available to garner authentication information from the user");
    }

    Callback[] callbacks = new Callback[2];
    callbacks[0] = new NameCallback("user name: ");
    callbacks[1] = new PrincipalCallback();

    try {
      callbackHandler.handle(callbacks);
      username = ((NameCallback) callbacks[0]).getName();
      userPrincipal = ((PrincipalCallback) callbacks[1]).getPrincipal();
    } catch (IOException ioe) {
      throw new LoginException(ioe.toString());
    } catch (UnsupportedCallbackException uce) {
      throw new LoginException(
          "Error: "
              + uce.getCallback().toString()
              + " not available to garner authentication information "
              + "from the user");
    }
    succeeded = true;
    return true;
  }

  @Override
  public boolean commit() {
    if (super.commit()) {
      if (username != null) {
        subject.getPrincipals().add(new UserPrincipal(username));
        subject.getPrincipals().add(new AuthHandlerPrincipal("SSLCertificate"));
      }
      return true;
    }
    return false;
  }

  @Override
  protected boolean validate(String username, char[] password) {
    // Not called in this instance
    return true;
  }
}
