/*
 * Copyright 2017-2020 original authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.micronaut.annotation.processing;

import io.micronaut.annotation.processing.visitor.JavaClassElement;
import io.micronaut.annotation.processing.visitor.JavaElementFactory;
import io.micronaut.annotation.processing.visitor.JavaNativeElement;
import io.micronaut.context.annotation.Requires;
import io.micronaut.core.annotation.Generated;
import io.micronaut.core.annotation.NonNull;
import io.micronaut.core.annotation.Nullable;
import io.micronaut.core.io.service.SoftServiceLoader;
import io.micronaut.core.order.OrderUtil;
import io.micronaut.core.util.CollectionUtils;
import io.micronaut.core.util.StringUtils;
import io.micronaut.core.version.VersionUtils;
import io.micronaut.inject.ast.ConstructorElement;
import io.micronaut.inject.ast.ElementQuery;
import io.micronaut.inject.ast.EnumConstantElement;
import io.micronaut.inject.ast.FieldElement;
import io.micronaut.inject.ast.MemberElement;
import io.micronaut.inject.ast.MethodElement;
import io.micronaut.inject.processing.ProcessingException;
import io.micronaut.inject.visitor.TypeElementVisitor;
import io.micronaut.inject.visitor.VisitorContext;
import io.micronaut.inject.writer.AbstractBeanDefinitionBuilder;

import javax.annotation.processing.ProcessingEnvironment;
import javax.annotation.processing.RoundEnvironment;
import javax.annotation.processing.SupportedOptions;
import javax.lang.model.element.TypeElement;
import javax.lang.model.type.TypeMirror;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * <p>The annotation processed used to execute type element visitors.</p>
 *
 * @author James Kleeh
 * @author graemerocher
 * @since 1.0
 */
@SupportedOptions({
    AbstractInjectAnnotationProcessor.MICRONAUT_PROCESSING_INCREMENTAL,
    AbstractInjectAnnotationProcessor.MICRONAUT_PROCESSING_ANNOTATIONS,
    VisitorContext.MICRONAUT_PROCESSING_PROJECT_DIR,
    VisitorContext.MICRONAUT_PROCESSING_GROUP,
    VisitorContext.MICRONAUT_PROCESSING_MODULE
})
public class TypeElementVisitorProcessor extends AbstractInjectAnnotationProcessor {
    private static final SoftServiceLoader<TypeElementVisitor> SERVICE_LOADER = SoftServiceLoader.load(TypeElementVisitor.class, TypeElementVisitorProcessor.class.getClassLoader()).disableFork();
    private static final Set<String> VISITOR_WARNINGS;
    private static final Set<String> SUPPORTED_ANNOTATION_NAMES;

    static {

        final HashSet<String> warnings = new HashSet<>();
        Set<String> names = new HashSet<>();
        for (TypeElementVisitor<?, ?> typeElementVisitor : findCoreTypeElementVisitors(warnings)) {
            final Set<String> supportedAnnotationNames;
            try {
                supportedAnnotationNames = typeElementVisitor.getSupportedAnnotationNames();
            } catch (Throwable e) {
                // ignore if annotations are not on the classpath
                continue;
            }
            if (!supportedAnnotationNames.equals(Collections.singleton("*"))) {
                names.addAll(supportedAnnotationNames);
            }
        }
        SUPPORTED_ANNOTATION_NAMES = names;

        if (warnings.isEmpty()) {
            VISITOR_WARNINGS = Collections.emptySet();
        } else {
            VISITOR_WARNINGS = Collections.unmodifiableSet(warnings);
        }
    }

    private List<LoadedVisitor> loadedVisitors;
    private Collection<? extends TypeElementVisitor<?, ?>> typeElementVisitors;
    private final Set<String> pendingTypes = new LinkedHashSet<>();

    /**
     * The visited annotation names.
     *
     * @return The names of all the visited annotations.
     */
    static Set<String> getVisitedAnnotationNames() {
        return SUPPORTED_ANNOTATION_NAMES;
    }

    @Override
    public synchronized void init(ProcessingEnvironment processingEnv) {
        super.init(processingEnv);

        Collection<? extends TypeElementVisitor<?, ?>> typeElementVisitors = findTypeElementVisitors();

        // set supported options as system properties to keep compatibility
        // in particular for micronaut-openapi
        processingEnv.getOptions().entrySet().stream()
            .filter(entry -> entry.getKey() != null && entry.getKey().startsWith(VisitorContext.MICRONAUT_BASE_OPTION_NAME))
            .forEach(entry -> System.setProperty(entry.getKey(), entry.getValue() == null ? "" : entry.getValue()));

        this.loadedVisitors = new ArrayList<>(typeElementVisitors.size());

        for (TypeElementVisitor<?, ?> visitor : typeElementVisitors) {
            TypeElementVisitor.VisitorKind visitorKind = visitor.getVisitorKind();
            TypeElementVisitor.VisitorKind incrementalProcessorKind = getIncrementalProcessorKind();

            if (incrementalProcessorKind == visitorKind) {
                try {
                    loadedVisitors.add(new LoadedVisitor(visitor, processingEnv));
                } catch (TypeNotPresentException | NoClassDefFoundError e) {
                    // ignored, means annotations referenced are not on the classpath
                }
            }
        }

        OrderUtil.reverseSort(loadedVisitors);

        for (LoadedVisitor loadedVisitor : loadedVisitors) {
            try {
                loadedVisitor.getVisitor().start(javaVisitorContext);
            } catch (Throwable e) {
                error("Error initializing type visitor [%s]: %s", loadedVisitor.getVisitor(), e.getMessage());
            }
        }
    }

    /**
     * Does this process have any visitors.
     *
     * @return True if visitors are present.
     */
    protected boolean hasVisitors() {
        for (TypeElementVisitor<?, ?> typeElementVisitor : findTypeElementVisitors()) {
            if (typeElementVisitor.getVisitorKind() == getVisitorKind()) {
                return true;
            }
        }
        return false;
    }

    /**
     * @return The loaded visitors.
     */
    protected List<LoadedVisitor> getLoadedVisitors() {
        return loadedVisitors;
    }

    /**
     * @return The incremental processor type.
     * @see #GRADLE_PROCESSING_AGGREGATING
     * @see #GRADLE_PROCESSING_ISOLATING
     */
    protected TypeElementVisitor.VisitorKind getIncrementalProcessorKind() {
        String type = getIncrementalProcessorType();
        if (type.equals(GRADLE_PROCESSING_AGGREGATING)) {
            return TypeElementVisitor.VisitorKind.AGGREGATING;
        }
        return TypeElementVisitor.VisitorKind.ISOLATING;
    }

    @Override
    public Set<String> getSupportedAnnotationTypes() {
        if (loadedVisitors.isEmpty()) {
            return Collections.emptySet();
        } else {
            return super.getSupportedAnnotationTypes();
        }
    }

    @Override
    public Set<String> getSupportedOptions() {
        Stream<String> baseOption = super.getSupportedOptions().stream();
        Collection<? extends TypeElementVisitor<?, ?>> typeElementVisitors = findTypeElementVisitors();
        Stream<String> visitorsOptions = typeElementVisitors
            .stream()
            .map(TypeElementVisitor::getSupportedOptions)
            .flatMap(Collection::stream);
        Stream<String> visitorsAnnotationsOptions = typeElementVisitors
            .stream()
            .filter(tev -> tev.getClass().isAnnotationPresent(SupportedOptions.class))
            .map(TypeElementVisitor::getClass)
            .map(cls -> (SupportedOptions) cls.getAnnotation(SupportedOptions.class))
            .flatMap((SupportedOptions supportedOptions) -> Arrays.stream(supportedOptions.value()));
        return Stream.of(baseOption, visitorsAnnotationsOptions, visitorsOptions)
            .flatMap(Stream::sequential)
            .collect(Collectors.toSet());
    }

    @Override
    public boolean process(Set<? extends TypeElement> annotations, RoundEnvironment roundEnv) {
        if (!loadedVisitors.isEmpty() && !(annotations.size() == 1
            && Generated.class.getName().equals(annotations.iterator().next().getQualifiedName().toString()))) {

            TypeElement groovyObjectTypeElement = elementUtils.getTypeElement("groovy.lang.GroovyObject");
            TypeMirror groovyObjectType = groovyObjectTypeElement != null ? groovyObjectTypeElement.asType() : null;
            Predicate<TypeElement> notGroovyObject = typeElement -> groovyObjectType == null || !typeUtils.isAssignable(typeElement.asType(), groovyObjectType);

            Set<TypeElement> elements = new LinkedHashSet<>();

            for (TypeElement annotation : annotations) {
                modelUtils.resolveTypeElements(
                    roundEnv.getElementsAnnotatedWith(annotation)
                ).filter(notGroovyObject).forEach(elements::add);
            }

            // This call to getRootElements() should be removed in Micronaut 4. It should not be possible
            // to process elements without at least one annotation present and this call breaks that assumption.
            modelUtils.resolveTypeElements(
                roundEnv.getRootElements()
            ).filter(notGroovyObject).forEach(elements::add);

            pendingTypes.stream().map(elementUtils::getTypeElement).filter(Objects::nonNull).forEach(elements::add);
            pendingTypes.clear();

            if (!elements.isEmpty()) {

                JavaElementFactory elementFactory = javaVisitorContext.getElementFactory();
                JavaElementAnnotationMetadataFactory elementAnnotationMetadataFactory = javaVisitorContext.getElementAnnotationMetadataFactory();

                // The visitor X with a higher priority should process elements of A before
                // the visitor Y which is processing elements of B but also using elements A

                // Micronaut Data use-case: EntityMapper with a higher priority needs to process entities first
                // before RepositoryMapper is going to process repositories and read entities

                List<JavaClassElement> javaClassElements = elements.stream()
                    .map(typeElement -> elementFactory.newSourceClassElement(typeElement, elementAnnotationMetadataFactory))
                    .toList();

                for (LoadedVisitor loadedVisitor : loadedVisitors) {
                    for (JavaClassElement javaClassElement : javaClassElements) {
                        try {
                            if (loadedVisitor.matchesClass(javaClassElement)) {
                                visitClass(loadedVisitor, javaClassElement);
                            }
                        } catch (ProcessingException e) {
                            JavaNativeElement originatingElement = (JavaNativeElement) e.getOriginatingElement();
                            if (originatingElement == null) {
                                originatingElement = javaClassElement.getNativeType();
                            }
                            error(originatingElement.element(), e.getMessage());
                        } catch (PostponeToNextRoundException e) {
                            pendingTypes.add(javaClassElement.getName());
                        }
                    }
                }
            }

            for (LoadedVisitor loadedVisitor : loadedVisitors) {
                try {
                    loadedVisitor.getVisitor().finish(javaVisitorContext);
                } catch (Throwable e) {
                    error("Error finalizing type visitor [%s]: %s", loadedVisitor.getVisitor(), e.getMessage());
                }
            }
        }

        final List<AbstractBeanDefinitionBuilder> beanDefinitionBuilders = javaVisitorContext.getBeanElementBuilders();
        if (CollectionUtils.isNotEmpty(beanDefinitionBuilders)) {
            try {
                AbstractBeanDefinitionBuilder.writeBeanDefinitionBuilders(classWriterOutputVisitor, beanDefinitionBuilders);
            } catch (IOException e) {
                // raise a compile error
                String message = e.getMessage();
                error("Unexpected error: %s", message != null ? message : e.getClass().getSimpleName());
            }
        }

        if (roundEnv.processingOver()) {
            javaVisitorContext.finish();
            writeBeanDefinitionsToMetaInf();
        }
        return false;
    }

    private void visitClass(LoadedVisitor visitor, JavaClassElement classElement) {
        visitor.getVisitor().visitClass(classElement, javaVisitorContext);

        for (ConstructorElement constructorElement : classElement.getSourceEnclosedElements(ElementQuery.CONSTRUCTORS)) {
            visitConstructor(visitor, constructorElement);
        }
        for (MemberElement memberElement : classElement.getSourceEnclosedElements(ElementQuery.ALL_FIELD_AND_METHODS)) {
            if (memberElement instanceof EnumConstantElement enumConstantElement) {
                visitEnumConstant(visitor, enumConstantElement);
            } else if (memberElement instanceof FieldElement fieldElement) {
                visitField(visitor, fieldElement);
            } else if (memberElement instanceof MethodElement methodElement) {
                visitMethod(visitor, methodElement);
            } else {
                throw new IllegalStateException("Unknown element: " + memberElement);
            }
        }
    }

    private void visitConstructor(LoadedVisitor visitor, ConstructorElement constructorElement) {
        if (visitor.matchesElement(constructorElement)) {
            visitor.getVisitor().visitConstructor(constructorElement, javaVisitorContext);
        }
    }

    private void visitMethod(LoadedVisitor visitor, MethodElement methodElement) {
        if (visitor.matchesElement(methodElement)) {
            visitor.getVisitor().visitMethod(methodElement, javaVisitorContext);
        }
    }

    private void visitEnumConstant(LoadedVisitor visitor, EnumConstantElement enumConstantElement) {
        if (visitor.matchesElement(enumConstantElement)) {
            visitor.getVisitor().visitEnumConstant(enumConstantElement, javaVisitorContext);
        }
    }

    private void visitField(LoadedVisitor visitor, FieldElement fieldElement) {
        if (visitor.matchesElement(fieldElement)) {
            visitor.getVisitor().visitField(fieldElement, javaVisitorContext);
        }
    }

    /**
     * Discovers the {@link TypeElementVisitor} instances that are available.
     *
     * @return A collection of type element visitors.
     */
    @NonNull
    protected synchronized Collection<? extends TypeElementVisitor<?, ?>> findTypeElementVisitors() {
        if (typeElementVisitors == null) {
            for (String visitorWarning : VISITOR_WARNINGS) {
                warning(visitorWarning);
            }
            typeElementVisitors = findCoreTypeElementVisitors(null);
        }
        return typeElementVisitors;
    }

    /**
     * Writes {@link io.micronaut.inject.BeanDefinitionReference} into /META-INF/services/io.micronaut.inject.BeanDefinitionReference.
     */
    private void writeBeanDefinitionsToMetaInf() {
        try {
            classWriterOutputVisitor.finish();
        } catch (Exception e) {
            String message = e.getMessage();
            error("Error occurred writing META-INF files: %s", message != null ? message : e);
        }
    }

    @NonNull
    private static Collection<? extends TypeElementVisitor<?, ?>> findCoreTypeElementVisitors(
        @Nullable Set<String> warnings) {
        return SERVICE_LOADER.collectAll(visitor -> {
                if (!visitor.isEnabled()) {
                    return false;
                }

                final Requires requires = visitor.getClass().getAnnotation(Requires.class);
                if (requires != null) {
                    final Requires.Sdk sdk = requires.sdk();
                    if (sdk == Requires.Sdk.MICRONAUT) {
                        final String version = requires.version();
                        if (StringUtils.isNotEmpty(version) && !VersionUtils.isAtLeastMicronautVersion(version)) {
                            try {
                                if (warnings != null) {
                                    warnings.add("TypeElementVisitor [" + visitor.getClass().getName() + "] will be ignored because Micronaut version [" + VersionUtils.MICRONAUT_VERSION + "] must be at least " + version);
                                }
                                return false;
                            } catch (IllegalArgumentException e) {
                                // shouldn't happen, thrown when invalid version encountered
                            }
                        }
                    }
                }
                return true;
            }).stream()
            .filter(Objects::nonNull)
            .<TypeElementVisitor<?, ?>>map(e -> e)
            // remove duplicate classes
            .collect(Collectors.toMap(Object::getClass, v -> v, (a, b) -> a)).values();
    }
}
