/*
 * Copyright 2017-2020 original authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.micronaut.rabbitmq.annotation;

import io.micronaut.core.bind.annotation.Bindable;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Inherited;
import java.lang.annotation.Repeatable;
import java.lang.annotation.Retention;
import java.lang.annotation.Target;

import static java.lang.annotation.RetentionPolicy.RUNTIME;

/**
 * Used to set properties while publishing or bind to properties while consuming.
 *
 * For example while producing messages, the annotation can be set at the method
 * or class level to provide static data. It can be applied at the parameter level
 * to set the property differently per execution.
 *
 * While consuming the annotation can be applied to a parameter to bind the
 * property to the argument. In addition, it may also be used to supply any arguments
 * to the broker.
 *
 * @see com.rabbitmq.client.BasicProperties
 *
 * @author James Kleeh
 * @since 1.1.0
 */
@Documented
@Retention(RUNTIME)
@Target({ElementType.PARAMETER, ElementType.TYPE, ElementType.METHOD})
@Repeatable(value = RabbitProperties.class)
@Bindable
@Inherited
public @interface RabbitProperty {

    /**
     * If used as a bound parameter, this is the property name. If used on a class
     * level this is value and not the property name.
     *
     * @return The name of the property, otherwise it is inferred from the {@link #name()}
     */
    String value() default "";

    /**
     * Never used if applied to a parameter. Supplies the property name if used on
     * a class or method.
     *
     * @return The name of property
     */
    String name() default "";


    /**
     * Only to be used when defining consumer arguments to convert the value to the
     * required type by RabbitMQ.
     *
     * @return The type to convert the value to
     */
    Class type() default Void.class;
}
