package com.moneyhash.shared.datasource.network

import com.moneyhash.shared.errorhandling.ErrorMapper
import com.moneyhash.shared.errorhandling.ErrorInfo
import com.moneyhash.shared.localization.LocalizationManager
import com.moneyhash.shared.util.Constants
import com.moneyhash.shared.util.DefaultLogManager
import com.moneyhash.shared.util.extensions.JsonWithIgnoredUnknownKeys
import io.ktor.client.*
import io.ktor.client.call.body
import io.ktor.client.plugins.*
import io.ktor.client.statement.HttpResponse
import io.ktor.client.statement.bodyAsText
import kotlinx.serialization.json.JsonArray
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive
import kotlinx.serialization.json.jsonArray
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.jsonPrimitive

expect class KtorClientFactory() {
    fun build(enableLogs: Boolean, sdkVersion: String): HttpClient
}

fun DefaultRequest.DefaultRequestBuilder.handleSdkVersionHeaderHeader(sdkVersion: String) {
    DefaultLogManager.debug("Appending SDK version header: $sdkVersion")
    headers.append(
        Constants.SDK_VERSION_HEADER,
        sdkVersion
    )
    this.url.parameters.append("lang", LocalizationManager.locale.name)
}

suspend fun HttpCallValidator.Config.handleResponseValidation(response: HttpResponse) {
    val statusCode = response.status.value
    DefaultLogManager.info("Handling response validation for status code: $statusCode")

    when (statusCode) {
        in 300..399 -> {
            DefaultLogManager.error("Redirect response exception with status code: $statusCode")
            throw RedirectResponseException(response, response.bodyAsText())
        }

        in 400..499 -> {
            DefaultLogManager.error("Client request exception with status code: $statusCode")
            throw ClientRequestException(response, response.bodyAsText())
        }

        in 500..599 -> {
            DefaultLogManager.error("Server response exception with status code: $statusCode")
            throw ServerResponseException(response, response.bodyAsText())
        }
    }

    if (statusCode >= 600) {
        DefaultLogManager.error("Unexpected response exception with status code: $statusCode")
        throw ResponseException(response, response.bodyAsText())
    }

    DefaultLogManager.info("Response validation completed successfully for status code: $statusCode")
}

suspend fun Throwable.handleNetworkException() {
    DefaultLogManager.error("Handling network exception", this)

    val isResponseException = this is ResponseException
    val isClientRequestException = this is ClientRequestException
    val isServerResponseException = this is ServerResponseException
    val isRedirectResponseException = this is RedirectResponseException

    val response = when {
        isResponseException -> (this as? ResponseException)?.response
        isClientRequestException -> (this as? ClientRequestException)?.response
        isServerResponseException -> (this as? ServerResponseException)?.response
        isRedirectResponseException -> (this as? RedirectResponseException)?.response
        else -> null
    }

    if (response == null) {
        DefaultLogManager.error("No response available. Mapping generic throwable.")
        ErrorMapper.mapThrowable(this)
    } else {
        val bytes = response.body<ByteArray>()
        val string = bytes.decodeToString()
        DefaultLogManager.debug("Response body as string: $string")


        val errorResponse = JsonWithIgnoredUnknownKeys.decodeFromString(string) as JsonObject
        val errors = mutableListOf<ErrorInfo>()
        var errorMessage: String? = null
        if (errorResponse.keys.contains("status")) {
            val jsonError = errorResponse["status"]?.jsonObject
            val jsonErrors = jsonError?.get("errors")?.jsonArray
            errorMessage = jsonError?.get("message")?.jsonPrimitive?.content
            jsonErrors.orEmpty().forEach {
                DefaultLogManager.debug("Processing error: $it")
                if (it is JsonObject) {
                    it.entries.forEach { element ->
                        DefaultLogManager.debug("Mapping network error: key=${element.key}, message=${element.value.jsonPrimitive.content}")
                        errors.add(
                            ErrorInfo(
                                key = element.key,
                                message = element.value.jsonPrimitive.content
                            )
                        )
                    }
                }
            }
        } else {
            errorResponse.forEach {

                val value =
                    if (it.value is JsonPrimitive) {
                        it.value.jsonPrimitive.content
                    } else if (it.value is JsonArray) {
                        val value = it.value.jsonArray.firstOrNull()
                        if (value is JsonPrimitive) {
                            value.content
                        } else {
                            value.toString()
                        }
                    } else if (it.value is JsonPrimitive) {
                        it.value.jsonPrimitive.content
                    } else {
                        it.value.toString()
                    }

                DefaultLogManager.debug("Mapping network error: key=${it.key}, message=${value}")

                errors.add(
                    ErrorInfo(
                        key = it.key,
                        message = value
                    )
                )
            }
        }

        DefaultLogManager.debug("Mapped network errors: $errors")

        val error = ErrorMapper.mapServerError(
            errors = errors,
            errorMessage = errorMessage,
            throwable = this
        )
        DefaultLogManager.error("Mapped server error: $error")
        throw error
    }
}