package com.moneyhash.shared.datasource.network.model.payment

import com.moneyhash.shared.CommonParcelable
import com.moneyhash.shared.CommonParcelize
import com.moneyhash.shared.datasource.network.model.payment.billingfields.InputField
import com.moneyhash.shared.datasource.network.model.payment.billingfields.InputFieldType
import com.moneyhash.shared.datasource.network.model.payment.methods.IntentMethods
import com.moneyhash.shared.util.extensions.JsonWithIgnoredUnknownKeys
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import kotlinx.serialization.modules.SerializersModule

/**
 * A class representing various states of an intent.
 *
 * This class encapsulates different possible states that an intent can be in, such as method selection,
 * transaction processing, or dealing with user actions.
 */
@CommonParcelize
@Serializable
sealed class IntentStateDetails: CommonParcelable {
    /**
     * Represents the state where the user can select from available payment methods.
     *
     * @property methods The methods available for selection during intent processing.
     */
    @Serializable
    @SerialName("method_selection")
    data class MethodSelection(
        @SerialName("methods")
        val methods: IntentMethods
    ) : IntentStateDetails()

    /**
     * Represents the state where the intent form is need to be displayed for user input.
     */
    @Serializable
    @SerialName("intent_form")
    data object IntentForm : IntentStateDetails()

    /**
     * Represents the state where the intent has been successfully processed.
     */
    @Serializable
    @SerialName("intent_processed")
    data object IntentProcessed : IntentStateDetails()

    /**
     * Represents the state where the transaction is waiting for user action.
     */
    @Serializable
    @SerialName("transaction_waiting_user_action")
    data object TransactionWaitingUserAction : IntentStateDetails()

    /**
     * Represents the state where the transaction has failed.
     *
     * @property recommendedMethods A list of recommended methods for the user to attempt another transaction if available.
     */
    @Serializable
    @SerialName("transaction_failed")
    data class TransactionFailed(val recommendedMethods: IntentMethods?) : IntentStateDetails()

    /**
     * Represents a state indicating that the intent has expired.
     */
    @Serializable
    @SerialName("expired")
    data object Expired : IntentStateDetails()

    /**
     * Represents a state indicating that the intent has been closed.
     */
    @Serializable
    @SerialName("closed")
    data object Closed : IntentStateDetails()

    /**
     * Represents the state where form fields are presented for user input.
     *
     * @property tokenizeCardInfo The information required to render/tokenize card data, if applicable.
     * @property billingFields A list of input fields required for billing information.
     * @property shippingFields A list of input fields required for shipping information.
     */
    @Serializable
    @SerialName("form_fields")
    data class FormFields(
        @SerialName("tokenizeCardInfo")
        val tokenizeCardInfo: TokenizeCardInfo? = null,
        @SerialName("billingFields")
        val billingFields: List<InputField>? = null,
        @SerialName("shippingFields")
        val shippingFields: List<InputField>? = null
    ) : IntentStateDetails()

    /**
     * Represents the state where a URL needs to be rendered, potentially for web interaction.
     *
     * @property url The URL to be rendered.
     * @property renderStrategy The strategy for rendering the URL (e.g., IFRAME or REDIRECT).
     */
    @Serializable
    @SerialName("url_to_render")
    data class UrlToRender(
        @SerialName("url")
        val url: String?,
        @SerialName("renderStrategy")
        val renderStrategy: RenderStrategy?
    ) : IntentStateDetails()

    /**
     * Represents the state where a saved card's CVV is required for processing.
     *
     * @property cvvField The input field used to capture the CVV.
     * @property cardTokenData The tokenized data for the card being processed.
     */
    @Serializable
    @SerialName("saved_card_cvv")
    data class SavedCardCVV(
        @SerialName("cvvField")
        val cvvField: InputField,
        @SerialName("cardTokenData")
        val cardTokenData: CardTokenData?
    ) : IntentStateDetails() {
        companion object {
            fun getDefaultCVVField(cardTokenDate: CardTokenData): SavedCardCVV {
                return SavedCardCVV(
                    InputField(
                        InputFieldType.Text,
                        "cvv",
                        null,
                        "CVV",
                        4,
                        true,
                        null,
                        null,
                        null,
                        3,
                        false,
                        null
                    ),
                    cardTokenDate
                )
            }
        }
    }

    /**
     * Represents the state where native payment options (e.g., Apple Pay, Google Pay) are available.
     *
     * @property nativePayData The data required to process a payment using a native payment method.
     */
    @Serializable
    @SerialName("native_pay")
    data class NativePay(
        @SerialName("nativePayData")
        val nativePayData: NativePayData?
    ) : IntentStateDetails()
}