package com.moneyhash.shared.datasource.network.model.payout

import com.moneyhash.shared.CommonParcelable
import com.moneyhash.shared.CommonParcelize
import com.moneyhash.shared.CommonRawValue
import com.moneyhash.shared.datasource.network.model.BrandSettingsData
import com.moneyhash.shared.datasource.network.model.Redirect
import com.moneyhash.shared.datasource.network.model.common.IntentStatus
import com.moneyhash.shared.datasource.network.model.common.IntentState
import com.moneyhash.shared.datasource.network.model.common.TransactionOperation
import com.moneyhash.shared.datasource.network.model.payment.FormFields
import com.moneyhash.shared.datasource.network.model.payment.IntentStateDetails
import com.moneyhash.shared.datasource.network.model.common.ProviderExtraData
import com.moneyhash.shared.datasource.network.model.payment.CardTokenData
import com.moneyhash.shared.datasource.network.model.payment.IntentMethodItem
import com.moneyhash.shared.datasource.network.model.payment.RenderStrategy
import com.moneyhash.shared.datasource.network.model.payment.Status
import com.moneyhash.shared.datasource.network.model.payment.toInputField
import com.moneyhash.shared.domain.model.PaymentActions
import com.moneyhash.shared.util.Constants
import com.moneyhash.shared.util.extensions.toChoicesList
import com.moneyhash.shared.util.extensions.toChoicesMap
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import com.moneyhash.shared.datasource.network.model.payment.billingfields.InputField
import com.moneyhash.shared.datasource.network.model.payment.billingfields.InputFieldType
import com.moneyhash.shared.datasource.network.model.payment.methods.toPayoutMethods
import com.moneyhash.shared.util.MoneyHashUtils

@Serializable
@CommonParcelize
data class PayoutDetails(
    @SerialName("data")
    val data: PayoutData? = null,
    @SerialName("status")
    val status: Status? = null
): CommonParcelable

@Serializable
@CommonParcelize
data class PayoutData(
    @SerialName("selected_method")
    val selectedMethod: String? = null,
    @SerialName("intent_sdk_state")
    val intentSdkState: IntentState? = null,
    @SerialName("brand_settings")
    val brandSettings: BrandSettingsData? = null,
    @SerialName("next_action")
    val nextAction: String? = null,
    @SerialName("intent")
    val intent: PayoutIntentData?,
    @SerialName("action_data")
    val actionData: PayoutActionData? = null
): CommonParcelable

@Serializable
@CommonParcelize
data class PayoutActionData(
    @SerialName("transaction")
    val transaction: PayoutTransactionData? = null,
    @SerialName("redirect")
    val redirect: Redirect? = null,
    @SerialName("form_fields")
    val formFields: FormFields? = null,
    @SerialName("provider_extra_data")
    val providerExtraData: ProviderExtraData? = null,
    @SerialName("card_token_data")
    val cardTokenData: CardTokenData? = null
): CommonParcelable

@Serializable
@CommonParcelize
data class PayoutTransactionData(
    @SerialName("billing_data")
    val billingData: @CommonRawValue JsonObject? = null,
    @SerialName("amount")
    val amount: Double? = null,
    @SerialName("custom_message")
    val customMessage: String? = null,
    @SerialName("external_action_message")
    val externalActionMessage: List<@CommonRawValue JsonElement>? = null,
    @SerialName("created")
    val createdDate: String? = null,
    @SerialName("custom_fields")
    val customFields: @CommonRawValue JsonObject? = null,
    @SerialName("amount_currency")
    val amountCurrency: String? = null,
    @SerialName("max_payout_amount")
    val maxPayoutAmount: Double? = null,
    @SerialName("payout_method_name")
    val payoutMethodName: String? = null,
    @SerialName("pay_out_method")
    val payoutMethod: String? = null,
    @SerialName("id")
    val id: String? = null,
    @SerialName("trx_status")
    val status: String? = null,
    @SerialName("operations")
    val operations: List<TransactionOperation>? = null,
    @SerialName("provider_transaction_fields")
    val providerTransactionFields: @CommonRawValue JsonObject? = null,
    @SerialName("custom_form_answers")
    val customFormAnswers: @CommonRawValue JsonObject? = null
): CommonParcelable {
    val billingDataString: String?
        get() = MoneyHashUtils.convertToString(billingData)

    val customFieldsString: String?
        get() = MoneyHashUtils.convertToString(customFields)

    val providerTransactionFieldsString: String?
        get() = MoneyHashUtils.convertToString(providerTransactionFields)

    val customFormAnswersString: String?
        get() = MoneyHashUtils.convertToString(customFormAnswers)

    val externalActionMessageString: List<String>?
        get() = MoneyHashUtils.convertToStringList(externalActionMessage)
}

@Serializable
@CommonParcelize
data class PayoutIntentData(
    @SerialName("amount")
    val amount: Double? = null,
    @SerialName("amount_currency")
    val amountCurrency: String? = null,
    @SerialName("is_live")
    val isLive: Boolean? = false,
    @SerialName("max_payout_amount")
    val maxPayoutAmount: Double? = null,
    @SerialName("id")
    val id: String? = null,
    @SerialName("secret")
    val secret: String? = null,
    @SerialName("form_only")
    val formOnly: Boolean? = false,
    @SerialName("payout_methods")
    val payoutMethods: List<IntentMethodItem>? = null,
    @SerialName("status")
    val status: IntentStatus? = null,
    @SerialName("hide_amount_sidebar")
    val hideAmountSidebar: Boolean? = null,
    @SerialName("hide_header")
    val hideHeader: Boolean? = null,
): CommonParcelable


fun PayoutData.getIntentStateDetails(): IntentStateDetails? {
    when(this.intentSdkState) {
        IntentState.CLOSED -> {
            return IntentStateDetails.Closed
        }

        IntentState.METHOD_SELECTION -> {
            return  IntentStateDetails.MethodSelection(this.toPayoutMethods())
        }

        IntentState.INTENT_FORM -> {
            return  IntentStateDetails.IntentForm
        }
        IntentState.INTENT_PROCESSED -> {
            return  IntentStateDetails.IntentProcessed
        }
        IntentState.TRANSACTION_WAITING_USER_ACTION -> {
            return  IntentStateDetails.TransactionWaitingUserAction
        }
        IntentState.TRANSACTION_FAILED -> {
            return IntentStateDetails.TransactionFailed(null)
        }
        IntentState.EXPIRED -> {
            return  IntentStateDetails.Expired
        }
        IntentState.FORM_FIELDS -> {
            return  IntentStateDetails.FormFields(
                tokenizeCardInfo = this.actionData?.formFields?.tokenizeCardInfo,
                billingFields = this.actionData?.getBillingFields(),
                shippingFields = this.actionData?.getShippingFields()
            )
        }
        IntentState.URL_TO_RENDER -> {
            return IntentStateDetails.UrlToRender(url = this.actionData?.redirect?.redirectUrl, renderStrategy = this.getRedirectStrategy())
        }
        IntentState.SAVED_CARD_CVV -> {
            this.actionData?.cardTokenData ?: return null
            return IntentStateDetails.SavedCardCVV.getDefaultCVVField(this.actionData.cardTokenData)
        }
        else -> return null
    }
}

fun PayoutData.getRedirectStrategy(): RenderStrategy {
    return when (nextAction) {
        PaymentActions.REDIRECT -> {
            return RenderStrategy.REDIRECT
        }
        PaymentActions.POPUP_IFRAME -> {
            return RenderStrategy.POPUP_IFRAME
        }
        PaymentActions.IFRAME -> {
            return  RenderStrategy.IFRAME
        }

        else -> {
            RenderStrategy.NONE
        }
    }
}

fun PayoutActionData.getBillingFields(): List<InputField> {
    val fields = mutableListOf<InputField>()

    this.formFields?.billingFields?.forEach { fieldItem ->
        val fieldType = InputFieldType.from(fieldItem.type.orEmpty())

        when (fieldType) {
            InputFieldType.Select -> {
                val optionsList = this.providerExtraData?.stateCity?.toChoicesList()
                val optionsMap = this.providerExtraData?.stateCity?.toChoicesMap()

                if (fieldItem.fieldName == Constants.STATE_API_NAME_FILED) {
                    fields.add(fieldItem.toInputField(optionsList = optionsList))
                } else {
                    fields.add(
                        fieldItem.toInputField(
                            optionsMap = optionsMap,
                            dependsOn = Constants.STATE_API_NAME_FILED
                        )
                    )
                }
            }
            else -> {
                fields.add(fieldItem.toInputField())
            }
        }
    }

    return fields
}

fun PayoutActionData.getShippingFields(): List<InputField> {
    val fields = mutableListOf<InputField>()

    this.formFields?.shippingFields?.forEach { fieldItem ->
        fields.add(fieldItem.toInputField())
    }

    return fields
}