package com.moneyhash.shared.datasource.network.requestbuilder

import io.ktor.http.*

class RequestBuilder {
    private lateinit var requestEndpoint: String
    val contents: MutableList<RequestContent> = mutableListOf()

    fun endpoint(endpoint: String) {
        this.requestEndpoint = endpoint
    }

    fun query(key: String, value: String) = apply {
        contents.add(RequestContent.Query(key, value))
    }

    fun path(key: String, value: String) = apply {
        contents.add(RequestContent.Path(key, value))
    }

    inline fun <reified T> body(value: T) = apply {
        if (this.contents.any { it is RequestContent.Body<*> }) {
            throw IllegalStateException("You can pass only one body data inside the request")
        }

        header(HttpHeaders.ContentType, "application/json")
        this.contents.add(RequestContent.Body(value))
    }

    fun header(key: String, value: String) = apply {
        contents.add(RequestContent.Header(key, value))
    }

    fun build(): Pair<String, List<RequestContent>> {
        val pathContents = contents.filterIsInstance<RequestContent.Path>()
        var updatedUrl = requestEndpoint

        pathContents.forEach {
            updatedUrl = updatedUrl.replace("{${it.key}}", it.value)
        }

        return updatedUrl to contents.filter { it !is RequestContent.Path }
    }
}

sealed class RequestContent {
    class Query(val key: String, val value: String) : RequestContent()
    class Path(val key: String, val value: String) : RequestContent()
    class Header(val key: String, val value: String) : RequestContent()
    class Body<T>(val value: T) : RequestContent()
}