package com.moneyhash.shared.datasource.network.services

import com.moneyhash.shared.datasource.network.base.BaseService
import com.moneyhash.shared.datasource.network.model.StatusResponse
import com.moneyhash.shared.datasource.network.model.card.CardDetails
import com.moneyhash.shared.datasource.network.model.card.DeleteSavedCardRequest
import com.moneyhash.shared.datasource.network.model.discount.DiscountItem
import com.moneyhash.shared.datasource.network.model.discount.DiscountResponse
import com.moneyhash.shared.datasource.network.model.fees.FeeItem
import com.moneyhash.shared.datasource.network.model.fees.FeesResponse
import com.moneyhash.shared.datasource.network.model.payment.*
import com.moneyhash.shared.datasource.network.model.payout.PayoutDetails
import com.moneyhash.shared.datasource.network.model.vault.VaultData
import com.moneyhash.shared.datasource.services.PaymentService
import com.moneyhash.shared.util.Constants
import com.moneyhash.shared.util.DefaultLogManager
import com.moneyhash.shared.util.Type
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.JsonPrimitive
import kotlinx.serialization.json.jsonObject

internal class PaymentServiceImpl(
    private val paymentService: BaseService,
    ) : PaymentService {

    @Throws(Throwable::class)
    override suspend fun getPaymentInformation(paymentIntentId: String): PaymentIntentData {
        DefaultLogManager.info("Start fetching payment information for intent ID: $paymentIntentId")
        DefaultLogManager.debug("Sending GET request to end point: ${Type.PAYMENT.resultSuffix}$paymentIntentId")
        val body: PaymentIntentData = try {
            paymentService.get<PaymentIntentData>() {
                endpoint("${Type.PAYMENT.resultSuffix}$paymentIntentId")
            }
        } catch (e: Throwable) {
            DefaultLogManager.error("Failed to fetch payment information for intent ID: $paymentIntentId", e)
            throw e
        }
        DefaultLogManager.verbose("Payment information response: $body")
        DefaultLogManager.info("Successfully fetched payment information for intent ID: $paymentIntentId")
        return body
    }

    @Throws(Throwable::class)
    override suspend fun getCardInformation(cardIntentId: String): CardDetails {
        DefaultLogManager.info("Start fetching card information for card intent ID: $cardIntentId")
        DefaultLogManager.debug("Sending GET request to end point: ${Type.CARDS.resultSuffix}$cardIntentId")
        val body: CardDetails = try {
            paymentService.get<CardDetails>() {
                endpoint("${Type.CARDS.resultSuffix}$cardIntentId")
            }

        } catch (e: Throwable) {
            DefaultLogManager.error("Failed to fetch card information for card intent ID: $cardIntentId", e)
            throw e
        }
        DefaultLogManager.verbose("Card details response: $body")
        DefaultLogManager.info("Successfully fetched card information for card intent ID: $cardIntentId")
        return body
    }


    /**
     * Updates the discount for a specific payment intent.
     *
     * Sends a POST request to the server with the discount details to be applied to a payment intent.
     * The server is expected to return the details of the discount applied.
     *
     * @param intentId The ID of the payment intent to update the discount for.
     * @param amount The total amount before applying the discount.
     * @param discount The discount details to be applied.
     * @return DiscountResponse The response from the server including the discount details.
     * @throws Throwable If there is any issue with the network request or server response.
     */
    @Throws(Throwable::class)
    override suspend fun updateDiscount(intentId: String, discount: DiscountItem): DiscountResponse {
        DefaultLogManager.info("Start updating discount for payment intent ID: $intentId")
        DefaultLogManager.debug("Sending POST request to endpoint: /api/v1/client/payments/intents/$intentId/discount/update/")
        DefaultLogManager.debug("Discount: $discount")

        val requestBody = mapOf("discount" to discount)
        DefaultLogManager.debug("Request Body: $requestBody")

        return try {
            paymentService.post<DiscountResponse> {
                endpoint("client/payments/intents/$intentId/discount/update/")
                body(requestBody)
            }.also {
                DefaultLogManager.verbose("Response after updating discount: $it")
                DefaultLogManager.info("Successfully updated discount for payment intent ID: $intentId")
            }
        } catch (e: Throwable) {
            DefaultLogManager.error("Failed to update discount for payment intent ID: $intentId", e)
            throw e
        }
    }


    @Throws(Throwable::class)
    override suspend fun getPayoutInformation(payoutIntentId: String): PayoutDetails {
        DefaultLogManager.info("Start fetching payout information for payout intent ID: $payoutIntentId")
        DefaultLogManager.debug("Sending GET request to end point: ${Type.PAYOUT.resultSuffix}$payoutIntentId")
        val body: PayoutDetails = try {
            paymentService.get<PayoutDetails>() {
                endpoint("${Type.PAYOUT.resultSuffix}$payoutIntentId")
            }
        } catch (e: Throwable) {
            DefaultLogManager.error("Failed to fetch payout information for payout intent ID: $payoutIntentId", e)
            throw e
        }
        DefaultLogManager.verbose("Payout details response: $body")
        DefaultLogManager.info("Successfully fetched payout information for payout intent ID: $payoutIntentId")
        return body
    }

    /**
     * Updates the fees for a specific payment intent.
     *
     * This method sends a POST request to the server with a list of fees to be applied to a payment intent.
     * The server is expected to return the total updated amount along with the details of the fees applied.
     *
     * @param intentId The ID of the payment intent to update fees for.
     * @param fees A list of FeeItem representing the fees to be updated.
     * @return FeesResponse The response from the server, including the total amount and detailed fees.
     * @throws Throwable If there is any issue with the network request or server response.
     */
    @Throws(Throwable::class)
    override suspend fun updateFees(intentId: String, fees: List<FeeItem>): FeesResponse {
        DefaultLogManager.info("Start updating fees for payment intent ID: $intentId")
        DefaultLogManager.debug("Sending POST request to endpoint: /api/v1/client/payments/intents/$intentId/fees/update/")
        DefaultLogManager.debug("Fees: $fees")

        val requestBody = mapOf("fees" to fees)
        DefaultLogManager.debug("Request Body: $requestBody")

        return try {
            paymentService.post<FeesResponse> {
                endpoint("client/payments/intents/$intentId/fees/update/")
                body(requestBody)
            }.also {
                DefaultLogManager.verbose("Response after updating fees: $it")
                DefaultLogManager.info("Successfully updated fees for payment intent ID: $intentId")
            }
        } catch (e: Throwable) {
            DefaultLogManager.error("Failed to update fees for payment intent ID: $intentId", e)
            throw e
        }
    }

    @Throws(Throwable::class)
    override suspend fun useSavedCard(paymentIntentId: String, cardTokenId: String, cvv: String?): PaymentIntentData {
        DefaultLogManager.info("Start using saved card for payment intent ID: $paymentIntentId with card token ID: $cardTokenId")
        DefaultLogManager.debug("Sending POST request to end point: ${Type.PAYMENT.resultSuffix}${paymentIntentId}/card_token/")
        DefaultLogManager.debug("CVV: $cvv")

        val requestBody = UseCardTokenRequest(cardTokenId, cvv)
        DefaultLogManager.debug("Request Body: $requestBody")

        val body: PaymentIntentData = try {
            paymentService.post<PaymentIntentData> {
                endpoint("${Type.PAYMENT.resultSuffix}$paymentIntentId/card_token/")
                body(requestBody)
            }
        } catch (e: Throwable) {
            DefaultLogManager.error("Failed to use saved card for payment intent ID: $paymentIntentId", e)
            throw e
        }
        DefaultLogManager.verbose("Response after using saved card: $body")
        DefaultLogManager.info("Successfully used saved card for payment intent ID: $paymentIntentId")
        return body
    }

    @Throws(Throwable::class)
    override suspend fun useSelfWallet(paymentIntentId: String): PaymentIntentData {
        DefaultLogManager.info("Start using self wallet for payment intent ID: $paymentIntentId")
        DefaultLogManager.debug("Sending POST request to end point: ${Type.PAYMENT.resultSuffix}${paymentIntentId}/${Constants.SELFSERVE_WALLET}/")

        val requestBody = UseSelfWalletRequest(true)
        DefaultLogManager.debug("Request Body: $requestBody")

        val body: PaymentIntentData = try {
            paymentService.post<PaymentIntentData>() {
                endpoint("${Type.PAYMENT.resultSuffix}${paymentIntentId}/${Constants.SELFSERVE_WALLET}/")
                body(requestBody)
            }
        } catch (e: Throwable) {
            DefaultLogManager.error("Failed to use self wallet for payment intent ID: $paymentIntentId", e)
            throw e
        }
        DefaultLogManager.verbose("Response after using self wallet: $body")
        DefaultLogManager.info("Successfully used self wallet for payment intent ID: $paymentIntentId")
        return body
    }

    @Throws(Throwable::class)
    override suspend fun usePaymentMethod(intentId: String, methodName: String): PaymentIntentData {
        DefaultLogManager.info("Start using payment method '$methodName' for intent ID: $intentId")
        DefaultLogManager.debug("Sending POST request to end point: ${Type.PAYMENT.resultSuffix}${intentId}/update-method/")

        val requestBody = UpdateMethodRequest(paymentMethod = methodName)
        DefaultLogManager.debug("Request Body: $requestBody")

        val body: PaymentIntentData = try {
            paymentService.post<PaymentIntentData>() {
                endpoint("${Type.PAYMENT.resultSuffix}${intentId}/update-method/")
                body(requestBody)
            }
        } catch (e: Throwable) {
            DefaultLogManager.error("Failed to use payment method '$methodName' for intent ID: $intentId", e)
            throw e
        }
        DefaultLogManager.verbose("Response after using payment method '$methodName': $body")
        DefaultLogManager.info("Successfully used payment method '$methodName' for intent ID: $intentId")
        return body
    }

    @Throws(Throwable::class)
    override suspend fun usePayoutMethod(intentId: String, methodName: String): PayoutDetails {
        DefaultLogManager.info("Start using payout method '$methodName' for intent ID: $intentId")
        DefaultLogManager.debug("Sending POST request to end point: ${Type.PAYOUT.resultSuffix}${intentId}/update-method/")

        val requestBody = UpdateMethodRequest(payoutMethod = methodName)
        DefaultLogManager.debug("Request Body: $requestBody")

        val body: PayoutDetails = try {
            paymentService.post<PayoutDetails>() {
                endpoint("${Type.PAYOUT.resultSuffix}${intentId}/update-method/")
                body(requestBody)
            }
        } catch (e: Throwable) {
            DefaultLogManager.error("Failed to use payout method '$methodName' for intent ID: $intentId", e)
            throw e
        }
        DefaultLogManager.verbose("Response after using payout method '$methodName': $body")
        DefaultLogManager.info("Successfully used payout method '$methodName' for intent ID: $intentId")
        return body
    }

    @Throws(Throwable::class)
    override suspend fun resetPaymentSelectedMethod(paymentIntentId: String): PaymentIntentData {
        DefaultLogManager.info("Start resetting payment method for payment intent ID: $paymentIntentId")
        DefaultLogManager.debug("Sending POST request to end point: ${Type.PAYMENT.resultSuffix}${paymentIntentId}/update-method/")

        val requestBody = UpdateMethodRequest(null, null, true)
        DefaultLogManager.debug("Request Body: $requestBody")

        val body: PaymentIntentData = try {
            paymentService.post<PaymentIntentData>() {
                endpoint("${Type.PAYMENT.resultSuffix}${paymentIntentId}/update-method/")
                body(requestBody)
            }
        } catch (e: Throwable) {
            DefaultLogManager.error("Failed to reset payment method for payment intent ID: $paymentIntentId", e)
            throw e
        }
        DefaultLogManager.verbose("Response after resetting payment method: $body")
        DefaultLogManager.info("Successfully reset payment method for payment intent ID: $paymentIntentId")
        return body
    }

    @Throws(Throwable::class)
    override suspend fun resetPayoutSelectedMethod(payoutIntentId: String): PayoutDetails {
        DefaultLogManager.info("Start resetting payout method for payout intent ID: $payoutIntentId")
        DefaultLogManager.debug("Sending POST request to end point: ${Type.PAYOUT.resultSuffix}${payoutIntentId}/update-method/")

        val requestBody = UpdateMethodRequest(null, null, true)
        DefaultLogManager.debug("Request Body: $requestBody")

        val body: PayoutDetails = try {
            paymentService.post<PayoutDetails>() {
                endpoint("${Type.PAYOUT.resultSuffix}${payoutIntentId}/update-method/")
                body(requestBody)
            }
        } catch (e: Throwable) {
            DefaultLogManager.error("Failed to reset payout method for payout intent ID: $payoutIntentId", e)
            throw e
        }
        DefaultLogManager.verbose("Response after resetting payout method: $body")
        DefaultLogManager.info("Successfully reset payout method for payout intent ID: $payoutIntentId")
        return body
    }

    @Throws(Throwable::class)
    override suspend fun deleteSavedCard(cardTokenId: String, secret: String): StatusResponse {
        DefaultLogManager.info("Start deleting saved card with card token ID: $cardTokenId")
        DefaultLogManager.debug("Sending DELETE request to end point: client/card_tokens/${cardTokenId}/")
        DefaultLogManager.debug("Card Token ID: $cardTokenId")

        val requestBody = DeleteSavedCardRequest(secret)
        DefaultLogManager.debug("Request Body: $requestBody")

        val body: StatusResponse = try {
            paymentService.delete<StatusResponse>() {
                endpoint("client/card_tokens/${cardTokenId}/")
                body(requestBody)
            }
        } catch (e: Throwable) {
            DefaultLogManager.error("Failed to delete saved card with card token ID: $cardTokenId", e)
            throw e
        }
        DefaultLogManager.verbose("Response after deleting saved card: $body")
        DefaultLogManager.info("Successfully deleted saved card with card token ID: $cardTokenId")
        return body
    }

    @Throws(Throwable::class)
    override suspend fun submitReceipt(paymentIntentId: String, receipt: String): PaymentInformation {
        DefaultLogManager.info("Start submitting receipt for payment intent ID: $paymentIntentId")
        DefaultLogManager.debug("Sending POST request to end point: ${Type.PAYMENT.resultSuffix}${paymentIntentId}/receipt/")
        DefaultLogManager.debug("Receipt: $receipt")

        val requestBody = SubmitReceiptRequest(receipt)
        DefaultLogManager.debug("Request Body: $requestBody")

        val body: PaymentInformation = try {
            paymentService.post<PaymentInformation>() {
                endpoint("${Type.PAYMENT.resultSuffix}${paymentIntentId}/receipt/")
                body(requestBody)
            }
        } catch (e: Throwable) {
            DefaultLogManager.error("Failed to submit receipt for payment intent ID: $paymentIntentId", e)
            throw e
        }
        DefaultLogManager.verbose("Response after submitting receipt: $body")
        DefaultLogManager.info("Successfully submitted receipt for payment intent ID: $paymentIntentId")
        return body
    }

    @Throws(Throwable::class)
    override suspend fun submitForm(
        intentId: String,
        currentSelectedMethod: String,
        cardFields: JsonObject?,
        billingFields: JsonObject?,
        shippingFields: JsonObject?,
        vaultData: VaultData?
    ): PaymentIntentData {
        DefaultLogManager.info("Start submitting form for intent ID: $intentId with method: $currentSelectedMethod")
        DefaultLogManager.debug("Card Fields: $cardFields")
        DefaultLogManager.debug("Billing Fields: $billingFields")
        DefaultLogManager.debug("Shipping Fields: $shippingFields")
        DefaultLogManager.debug("Card Embed: $vaultData")
        val nativeForm = JsonObject(mutableMapOf<String, JsonObject>().apply {
            this["card_fields"] = cardFields ?: JsonObject(emptyMap())
            this["billing_fields"] = billingFields ?: JsonObject(emptyMap())
            this["shipping_fields"] = shippingFields ?: JsonObject(emptyMap())
            this["card_embed"] = vaultData?.let { Json.parseToJsonElement(Json.encodeToString(it)).jsonObject } ?: JsonObject(emptyMap())
        })
        DefaultLogManager.debug("Native Form: $nativeForm")

        DefaultLogManager.debug("Sending POST request to end point: ${Type.PAYMENT.resultSuffix}${intentId}/${currentSelectedMethod}/")
        val body: PaymentIntentData = try {
            paymentService.post<PaymentIntentData>() {
                endpoint("${Type.PAYMENT.resultSuffix}${intentId}/${currentSelectedMethod}/")
                body(SubmitFormRequest(nativeForm))
            }
        } catch (e: Throwable) {
            DefaultLogManager.error("Failed to submit form for intent ID: $intentId", e)
            throw e
        }
        DefaultLogManager.verbose("Response after submitting form: $body")
        DefaultLogManager.info("Successfully submitted form for intent ID: $intentId")
        return body
    }

    @Throws(Throwable::class)
    override suspend fun submitCardCVV(intentId: String, cvv: String): PaymentIntentData {
        DefaultLogManager.info("Start sending CVV for intent ID: $intentId, CVV: ${cvv}")
        val cvvBody = JsonObject(mapOf("cvv" to JsonPrimitive(cvv)))
        DefaultLogManager.debug("Sending POST request to end point: ${Type.PAYMENT.resultSuffix}${intentId}/${Constants.CVV_END_POINT}/")
        DefaultLogManager.debug("Request Body: ${cvvBody}/")
        val body: PaymentIntentData = try {
            paymentService.post<PaymentIntentData>() {
                endpoint("${Type.PAYMENT.resultSuffix}${intentId}/${Constants.CVV_END_POINT}/")
                body(cvvBody)
            }
        } catch (e: Throwable) {
            DefaultLogManager.error("Failed to send CVV for intent ID: $intentId", e)
            throw e
        }
        DefaultLogManager.verbose("Response after sending CVV: $body")
        DefaultLogManager.info("Successfully sent CVV for intent ID: $intentId")
        return body
    }
}
