package com.moneyhash.shared.interacators.payment

import com.moneyhash.shared.datasource.network.model.StatusResponse
import com.moneyhash.shared.datasource.network.model.card.toCardState
import com.moneyhash.shared.datasource.network.model.common.MethodMetaData
import com.moneyhash.shared.datasource.network.model.discount.DiscountItem
import com.moneyhash.shared.datasource.network.model.discount.DiscountResponse
import com.moneyhash.shared.datasource.network.model.fees.FeeItem
import com.moneyhash.shared.datasource.network.model.fees.FeesResponse
import com.moneyhash.shared.datasource.network.model.payment.PaymentInformation
import com.moneyhash.shared.datasource.network.model.payment.PaymentIntentData
import com.moneyhash.shared.datasource.network.model.payment.methods.MethodType
import com.moneyhash.shared.datasource.network.model.payment.methods.IntentMethods
import com.moneyhash.shared.datasource.network.model.payment.methods.toPaymentMethods
import com.moneyhash.shared.datasource.network.model.payment.methods.toPayoutMethods
import com.moneyhash.shared.datasource.network.model.payout.PayoutData
import com.moneyhash.shared.datasource.network.model.payout.PayoutDetails
import com.moneyhash.shared.datasource.network.model.vault.VaultData
import com.moneyhash.shared.datasource.services.PaymentService
import com.moneyhash.shared.domain.model.CardState
import com.moneyhash.shared.errorhandling.ErrorType
import com.moneyhash.shared.errorhandling.MHThrowable
import com.moneyhash.shared.errorhandling.executeCatching
import com.moneyhash.shared.localization.LocalizationManager
import com.moneyhash.shared.util.Type
import com.moneyhash.shared.util.extensions.toJsonObject

class PaymentUseCase(
    private val paymentService: PaymentService,
) {
    @Throws(Throwable::class)
    suspend fun getPayoutInformation(payoutId: String): PayoutDetails =
        executeCatching(block=  {
            paymentService.getPayoutInformation(payoutIntentId = payoutId)
        })
    @Throws(Throwable::class)
    suspend fun getCardInformation(cardIntentId: String): CardState =
        executeCatching(block =  {
            val cardInfo = paymentService.getCardInformation(cardIntentId = cardIntentId)
            cardInfo.toCardState()
        })
    @Throws(Throwable::class)
    suspend fun getPaymentInformation(paymentId: String): PaymentIntentData =
        executeCatching(block = {
            paymentService.getPaymentInformation(paymentIntentId = paymentId)
        })
    @Throws(Throwable::class)
    suspend fun getIntentMethods(paymentIntentId: String, type: Type): IntentMethods =
        executeCatching(block =  {
            if(type == Type.PAYMENT) {
                val paymentInfo = paymentService.getPaymentInformation(paymentIntentId = paymentIntentId)
                paymentInfo.toPaymentMethods()
            } else if (type == Type.PAYOUT) {
                val payoutInfo = paymentService.getPayoutInformation(payoutIntentId = paymentIntentId)
                payoutInfo.data.toPayoutMethods()
            } else {
                throw MHThrowable(
                    message = LocalizationManager.strings.invalid_intent_type,
                    type = ErrorType.UNKNOWN
                )
            }
        })
    @Throws(Throwable::class)
    suspend fun proceedWithPaymentMethod(
        intentId: String,
        selectedMethodId: String,
        methodType: MethodType,
        metaData: MethodMetaData? = null
    ):Pair<IntentMethods, PaymentInformation> = executeCatching(block =  {
        val paymentInfo = when (methodType) {
            MethodType.SAVE_CARD -> {
                paymentService.useSavedCard(
                    paymentIntentId = intentId,
                    cardTokenId = selectedMethodId,
                    cvv = metaData?.cvv
                )
            }
            MethodType.PAYMENT_METHOD, MethodType.EXPRESS_METHOD -> {
                paymentService.usePaymentMethod(
                    intentId = intentId,
                    methodName = selectedMethodId
                )
            }
            MethodType.CUSTOMER_BALANCE -> {
                paymentService.useSelfWallet(paymentIntentId = intentId)
            }
            else -> throw MHThrowable(
                message = LocalizationManager.strings.unsupported_payment_intent_method("$methodType") ,
                type = ErrorType.UNKNOWN
            )
        }
        val paymentMethods = paymentInfo.toPaymentMethods()
        paymentMethods to paymentInfo.data!!
    })
    @Throws(Throwable::class)
    suspend fun proceedWithPayoutMethod(
        intentId: String,
        selectedMethodId: String,
        methodType: MethodType,
        metaData: MethodMetaData? = null
    ): Pair<IntentMethods, PayoutData> = executeCatching(block =  {
        val payoutInfo = when (methodType) {
            MethodType.PAYOUT_METHOD -> {
                paymentService.usePayoutMethod(
                    intentId = intentId,
                    methodName = selectedMethodId
                )
            }
            else -> throw MHThrowable(
                message = LocalizationManager.strings.unsupported_payout_intent_method("$methodType"),
                type = ErrorType.UNKNOWN
            )
        }
        val payoutMethods = payoutInfo.data.toPayoutMethods()
        payoutMethods to payoutInfo.data!!
    })

    @Throws(Throwable::class)
    suspend fun deleteSavedCard(
        cardTokenId: String,
        intentSecret: String
    ):StatusResponse =
        executeCatching(block =  {
            paymentService.deleteSavedCard(cardTokenId = cardTokenId, secret = intentSecret)
        })

    @Throws(Throwable::class)
    suspend fun resetPaymentSelectedMethod(
        intentId: String,
    ):Pair<IntentMethods, PaymentInformation> = executeCatching(block =  {
        val paymentInfo = paymentService.resetPaymentSelectedMethod(paymentIntentId = intentId)
        val paymentMethods = paymentInfo.toPaymentMethods()
        paymentMethods to paymentInfo.data!!
    })

    @Throws(Throwable::class)
    suspend fun resetPayoutSelectedMethod(
        intentId: String,
    ):Pair<IntentMethods, PayoutData> = executeCatching(block =  {
        val payoutInfo = paymentService.resetPayoutSelectedMethod(payoutIntentId = intentId)
        val payoutMethods = payoutInfo.data.toPayoutMethods()
        payoutMethods to payoutInfo.data!!
    })
    @Throws(Throwable::class)
    suspend fun submitPaymentReceipt(
        paymentIntentId: String,
        receipt: String,
    ):PaymentInformation = executeCatching(block =  {
        paymentService.submitReceipt(paymentIntentId, receipt)
    })

    @Throws(Throwable::class)
    suspend fun submitForm(
        paymentIntentId: String,
        method: String,
        billingFields: Map<String,String>? = null,
        shippingFields: Map<String,String>? = null,
        vaultData: VaultData? = null
    ):PaymentIntentData = executeCatching(block =  {
        paymentService.submitForm(
            paymentIntentId,
            method,
            billingFields = billingFields?.toJsonObject(),
            shippingFields = shippingFields?.toJsonObject(),
            vaultData = vaultData
        )
    })

    @Throws(Throwable::class)
    suspend fun submitCardCVV(
        intentId: String,
        cvv: String
    ): PaymentIntentData = executeCatching(block = {
        paymentService.submitCardCVV(intentId, cvv)
    })

    @Throws(Throwable::class)
    suspend fun updateFees(
        intentId: String
        , fees: List<FeeItem>
    ): FeesResponse = executeCatching(block = {
        paymentService.updateFees(intentId, fees)
    })

    @Throws(Throwable::class)
    suspend fun updateDiscount(
        intentId: String,
        discount: DiscountItem
    ): DiscountResponse = executeCatching(block = {
        paymentService.updateDiscount(intentId, discount)
    })
}