package com.moneyhash.shared.securevault.collectors

import com.moneyhash.shared.datasource.network.model.vault.Brand
import com.moneyhash.shared.datasource.network.model.vault.CardBrand
import com.moneyhash.shared.datasource.network.model.vault.CardDetails
import com.moneyhash.shared.datasource.network.model.vault.VaultData
import com.moneyhash.shared.datasource.services.VaultService
import com.moneyhash.shared.localization.LocalizationManager
import com.moneyhash.shared.securevault.Weak
import com.moneyhash.shared.securevault.exceptions.CardFormExceptions
import com.moneyhash.shared.securevault.fields.FieldType
import com.moneyhash.shared.securevault.fields.SecureTextFieldVM
import com.moneyhash.shared.securevault.formatters.TextFormatter
import com.moneyhash.shared.securevault.models.CardBrandInternal
import com.moneyhash.shared.securevault.validators.Validator
import com.moneyhash.shared.util.DefaultLogManager
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.datetime.Clock
import kotlinx.datetime.TimeZone
import kotlinx.datetime.number
import kotlinx.datetime.toLocalDateTime

class CardCollector(private val vaultService: VaultService) {

    private val _brandChangeFlow = MutableStateFlow<CardBrand>(CardBrandInternal.UNKNOWN.getCardBrand(""))
    private var lastCardBrandInternal: CardBrandInternal? = null

    internal var store: MutableMap<FieldType, SecureTextFieldVM> = mutableMapOf()
    internal fun registerField(fieldType: FieldType, collectableField: SecureTextFieldVM) {
        DefaultLogManager.info("Registering field: $fieldType")
        collectableField.setCollector(Weak(this))
        store[fieldType] = collectableField
    }

    val brandChangeFlow = _brandChangeFlow.asStateFlow()
    internal val cardBrandInternal: CardBrandInternal
        get() {
            val cardCollectable = store[FieldType.CARD_NUMBER]?.data.toString()
            return CardBrandInternal.fromCardNumber(cardCollectable)
        }

    internal fun formatter(type: FieldType): TextFormatter {
        DefaultLogManager.info("Retrieving formatter for type: $type")
        return cardBrandInternal.formatter(type)
    }

    internal fun validator(type: FieldType): Validator {
        DefaultLogManager.info("Retrieving validator for type: $type")
        return cardBrandInternal.validator(type)
    }

    private val cardNumber: String?
        get() = store[FieldType.CARD_NUMBER]?.data as? String
    private val cardHolderName: String?
        get() = store[FieldType.CARD_HOLDER_NAME]?.data as? String
    private val cardCVV: String?
        get() = store[FieldType.CVV]?.data as? String
    private val cardExpireMonth: String?
        get() = store[FieldType.EXPIRE_MONTH]?.data as? String
    private val cardExpireYear: String?
        get() = store[FieldType.EXPIRE_YEAR]?.data as? String

    @Throws(Throwable::class)
    suspend fun collect(token: String, intentID: String, shouldSaveCard:Boolean = false): VaultData? {
        DefaultLogManager.info("Collecting card data for intent ID: $intentID with token: $token")
        DefaultLogManager.debug("Card details: holderName=$cardHolderName, number=$cardNumber, cvv=$cardCVV, expireMonth=$cardExpireMonth, expireYear=$cardExpireYear")

        return try {
            val cardEmbed = vaultService.postTokens(
                accessToken = token,
                cardDetails = CardDetails(
                    cardHolderName = cardHolderName,
                    cardNumber = cardNumber,
                    expiryMonth = cardExpireMonth,
                    expiryYear = cardExpireYear,
                    cvv = cardCVV,
                    saveCard = shouldSaveCard
                )
            )
            DefaultLogManager.info("Successfully collected card data for intent ID: $intentID")
            cardEmbed
        } catch (e: Throwable) {
            DefaultLogManager.error("Failed to collect card data for intent ID: $intentID", e)
            throw e
        }
    }

    @Throws(Throwable::class)
    fun validate(): Boolean {
        DefaultLogManager.info("Validating card data")

        val fieldTypes = FieldType.entries.filterNot { it == FieldType.CARD_HOLDER_NAME }.toTypedArray() // Assuming CARD_HOLDER_NAME is not required
        val storeFieldTypes = store.keys.filter { store[it] != null }

        val missingFieldTypes = fieldTypes.filter { it !in storeFieldTypes }
        val redundantFieldTypes =
            storeFieldTypes.groupingBy { it }.eachCount().filter { it.value > 1 }.keys

        if (missingFieldTypes.isNotEmpty()) {
            val message = "Missing: ${missingFieldTypes.joinToString { it.label }}"
            DefaultLogManager.error(message)
            throw CardFormExceptions.MissingFieldTypeException(LocalizationManager.strings.missing_fields(missingFieldTypes.joinToString { it.label }))
        }

        if (redundantFieldTypes.isNotEmpty()) {
            val message = "Redundant: ${redundantFieldTypes.joinToString { it.label }}"
            DefaultLogManager.error(message)
            throw CardFormExceptions.RedundantFieldTypeException(LocalizationManager.strings.redundant_fields(redundantFieldTypes.joinToString { it.label }))
        }

        val isAllFieldsValid = store.values.all {
            try {
                it.validate()
            } catch (e: Throwable) {
                val message = "Invalid ${it.type.label}"
                DefaultLogManager.error(message, e)
                throw CardFormExceptions.InvalidFieldException(LocalizationManager.strings.invalid_field(it.type.label))
            }
        }

        val currentDate = Clock.System.now().toLocalDateTime(TimeZone.currentSystemDefault())
        val currentYear = currentDate.year
        val currentMonth = currentDate.month.number
        val inputYear = cardExpireYear!!.toInt() + 2000
        val isDateValid = when {
            inputYear < currentYear -> false
            inputYear == currentYear && currentMonth > cardExpireMonth!!.toInt() -> {
                val message = "Card expiration date ($cardExpireMonth/$inputYear) is in the past."
                DefaultLogManager.error(message)
                throw CardFormExceptions.ExpireDateIsInThePast(LocalizationManager.strings.expiration_date_past)
            }

            else -> true
        }

        DefaultLogManager.info("Validation successful: $isAllFieldsValid and date valid: $isDateValid")
        return isAllFieldsValid && isDateValid
    }

    internal fun contentChanged() {
        DefaultLogManager.debug("Content Changed")
        DefaultLogManager.debug("Current brand: $cardBrandInternal")
        DefaultLogManager.debug("last brand: $lastCardBrandInternal")

        val currentBrand = cardBrandInternal
        DefaultLogManager.info("Card brand changed from $lastCardBrandInternal to $currentBrand")
        lastCardBrandInternal = currentBrand
        val cardNumberPrefix = cardNumber?.take(6) ?: ""
        if (cardNumberPrefix.length > 5)  {
            _brandChangeFlow.update { currentBrand.getCardBrand(cardNumberPrefix) }
        } else {
            _brandChangeFlow.update {  CardBrandInternal.UNKNOWN.getCardBrand(cardNumber ?: "") }
        }
    }
}
