package com.moneyhash.shared.securevault.collectors

import com.moneyhash.shared.CommonParcelable
import com.moneyhash.shared.CommonParcelize
import com.moneyhash.shared.datasource.network.model.vault.CardBrand
import com.moneyhash.shared.datasource.network.model.vault.CardDetails
import com.moneyhash.shared.datasource.network.model.vault.VaultData
import com.moneyhash.shared.datasource.services.VaultService
import com.moneyhash.shared.localization.LocalizationManager
import com.moneyhash.shared.securevault.exceptions.CardFormExceptions
import com.moneyhash.shared.securevault.fields.FieldType
import com.moneyhash.shared.securevault.models.CardBrandInternal
import com.moneyhash.shared.util.DefaultLogManager
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable

/**
* This is implemented to collect card fields and card the case of integrating with cross-platform SDKs like Flutter or React Native.
 * The point that flutter or react native can't use the custom fields or the view models directly
 * So each platform should implement their own fields and view models and use this class to collect the fields and interact with the card collector.
 * This class is not used in the Android SDK or iOS SDK and it will be used only in the cross-platform SDKs and exposed via the native SDKs with custom versions.
 **/

@Serializable
@CommonParcelize
data class CardFieldState(
    @SerialName("isValid")
    val isValid: Boolean,
    @SerialName("errorMessage")
    val errorMessage: String? = null,
    @SerialName("formattedValue")
    val formattedValue: String,
    @SerialName("fieldType")
    val fieldType: FieldType,
    @SerialName("cardBrand")
    val cardBrand: CardBrand? = null,
    @SerialName("length")
    val length: Int? = null
) : CommonParcelable

@Serializable
@CommonParcelize
data class CardFormState(
    @SerialName("isValid")
    val isValid:Boolean,
    @SerialName("fieldsState")
    val fieldsState: List<CardFieldState>
) : CommonParcelable
class CardFieldsCollector(private val vaultService: VaultService) {

    internal val cardBrandInternal: CardBrandInternal
        get() {
            return CardBrandInternal.fromCardNumber(lastCardNumber)
        }

    internal var lastCardNumber = ""
    /**
     *  Validate each field, and return the current formatted value
     *  DON'T throw any exception, just return the state of the field and the error message if any
     **/
    fun validateField(fieldType: FieldType, currentValue: String): CardFieldState {
        if (fieldType == FieldType.CARD_NUMBER) {
            lastCardNumber = currentValue
                .replace(" ", "")
                .replace("-", "")
                .replace("/", "")
        }
        val formatter = cardBrandInternal.formatter(fieldType)
        val validator = cardBrandInternal.validator(fieldType)
        val unFormattedValue = formatter.removeFormat(currentValue)
        val formattedValue = formatter.format(unFormattedValue)

        val cardNumberPrefix = lastCardNumber.take(6)
        val cardBrand = if (cardNumberPrefix.length > 5)  {
            cardBrandInternal.getCardBrand(cardNumberPrefix)
        } else {
            CardBrandInternal.UNKNOWN.getCardBrand(cardNumberPrefix)
        }
        return try {
            val isValid = validator.validate(formatter.removeFormat(formattedValue))

            CardFieldState(
                isValid = isValid,
                formattedValue = formattedValue,
                fieldType = fieldType,
                cardBrand = cardBrand,
                length = unFormattedValue.length
            )
        } catch (e: Throwable) {
            // Capture any exception and return an invalid state with the error message
            CardFieldState(
                isValid = false,
                errorMessage = LocalizationManager.strings.invalid_field(fieldType.label),
                formattedValue = formattedValue,
                fieldType = fieldType,
                cardBrand = cardBrand,
                length = unFormattedValue.length
            )
        }
    }

    /**
     *  Validate the whole card fields, and return if the card is valid or not
     *  in Flutter or React Native, this method should be called to validate the whole card fields
     *  then based on the result, the user can enable or disable the submit button for all the fields
     **/
    fun isValidForm(
        fields: Map<FieldType, String>
    ): CardFormState {
        // Implement the validation logic for the whole card fields
        val fieldsState = fields.map { (fieldType, currentValue) ->
            validateField(fieldType, currentValue)
        }
        return CardFormState(fieldsState.all { it.isValid }, fieldsState)

    }

    /**
     *  Collect the card fields and return the card state
     *  This method should be called to collect the card fields and return the card state
     *  then based on the result, the user can show the error message or the success message
     **/
    @Throws(Throwable::class)
    suspend fun collect(
        fields: Map<FieldType, String>,
        token: String,
        intentID: String,
        shouldSaveCard:Boolean
    ): VaultData? {
        DefaultLogManager.info("Collecting card data for intent ID: $intentID with token: $token")
        DefaultLogManager.debug("Card details: holderName=${fields[FieldType.CARD_HOLDER_NAME]}, number=${fields[FieldType.CARD_NUMBER]}, cvv=${fields[FieldType.CVV]}, expireMonth=${fields[FieldType.EXPIRE_MONTH]}, expireYear=${fields[FieldType.EXPIRE_YEAR]}")
        return try {
            val cardDetails = CardDetails(
                cardHolderName = fields[FieldType.CARD_HOLDER_NAME]?.let { cardBrandInternal.formatter(FieldType.CARD_HOLDER_NAME).removeFormat(it) },
                cardNumber = fields[FieldType.CARD_NUMBER]?.let { cardBrandInternal.formatter(FieldType.CARD_NUMBER).removeFormat(it) },
                expiryMonth = fields[FieldType.EXPIRE_MONTH]?.let { cardBrandInternal.formatter(FieldType.EXPIRE_MONTH).removeFormat(it) },
                expiryYear = fields[FieldType.EXPIRE_YEAR]?.let { cardBrandInternal.formatter(FieldType.EXPIRE_YEAR).removeFormat(it) },
                cvv = fields[FieldType.CVV]?.let { cardBrandInternal.formatter(FieldType.CVV).removeFormat(it) },
                saveCard = shouldSaveCard
            )

            val cardEmbed = vaultService.postTokens(
                accessToken = token,
                cardDetails = cardDetails
            )
            lastCardNumber = ""
            DefaultLogManager.info("Successfully collected card data for intent ID: $intentID")
            cardEmbed
        } catch (e: Throwable) {
            DefaultLogManager.error("Failed to collect card data for intent ID: $intentID", e)
            throw e
        }
    }
}