package com.moneyhash.shared.securevault.fields

import com.moneyhash.shared.localization.LocalizationManager
import com.moneyhash.shared.securevault.Weak
import com.moneyhash.shared.securevault.collectors.CardCollector
import com.moneyhash.shared.securevault.exceptions.CardFormExceptions
import com.moneyhash.shared.securevault.formatters.TextFormatter
import com.moneyhash.shared.securevault.validators.Validator
import com.moneyhash.shared.util.DefaultLogManager

class SecureTextFieldVM(
    private val configuration: SecureTextFieldConfiguration
)  {

    private var _text = ""
    private var collector: Weak<CardCollector>? = null

    val formattedValue: String
        get() = formatter?.format(_text) ?: _text

    init {
        DefaultLogManager.info("Initializing SecureTextFieldVM with configuration type: ${configuration.type}")
        configuration.collector.registerField(configuration.type, this)
    }

    fun set(value: String) {
        DefaultLogManager.debug("Setting value: $value")
        _text = formatter?.removeFormat(formatter?.format(value) ?: value) ?: value
        DefaultLogManager.debug("Formatted value set to: $_text")
        collector?.get()?.contentChanged()
    }

    private val formatter: TextFormatter?
        get() {
            val formatter = collector?.get()?.formatter(configuration.type)
            DefaultLogManager.debug("Retrieved formatter for type: ${configuration.type} -> $formatter")
            return formatter
        }

    private val validator: Validator?
        get() {
            val validator = collector?.get()?.validator(configuration.type)
            DefaultLogManager.debug("Retrieved validator for type: ${configuration.type} -> $validator")
            return validator
        }

    fun setCollector(collector: Weak<CardCollector>) {
        DefaultLogManager.info("Setting collector for SecureTextFieldVM")
        this.collector = collector
    }

    @Throws(Throwable::class)
    fun validate(): Boolean {
        DefaultLogManager.info("Validating text: $_text")
        var validation: Boolean? = null
        try {
            validation = validator?.validate(_text)
        } catch (e: Throwable) {
            val message = "Invalid ${this.type.label}"
            DefaultLogManager.error(message, e)
            throw CardFormExceptions.InvalidFieldException(LocalizationManager.strings.invalid_field(this.type.label))
        }
        if (validation == null) {
            val message = "Validator does not exist for type: ${configuration.type}"
            DefaultLogManager.error(message)
            throw Throwable(LocalizationManager.strings.validator_not_exist("${configuration.type}"))
        }
        DefaultLogManager.info("Validation result for text: $_text -> $validation")
        return validation
    }

    val type: FieldType
        get() = configuration.type

    internal val data: String
        get() {
            DefaultLogManager.debug("Retrieving data: $_text")
            return _text
        }

    val unformattedInputLength: Int
        get() = _text.length

}
