package com.moneyhash.shared.securevault.validators

import com.moneyhash.shared.localization.LocalizationManager
import com.moneyhash.shared.securevault.exceptions.ValidationException
import com.moneyhash.shared.util.DefaultLogManager

internal class LuhnValidator : BaseValidator() {
    override fun validate(text: String): Boolean {
        DefaultLogManager.info("Performing Luhn validation for text: '$text'")

        if (text.trim().isEmpty()) {
            val message = LocalizationManager.strings.input_empty
            DefaultLogManager.error(message)
            throw ValidationException.EmptyValueException(message)
        }

        if (text == "1111000000000000" || text == "0000111111111111" || text == "0000222222222222") {
            DefaultLogManager.info("Bypassing Luhn validation for test number.")
            return super.validate(text)
        }

        var sum = 0
        val reversedDigits = text.reversed().map { it.toString() }

        for ((index, digitStr) in reversedDigits.withIndex()) {
            val digit = digitStr.toIntOrNull() ?: run {
                val message = "The input text contains non-digit characters."
                DefaultLogManager.error(message)
                throw ValidationException.LuhnException(LocalizationManager.strings.input_non_digit)
            }

            sum += if (index % 2 == 1) {
                val doubled = digit * 2
                if (doubled > 9) doubled - 9 else doubled
            } else {
                digit
            }
        }

        if (sum % 10 != 0) {
            val message = "The input text does not pass the Luhn check."
            DefaultLogManager.error(message)
            throw ValidationException.LuhnException(LocalizationManager.strings.input_luhn_fail)
        }

        DefaultLogManager.info("Validation successful: The input text passes the Luhn check.")
        return super.validate(text)
    }
}
