package com.moneyhash.shared.util

import co.touchlab.kermit.Logger
import co.touchlab.kermit.Severity
import com.moneyhash.shared.getPlatform

interface LoggerManager {
    fun setSeverity(severity: LoggerSeverity)
    fun verbose(message: String, throwable: Throwable? = null, tag: String? = null)
    fun debug(message: String, throwable: Throwable? = null, tag: String? = null)
    fun info(message: String, throwable: Throwable? = null, tag: String? = null)
    fun warning(message: String, throwable: Throwable? = null, tag: String? = null)
    fun error(message: String, throwable: Throwable? = null, tag: String? = null)
    fun assert(message: String, throwable: Throwable? = null, tag: String? = null)

    fun logDeviceInfo()
}


enum class LoggerSeverity {
    Verbose,
    Debug,
    Info,
    Warn,
    Error,
    Assert,
    None
}

internal fun LoggerSeverity.toKermitSeverity(): Severity {
    return when (this) {
        LoggerSeverity.Verbose -> Severity.Verbose
        LoggerSeverity.Debug -> Severity.Debug
        LoggerSeverity.Info -> Severity.Info
        LoggerSeverity.Warn -> Severity.Warn
        LoggerSeverity.Error -> Severity.Error
        LoggerSeverity.Assert -> Severity.Assert
        LoggerSeverity.None -> Severity.Error // Default to Error
    }
}

object DefaultLogManager : LoggerManager {
    private var logLevel: LoggerSeverity = LoggerSeverity.Info
    override fun setSeverity(severity: LoggerSeverity) {
        logLevel = severity
        Logger.setMinSeverity(severity.toKermitSeverity())
        logDeviceInfo()
        info("Log Level Set to $severity")
    }

    override fun verbose(message: String, throwable: Throwable?, tag: String?) {
        if (logLevel == LoggerSeverity.None) return
        Logger.v("[" + getPlatform().name + "] - " + message, throwable, tag = tag ?: Logger.tag)
    }

    override fun debug(message: String, throwable: Throwable?, tag: String?) {
        if (logLevel == LoggerSeverity.None) return
        Logger.d("[" + getPlatform().name + "] - " + message, throwable, tag = tag ?: Logger.tag)
    }

    override fun info(message: String, throwable: Throwable?, tag: String?) {
        if (logLevel == LoggerSeverity.None) return
        Logger.i("[" + getPlatform().name + "] - " + message, throwable, tag = tag ?: Logger.tag)
    }

    override fun warning(message: String, throwable: Throwable?, tag: String?) {
        if (logLevel == LoggerSeverity.None) return
        Logger.w("[" + getPlatform().name + "] - " + message, throwable, tag = tag ?: Logger.tag)
    }

    override fun error(message: String, throwable: Throwable?, tag: String?) {
        if (logLevel == LoggerSeverity.None) return
        Logger.e("[" + getPlatform().name + "] - " + message, throwable, tag = tag ?: Logger.tag)
    }

    override fun assert(message: String, throwable: Throwable?, tag: String?) {
        if (logLevel == LoggerSeverity.None) return
        Logger.a("[" + getPlatform().name + "] - " + message, throwable, tag = tag ?: Logger.tag)
    }

    override fun logDeviceInfo() {
        if (logLevel == LoggerSeverity.None) return
        info("Device Info: ${getPlatform().deviceInfo}")
    }
}