package com.moneyhash.shared.datasource.network.model.payment

import com.moneyhash.shared.CommonParcelable
import com.moneyhash.shared.CommonParcelize
import com.moneyhash.shared.datasource.network.model.BrandSettingsData
import com.moneyhash.shared.datasource.network.model.common.IntentState
import com.moneyhash.shared.datasource.network.model.payment.methods.IntentMethods
import com.moneyhash.shared.datasource.network.model.payment.methods.PaymentMethod
import com.moneyhash.shared.datasource.network.model.payment.methods.SavedCard
import com.moneyhash.shared.datasource.network.model.payment.methods.toPaymentMethods
import com.moneyhash.shared.domain.model.PaymentActions
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable

@Serializable
@CommonParcelize
data class PaymentInformation(
    @SerialName("intent_sdk_state")
    val intentSdkState: IntentState? = null,
    @SerialName("selected_method")
    val selectedMethod: String? = null,
    @SerialName("next_action")
    val nextAction: String? = null,
    @SerialName("intent")
    val intent: PaymentIntent? = null,
    @SerialName("saved_cards")
    val savedCards: List<SavedCard>? = null,
    @SerialName("wallet")
    val wallet: Double? = null,
    @SerialName("brand_settings")
    val brandSettings: BrandSettingsData? = null,
    @SerialName("action_data")
    val actionData: IntentActionData? = null
): CommonParcelable

fun PaymentInformation.getRedirectStrategy():RenderStrategy {
    return when (nextAction) {
        PaymentActions.REDIRECT -> {
            return RenderStrategy.REDIRECT
        }
        PaymentActions.POPUP_IFRAME -> {
            return RenderStrategy.POPUP_IFRAME
        }
        PaymentActions.IFRAME -> {
            return  RenderStrategy.IFRAME
        }

        else -> {
            RenderStrategy.NONE
        }
    }
}

fun PaymentInformation.getIntentStateDetails(): IntentStateDetails? {
    when(this.intentSdkState) {
        IntentState.CLOSED -> {
            return IntentStateDetails.Closed
        }

        IntentState.METHOD_SELECTION -> {
            return  IntentStateDetails.MethodSelection(this.toPaymentMethods())
        }

        IntentState.INTENT_FORM -> {
            return  IntentStateDetails.IntentForm
        }
        IntentState.INTENT_PROCESSED -> {
            return  IntentStateDetails.IntentProcessed
        }
        IntentState.TRANSACTION_WAITING_USER_ACTION -> {
            return  IntentStateDetails.TransactionWaitingUserAction
        }
        IntentState.TRANSACTION_FAILED -> {
            val recommendedMethods: List<PaymentMethod>? = this.actionData?.paymentMethods?.toPaymentMethods(null)
            val intentMethods = recommendedMethods?.let { IntentMethods(paymentMethods = it) }
            return IntentStateDetails.TransactionFailed(intentMethods)
        }
        IntentState.EXPIRED -> {
            return  IntentStateDetails.Expired
        }
        IntentState.FORM_FIELDS -> {
            return  IntentStateDetails.FormFields(
                tokenizeCardInfo = this.actionData?.formFields?.tokenizeCardInfo,
                billingFields = this.actionData?.getBillingFields(),
                shippingFields = this.actionData?.getShippingFields()
            )
        }
        IntentState.URL_TO_RENDER -> {
            var url = this.actionData?.redirect?.redirectUrl
            if (url == null) {
                url = this.actionData?.iframeUrl
            }
            return  IntentStateDetails.UrlToRender(url = url, renderStrategy = this.getRedirectStrategy())
        }
        IntentState.SAVED_CARD_CVV -> {
            this.actionData?.cardTokenData ?: return null
            return IntentStateDetails.SavedCardCVV.getDefaultCVVField(this.actionData.cardTokenData)
        }
        IntentState.NATIVE_PAY -> {
            val selectedMethod = this.selectedMethod ?: return null
            val nativePayData = this.actionData?.nativePaymentData?.toNativePayData(selectedMethod) ?: return null
            return IntentStateDetails.NativePay(nativePayData)
        }
        else -> return null
    }
}

