package com.moneyhash.shared.securevault.collectors

import com.moneyhash.shared.datasource.network.model.vault.CardBrand
import com.moneyhash.shared.securevault.fields.FieldType
import com.moneyhash.shared.securevault.formatters.TextFormatter
import com.moneyhash.shared.securevault.models.CardBrandInternal
import com.moneyhash.shared.securevault.validators.Validator
import com.moneyhash.shared.util.DefaultLogManager
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update

internal class CardBrandManager {
    private val _brandChangeFlow = MutableStateFlow<CardBrand>(CardBrandInternal.UNKNOWN.getCardBrand(""))
    val brandChangeFlow = _brandChangeFlow.asStateFlow()
    private var lastCardBrandInternal: CardBrandInternal? = null

    fun getCardBrandInternal(cardNumber: String?): CardBrandInternal {
        return cardNumber?.let { CardBrandInternal.fromCardNumber(it) } ?: CardBrandInternal.UNKNOWN
    }

    fun updateBrand(cardNumber: String?) {
        val currentBrand = getCardBrandInternal(cardNumber)
        DefaultLogManager.info("Card brand changed from $lastCardBrandInternal to $currentBrand")
        lastCardBrandInternal = currentBrand
        val cardNumberPrefix = cardNumber?.take(6) ?: ""
        _brandChangeFlow.update {
            if (cardNumberPrefix.length > 5) currentBrand.getCardBrand(cardNumberPrefix)
            else CardBrandInternal.UNKNOWN.getCardBrand(cardNumber ?: "")
        }
    }

    fun getFormatter(type: FieldType, cardNumber: String?): TextFormatter {
        val cardBrandInternal = getCardBrandInternal(cardNumber)
        DefaultLogManager.info("Retrieving formatter for type: $type")
        return cardBrandInternal.formatter(type)
    }

    fun getValidator(type: FieldType, cardNumber: String?): Validator {
        val cardBrandInternal = getCardBrandInternal(cardNumber)
        DefaultLogManager.info("Retrieving validator for type: $type")
        return cardBrandInternal.validator(type)
    }
}