package com.moneyhash.shared.securevault.collectors

import com.moneyhash.shared.datasource.network.model.card.getIntentStateDetails
import com.moneyhash.shared.datasource.network.model.payment.IntentStateDetails
import com.moneyhash.shared.datasource.network.model.payment.PaymentIntentData
import com.moneyhash.shared.datasource.network.model.vault.CardFieldState
import com.moneyhash.shared.datasource.network.model.vault.CardFormState
import com.moneyhash.shared.datasource.network.model.vault.VaultData
import com.moneyhash.shared.datasource.services.PaymentService
import com.moneyhash.shared.datasource.services.VaultService
import com.moneyhash.shared.localization.LocalizationManager
import com.moneyhash.shared.securevault.fields.FieldType
import com.moneyhash.shared.securevault.fields.SecureTextFieldVM
import com.moneyhash.shared.util.DefaultLogManager
import com.moneyhash.shared.util.extensions.toJsonObject
import kotlinx.serialization.json.JsonObject

class CardCollector(
    private val vaultService: VaultService,
    private val paymentService: PaymentService
) {

    private val fieldManager: FieldManager = FieldManager()
    private val cardBrandManager: CardBrandManager = CardBrandManager()
    private val formValidator: FormValidator = FormValidator(fieldManager, cardBrandManager)
    private val cardDataCollector: CardDataCollector = CardDataCollector(vaultService, paymentService)

    fun registerField(fieldType: FieldType, collectableField: SecureTextFieldVM) {
        fieldManager.registerField(fieldType, collectableField, this)
    }

    val brandChangeFlow = cardBrandManager.brandChangeFlow

    @Throws(Throwable::class)
    suspend fun collect(
        token: String? = null
    ): VaultData? {
        return collectCardData(token)
    }

    fun validate(): CardFormState {
        return formValidator.validateFields()
    }

    private fun updateBrand() {
        val cardNumber = fieldManager.getFieldValue(FieldType.CARD_NUMBER)
        cardBrandManager.updateBrand(cardNumber?.unformattedValue)
    }

    @Throws(Throwable::class)
    private suspend fun collectCardData(
        token: String? = null
    ): VaultData? {
        val store = fieldManager.getAllFields()
        return cardDataCollector.collectCardData(store, token)
    }

    @Throws(Throwable::class)
    suspend fun pay(
        intentId: String,
        cardData: VaultData,
        saveCard: Boolean,
        billingFields: Map<String,String>? = null,
        shippingFields: Map<String,String>? = null
    ) : PaymentIntentData {
        return cardDataCollector.pay(intentId, cardData, saveCard, billingFields?.toJsonObject(), shippingFields?.toJsonObject())
    }

    @Throws(Throwable::class)
    suspend fun createCardToken(
        cardIntentId: String,
        cardData: VaultData,
    ) : IntentStateDetails? {
        return cardDataCollector.createCardToken(cardIntentId, cardData).getIntentStateDetails()
    }

    /**
     * Updates each field and returns the current formatted value.
     * Does not throw exceptions; returns the state of the field and any error message.
     **/
    fun updateField(fieldType: FieldType, currentValue: String): CardFieldState {
        DefaultLogManager.debug("Updating field: $fieldType with value: $currentValue")

        val formatter = cardBrandManager.getFormatter(fieldType, fieldManager.getFieldValue(FieldType.CARD_NUMBER)?.unformattedValue)
        val validator = cardBrandManager.getValidator(fieldType, fieldManager.getFieldValue(FieldType.CARD_NUMBER)?.unformattedValue)
        val unFormattedValue = formatter.removeFormat(currentValue)
        val formattedValue = formatter.format(unFormattedValue)
        fieldManager.updateFieldValue(fieldType, formattedValue, unFormattedValue)
        updateBrand()

        return try {
            val isValid = validator.validate(formatter.removeFormat(formattedValue))
            CardFieldState(
                isValid = isValid,
                formattedValue = formattedValue,
                fieldType = fieldType,
                cardBrand = brandChangeFlow.value,
                length = unFormattedValue.length
            )
        } catch (e: Throwable) {
            // Capture any exception and return an invalid state with the error message
            CardFieldState(
                isValid = false,
                errorMessage = LocalizationManager.strings.invalid_field(fieldType.label),
                formattedValue = formattedValue,
                fieldType = fieldType,
                cardBrand = brandChangeFlow.value,
                length = unFormattedValue.length
            )
        }
    }


}