package com.moneyhash.shared.securevault.collectors

import com.moneyhash.shared.datasource.network.model.payment.PaymentIntentData
import com.moneyhash.shared.datasource.network.model.vault.VaultData
import com.moneyhash.shared.datasource.services.PaymentService
import com.moneyhash.shared.datasource.services.VaultService
import com.moneyhash.shared.di.MoneyHashStore
import com.moneyhash.shared.securevault.exceptions.VaultCollectNetworkError
import com.moneyhash.shared.securevault.fields.FieldType
import com.moneyhash.shared.util.DefaultLogManager
import kotlinx.serialization.json.JsonObject
import com.moneyhash.shared.datasource.network.model.card.CardDetails
import com.moneyhash.shared.datasource.network.model.vault.CardFields

internal class CardDataCollector(
    private val vaultService: VaultService,
    private val paymentService: PaymentService
) {
    @Throws(Throwable::class)
    internal suspend fun collectCardData(
        store: Map<FieldType, FieldValue>,
        token: String? = null
    ): VaultData? {
        DefaultLogManager.debug("Collecting card data")

        // Check if the provided token exists or generate a new access token
        val accessToken = token ?: run {
            val publicKey = MoneyHashStore.publicKey ?: throw VaultCollectNetworkError.MissingPublicKey
            DefaultLogManager.debug("Using public key: $publicKey")
            generateAccessToken(publicKey)
        }

        DefaultLogManager.debug("Using access token: $accessToken")
        val cardDetails = createCardDetails(store)

        return try {
            val cardEmbed = vaultService.postTokens(accessToken, cardDetails)
            DefaultLogManager.info("Successfully collected card data")
            cardEmbed
        } catch (e: Throwable) {
            DefaultLogManager.error("Failed to collect card data", e)
            throw e
        }
    }

    @Throws(Throwable::class)
    internal suspend fun pay(
        intentId: String,
        cardData: VaultData,
        saveCard: Boolean,
        billingFields: JsonObject?,
        shippingFields: JsonObject?
        ) : PaymentIntentData {
        val updatedCardData = cardData.copy(saveCard = saveCard)
        return paymentService.submitForm(intentId, "CARD", null, billingFields, shippingFields, updatedCardData)
    }

    @Throws(Throwable::class)
    internal suspend fun createCardToken(
        cardIntentId: String,
        cardData: VaultData,
    ) : CardDetails {
        val updatedCardData = cardData.copy(saveCard = true)
        return paymentService.createCardToken(cardIntentId, updatedCardData)
    }

    @Throws(Throwable::class)
    private suspend fun generateAccessToken(publicKey: String): String {
        val accessTokenResponse = paymentService.generateAccessToken(publicKey)
        val accessToken = accessTokenResponse.data?.accessToken
            ?: throw VaultCollectNetworkError.TokenizationError
        DefaultLogManager.debug("Obtained access token: $accessToken")
        return accessToken
    }

    private fun createCardDetails(store: Map<FieldType, FieldValue>): CardFields {
        return CardFields(
            cardHolderName = store[FieldType.CARD_HOLDER_NAME]?.unformattedValue,
            cardNumber = store[FieldType.CARD_NUMBER]?.unformattedValue,
            expiryMonth = store[FieldType.EXPIRE_MONTH]?.unformattedValue,
            expiryYear = store[FieldType.EXPIRE_YEAR]?.unformattedValue,
            cvv = store[FieldType.CVV]?.unformattedValue
        )
    }
}