package com.moneyhash.shared.securevault.collectors

import com.moneyhash.shared.securevault.Weak
import com.moneyhash.shared.securevault.fields.FieldType
import com.moneyhash.shared.securevault.fields.SecureTextFieldVM
import com.moneyhash.shared.util.DefaultLogManager

data class FieldValue(
    val formattedValue: String,
    val unformattedValue: String
)
internal class FieldManager {

    // The store now holds FieldValue objects, which contain both formatted and unformatted values
    private val store: MutableMap<FieldType, FieldValue> = mutableMapOf()

    // Registers the field and sets the initial empty formatted and unformatted values
    fun registerField(fieldType: FieldType, collectableField: SecureTextFieldVM, collector: CardCollector) {
        DefaultLogManager.info("Registering field: $fieldType")
        collectableField.setCollector(Weak(collector))
        store[fieldType] = FieldValue("", "") // Initially store empty formatted and unformatted values
    }

    // Updates both the formatted and unformatted values for a given field
    fun updateFieldValue(fieldType: FieldType, formattedValue: String, unformattedValue: String) {
        store[fieldType] = FieldValue(formattedValue, unformattedValue) // Store both values
    }

    // Retrieves the FieldValue (formatted and unformatted) for a given field type
    fun getFieldValue(fieldType: FieldType): FieldValue? {
        return store[fieldType]
    }

    // Retrieves only the unformatted value for a given field type
    fun getUnformattedValue(fieldType: FieldType): String? {
        return store[fieldType]?.unformattedValue
    }

    // Retrieves only the formatted value for a given field type
    fun getFormattedValue(fieldType: FieldType): String? {
        return store[fieldType]?.formattedValue
    }

    // Returns all fields with their FieldValue (formatted and unformatted values)
    fun getAllFields(): Map<FieldType, FieldValue> {
        return store.toMap()
    }

    // Returns the set of all registered field types
    fun getRegisteredFieldTypes(): Set<FieldType> {
        return store.keys
    }

    // Checks if a field is registered in the store
    fun containsField(fieldType: FieldType): Boolean {
        return store.containsKey(fieldType)
    }
}