package com.moneyhash.shared.securevault.collectors

import com.moneyhash.shared.datasource.network.model.vault.CardFieldState
import com.moneyhash.shared.datasource.network.model.vault.CardFormState
import com.moneyhash.shared.localization.LocalizationManager
import com.moneyhash.shared.securevault.exceptions.CardFormExceptions
import com.moneyhash.shared.securevault.fields.FieldType
import com.moneyhash.shared.util.DefaultLogManager
import kotlinx.datetime.Clock
import kotlinx.datetime.TimeZone
import kotlinx.datetime.number
import kotlinx.datetime.toLocalDateTime

internal class FormValidator(
    private val fieldManager: FieldManager,
    private val cardBrandManager: CardBrandManager
) {
    /**
     * Validates all fields and returns a CardFormState containing a map of field types and their states.
     */
    internal fun validateFields(): CardFormState {
        val fieldsState = mutableMapOf<FieldType, CardFieldState>() // Map to collect validation states for all required fields

        // Validate if all fields exist
        fieldsState.putAll(validateAllFieldsExist())

        // Validate all fields' correctness
        fieldsState.putAll(validateAllFieldsValid())

        // Validate expiration month and year and add them to the map
        fieldsState.putAll(validateDateIsNotInPast())

        // Determine if the entire form is valid based on all the field states
        val isValid = fieldsState.values.all { it.isValid } // The form is valid if all fields are valid
        return CardFormState(isValid, fieldsState) // Return the overall form state with the map of field states
    }


    /**
     * Validates if all required fields are present in the form.
     * Returns a map with CardFieldState for each required field.
     */
    internal fun validateAllFieldsExist(): Map<FieldType, CardFieldState> {
        val fieldsState = mutableMapOf<FieldType, CardFieldState>()

        // List of required fields
        val requiredFields = FieldType.entries.filterNot { it == FieldType.CARD_HOLDER_NAME }
        val missingFields = requiredFields.filterNot { fieldManager.containsField(it) }

        // Create CardFieldState for each required field and its presence
        requiredFields.forEach { fieldType ->
            val isValid = !missingFields.contains(fieldType) // Check if field is missing
            fieldsState[fieldType] = CardFieldState(
                isValid = isValid,
                errorMessage = if (isValid) null else LocalizationManager.strings.missing_fields(
                    fieldType.label
                ),
                formattedValue = fieldManager.getFormattedValue(fieldType)
                    ?: "", // Get the formatted field value
                fieldType = fieldType,
                cardBrand = cardBrandManager.brandChangeFlow.value, // Current card brand (if applicable)
                length = fieldManager.getFormattedValue(fieldType)?.length ?: 0 // Length of the formatted value
            )
        }
        return fieldsState
    }

    /**
     * Validates the correctness of all fields using validators from CardBrandManager.
     * Returns a map with CardFieldState for each validated field.
     */
    private fun validateAllFieldsValid(): Map<FieldType, CardFieldState> {
        val fieldsState = mutableMapOf<FieldType, CardFieldState>()
        val fields = fieldManager.getAllFields() // Get all fields from fieldManager

        fields.forEach { (fieldType, fieldValue) ->
            try {
                // Get the appropriate validator for the field type based on the card number
                val validator = cardBrandManager.getValidator(fieldType, fieldManager.getUnformattedValue(FieldType.CARD_NUMBER))
                val isValid = validator.validate(fieldValue.unformattedValue) // Validate the unformatted value
                fieldsState[fieldType] = CardFieldState(
                    isValid = isValid,
                    errorMessage = if (isValid) null else LocalizationManager.strings.invalid_field(fieldType.label), // Error message if invalid
                    formattedValue = fieldValue.formattedValue, // The formatted value to display
                    fieldType = fieldType, // The type of the field
                    cardBrand = cardBrandManager.brandChangeFlow.value, // Current card brand (if applicable)
                    length = fieldValue.unformattedValue.length // Length of the unformatted value
                )
            } catch (e: Throwable) {
                // Return a CardFieldState with an error message if validation fails
                fieldsState[fieldType] = CardFieldState(
                    isValid = false,
                    errorMessage = LocalizationManager.strings.invalid_field(fieldType.label), // Error message for invalid field
                    formattedValue = fieldValue.formattedValue, // The formatted value
                    fieldType = fieldType, // The type of the field
                    cardBrand = cardBrandManager.brandChangeFlow.value, // Current card brand (if applicable)
                    length = fieldValue.unformattedValue.length // Length of the unformatted value
                )
            }
        }
        return fieldsState
    }


    /**
     * Validates that the expiration date (month and year) is not in the past.
     * Returns a map with two CardFieldState objects: one for the month and one for the year.
     */
    private fun validateDateIsNotInPast(): Map<FieldType, CardFieldState> {
        // Get the current date for comparison
        val currentDate = Clock.System.now().toLocalDateTime(TimeZone.currentSystemDefault())
        val currentYear = currentDate.year
        val currentMonth = currentDate.month.number

        // Get the unformatted year and month from the field manager
        val unformattedYear = fieldManager.getUnformattedValue(FieldType.EXPIRE_YEAR)
        val unformattedMonth = fieldManager.getUnformattedValue(FieldType.EXPIRE_MONTH)

        // Get the formatted year and month from the field manager (for displaying in the error message)
        val formattedYear = fieldManager.getFormattedValue(FieldType.EXPIRE_YEAR) ?: ""
        val formattedMonth = fieldManager.getFormattedValue(FieldType.EXPIRE_MONTH) ?: ""

        // Get the expiration year and month from the field manager
        val inputYear = unformattedYear?.toIntOrNull()?.plus(2000) // Convert 2-digit year to 4 digits
        val inputMonth = unformattedMonth?.toIntOrNull() // Get the month as an integer




        // Validate year: must not be in the past
        val isYearValid = inputYear != null && inputYear >= currentYear
        // Validate month: valid if year is in future, or if it's the current year and the month is valid
        val isMonthValid = inputMonth != null && ((inputYear != null && inputYear > currentYear) || (inputYear == currentYear && currentMonth <= inputMonth))

        // Create the CardFieldState for the expiration year
        val yearState = CardFieldState(
            isValid = isYearValid,
            errorMessage = if (isYearValid) null else LocalizationManager.strings.expiration_date_past, // Use the full expiration date in the error message
            formattedValue = formattedYear, // The formatted year to display
            fieldType = FieldType.EXPIRE_YEAR, // Field type for expiration year
            cardBrand = cardBrandManager.brandChangeFlow.value, // Current card brand
            length = unformattedYear?.length ?: 0 // Length of the formatted year
        )

        // Create the CardFieldState for the expiration month
        val monthState = CardFieldState(
            isValid = isMonthValid,
            errorMessage = if (isMonthValid) null else LocalizationManager.strings.expiration_date_past, // Use the same error message for the month
            formattedValue = formattedMonth, // The formatted month to display
            fieldType = FieldType.EXPIRE_MONTH, // Field type for expiration month
            cardBrand = cardBrandManager.brandChangeFlow.value, // Current card brand
            length = unformattedMonth?.length ?: 0 // Length of the formatted month
        )

        // Return both the year and month states as a map
        return mapOf(
            FieldType.EXPIRE_YEAR to yearState,
            FieldType.EXPIRE_MONTH to monthState
        )
    }
}
