package com.moneyhash.shared.securevault.models

import com.moneyhash.shared.CommonParcelable
import com.moneyhash.shared.CommonParcelize
import com.moneyhash.shared.datasource.network.model.vault.Brand
import com.moneyhash.shared.datasource.network.model.vault.CardBrand
import com.moneyhash.shared.securevault.fields.FieldType
import com.moneyhash.shared.securevault.formatters.CVVFormatter
import com.moneyhash.shared.securevault.formatters.CardHolderNameFormatter
import com.moneyhash.shared.securevault.formatters.CardNumberFormatter
import com.moneyhash.shared.securevault.formatters.ExpireYearFormatter
import com.moneyhash.shared.securevault.formatters.ExpireMonthFormatter
import com.moneyhash.shared.securevault.formatters.TextFormatter
import com.moneyhash.shared.securevault.validators.CVVValidator
import com.moneyhash.shared.securevault.validators.CardHolderNameValidator
import com.moneyhash.shared.securevault.validators.CreditCardValidator
import com.moneyhash.shared.securevault.validators.ExpireMonthValidator
import com.moneyhash.shared.securevault.validators.ExpireYearValidator
import com.moneyhash.shared.securevault.validators.Validator
import com.moneyhash.shared.util.Constants
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable

/**
 * An enumeration representing various brands of payment cards.
 *
 * Each enum constant corresponds to a specific card brand and holds its name.
 *
 * Available card brands:
 * - `ELO`: Represents the Elo card brand.
 * - `VISA_ELECTRON`: Represents the Visa Electron card brand.
 * - `MAESTRO`: Represents the Maestro card brand.
 * - `FORBRUGSFORENINGEN`: Represents the Forbrugsforeningen card brand.
 * - `DANKORT`: Represents the Dankort card brand.
 * - `VISA`: Represents the Visa card brand.
 * - `MASTERCARD`: Represents the Mastercard brand.
 * - `AMEX`: Represents the American Express card brand.
 * - `HIPERCARD`: Represents the Hipercard brand.
 * - `DINERS_CLUB`: Represents the Diners Club card brand.
 * - `DISCOVER`: Represents the Discover card brand.
 * - `UNIONPAY`: Represents the UnionPay card brand.
 * - `JCB`: Represents the JCB card brand.
 * - `UNKNOWN`: Represents an unknown card brand.
 */

@CommonParcelize
@Serializable
internal enum class CardBrandInternal(private val brandName: String): CommonParcelable {
    @SerialName("mada")
    MADA("Mada"),
    @SerialName("elo")
    ELO("ELO"),
    @SerialName("visa_electron")
    VISA_ELECTRON("Visa Electron"),
    @SerialName("maestro")
    MAESTRO("Maestro"),
    @SerialName("forbrugsforeningen")
    FORBRUGSFORENINGEN("Forbrugsforeningen"),
    @SerialName("dankort")
    DANKORT("Dankort"),
    @SerialName("visa")
    VISA("Visa"),
    @SerialName("mastercard")
    MASTERCARD("Mastercard"),
    @SerialName("amex")
    AMEX("American Express"),
    @SerialName("hipercard")
    HIPERCARD("HiperCard"),
    @SerialName("diners_club")
    DINERS_CLUB("Diners Club"),
    @SerialName("discover")
    DISCOVER("Discover"),
    @SerialName("unionpay")
    UNIONPAY("UnionPay"),
    @SerialName("jcb")
    JCB("JCB"),
    @SerialName("unknown")
    UNKNOWN("Unknown");

    internal fun defaultRegex(): String {
        return when (this) {
            MADA -> "^(446404|440795|440647|421141|474491|588845|968208|457997|457865|468540|468541|468542|468543|417633|446393|636120|968201|410621|409201|403024|458456|462220|968205|455708|484783|588848|455036|968203|486094|486095|486096|504300|440533|489318|489319|445564|968211|410685|406996|432328|428671|428672|428673|968206|446672|543357|434107|407197|407395|42689700|412565|604906|521076|588850|968202|529415|535825|543085|524130|554180|549760|968209|524514|529741|537767|535989|536023|513213|520058|558563|605141|968204|422817|422818|422819|410834|428331|483010|483011|483012|589206|968207|406136|419593|439954|407520|530060|531196|420132|242030|22402030|442463)\\d*$"
            AMEX -> "^3[47]\\d*$"
            DINERS_CLUB -> "^3(?:[689]|(?:0[059]+))\\d*$"
            DISCOVER -> "^(6011|65|64[4-9]|622)\\d*$"
            UNIONPAY -> "^62\\d*$"
            JCB -> "^(2131|1800|35)\\d*$"
            MASTERCARD -> "^(5[1-5][0-9]{4})\\d*$|^(222[1-9]|22[3-9]|2[3-6]\\d{2}|27[0-1]\\d|2720)([0-9]{2})\\d*$"
            VISA_ELECTRON -> "^4(026|17500|405|508|844|91[37])\\d*$"
            VISA -> "^4\\d*$"
            MAESTRO -> "^(5018|5020|5038|6304|6390[0-9]{2}|67[0-9]{4})\\d*$"
            FORBRUGSFORENINGEN -> "^600\\d*$"
            DANKORT -> "^5019\\d*$"
            ELO -> "^(4011(78|79)|43(1274|8935)|45(1416|7393|763(1|2))|50(4175|6699|67[0-7][0-9]|9000)|627780|63(6297|6368)|650(03([^4])|04([0-9])|05(0|1)|4(0[5-9]|3[0-9]|8[5-9]|9[0-9])|5([0-2][0-9]|3[0-8])|9([2-6][0-9]|7[0-8])|541|700|720|901)|651652|655000|655021)\\d*$"
            HIPERCARD -> "^(384100|384140|384160|606282|637095|637568|60(?!11))\\d*$"
            UNKNOWN -> "^\\d*$"
        }
    }

    internal fun defaultCardLengths(): List<Int> {
        return listOf(13, 14, 15, 16, 17, 18, 19)
    }

    // This commented code because we decided to relax the validation for the card length
    // and we will consider the card is valid if the card number is numeric and the length is between 13 and 19
//    internal fun defaultCardLengths(): List<Int> {
//        return when (this) {
//            MADA -> listOf(16)
//            AMEX -> listOf(15)
//            DINERS_CLUB -> listOf(14, 16)
//            DISCOVER -> listOf(16)
//            UNIONPAY -> listOf(16, 17, 18, 19)
//            JCB -> listOf(15, 16)
//            MASTERCARD -> listOf(16)
//            VISA_ELECTRON -> listOf(16)
//            VISA -> listOf(13, 16)
//            MAESTRO -> listOf(12, 13, 14, 15, 16, 17, 18, 19)
//            ELO -> listOf(16)
//            FORBRUGSFORENINGEN -> listOf(16)
//            DANKORT -> listOf(16)
//            HIPERCARD -> listOf(14, 15, 16, 17, 18, 19)
//            UNKNOWN -> listOf(16, 17, 18, 19)
//        }
//    }

    internal fun defaultCVCLengths(): List<Int> {
        return when (this) {
            AMEX -> listOf(4)
            UNKNOWN -> listOf(3, 4)
            else -> listOf(3)
        }
    }

    internal fun defaultFormatPattern(): String {
        return when (this) {
            AMEX -> "#### ###### #####"
            DINERS_CLUB -> "#### ###### ######"
            else -> "#### #### #### #### ###"
        }
    }

    internal fun cvcFormatPattern(availableCardBrandInternals: List<CardBrandInternal> = emptyList()): String {
        val maxLength =
            availableCardBrandInternals.firstOrNull { it == this }?.defaultCVCLengths()?.maxOrNull()
                ?: UNKNOWN.defaultCVCLengths().maxOrNull() ?: 0
        return "#".repeat(maxLength)
    }

    internal fun defaultName() = brandName

    fun getCardBrand(first6Digits:String): CardBrand {
        return when (this) {
            VISA -> CardBrand(Brand.VISA, first6Digits, carBrandUrl)
            MASTERCARD -> CardBrand(Brand.MASTERCARD, first6Digits, carBrandUrl)
            MADA ->  CardBrand(Brand.MADA, first6Digits, carBrandUrl)
            else ->  CardBrand(Brand.UNKNOWN, first6Digits, carBrandUrl)
        }
    }

    internal fun validator(fieldType: FieldType): Validator {
        return when (fieldType)
        {
            FieldType.CARD_NUMBER -> CreditCardValidator(this)
            FieldType.EXPIRE_MONTH -> ExpireMonthValidator()
            FieldType.EXPIRE_YEAR -> ExpireYearValidator()
            FieldType.CVV -> CVVValidator(this)
            FieldType.CARD_HOLDER_NAME -> CardHolderNameValidator()
        }
    }

    internal fun formatter(fieldType: FieldType): TextFormatter {
        return when (fieldType) {
            FieldType.CARD_NUMBER -> CardNumberFormatter(this)
            FieldType.EXPIRE_YEAR -> ExpireYearFormatter()
            FieldType.EXPIRE_MONTH -> ExpireMonthFormatter()
            FieldType.CVV -> CVVFormatter(this)
            FieldType.CARD_HOLDER_NAME -> CardHolderNameFormatter()
        }
    }

    private val carBrandUrl: String
        get() {
            return when (this) {
                VISA -> Constants.BASE_VAULT_FORM + "Visa.svg"
                MASTERCARD -> Constants.BASE_VAULT_FORM + "MasterCard.svg"
                MADA -> Constants.BASE_VAULT_FORM + "Mada.svg"
                else -> Constants.BASE_VAULT_FORM + "UnknownCard.svg"
            }
        }

    companion object {

        internal fun fromCardNumber(cardNumber: String): CardBrandInternal {
            return cardNumber.let {
                entries.firstOrNull { cardBrand ->
                    it.matches(Regex(cardBrand.defaultRegex()))
                }
            } ?: UNKNOWN
        }

    }
}