package io.nearpay.install.core

import android.app.DownloadManager
import android.app.PendingIntent
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.content.pm.PackageInstaller
import android.net.Uri
import android.os.Build
import android.os.Environment
import android.webkit.URLUtil.isValidUrl
import androidx.core.net.toUri
import io.nearpay.install.core.data.dto.InstallationFailure
import io.nearpay.install.core.listener.InstallationListener
import io.nearpay.install.core.utils.parcelable
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import java.io.File

 class InstallManager {

    private lateinit var downloadManager: DownloadManager
    private var downloadId: Long = -1

    private lateinit var context: Context
    private lateinit var downloadUrl: String
    private lateinit var apkName: String

    companion object {
        private var installationListener: InstallationListener? = null
    }

    fun start(
        setContext: Context,
        setDownloadUrl: String,
        setApkName: String,
        setListener: InstallationListener
    ) {
        context = setContext
        downloadUrl = setDownloadUrl
        apkName = setApkName
        installationListener = setListener

        if (!isValidUrl(downloadUrl)){
            installationListener?.onInstallationFailed(InstallationFailure.DownloadedFileNotFound)
            return
        }

        installNearpayPackage()
    }

    private fun installNearpayPackage() {
        if (hasDownloaded())
            installDownloadedApk()
        else
            downloadApk()
    }

    private fun getFile(): File {
        return File(context.getExternalFilesDir(Environment.DIRECTORY_DOWNLOADS), apkName)
    }

    fun hasDownloaded(): Boolean {
        val file = getFile()
        return file.exists() && file.isFile
    }

    private fun downloadApk() {
        val dialogTitle = "Payment Plugin"

        /***
         * Enqueue the download.The download will start automatically once the download manager is
         * ready to execute it and connectivity is available.
         */
        val request = DownloadManager.Request(Uri.parse(downloadUrl)).apply {
            setMimeType("application/vnd.android.package-archive")
            setTitle(dialogTitle)
            setDescription("File is downloading...")
            setDestinationInExternalFilesDir(context, Environment.DIRECTORY_DOWNLOADS, apkName)
            setNotificationVisibility(DownloadManager.Request.VISIBILITY_VISIBLE_NOTIFY_COMPLETED)
        }

        downloadManager = (context.getSystemService(Context.DOWNLOAD_SERVICE) as DownloadManager)
        downloadId = downloadManager.enqueue(request)

        installationListener?.onDownloadStart()

        observeDownloadingFiles(downloadId)

        CoroutineScope(Dispatchers.Main).launch {
            do {
                val downloadQuery = DownloadManager.Query().apply {
                    setFilterById(downloadId)
                }

                val cursor = downloadManager.query(downloadQuery).apply {
                    moveToFirst()
                }

                val bytesDownloaded: Long = try {
                    cursor.getLong(cursor.getColumnIndexOrThrow(DownloadManager.COLUMN_BYTES_DOWNLOADED_SO_FAR))
                } catch (throwable: Throwable) {
                    return@launch
                }
                val bytesTotal = cursor.getLong(cursor.getColumnIndexOrThrow(DownloadManager.COLUMN_TOTAL_SIZE_BYTES))
                val progress = try {
                    (bytesDownloaded * 100 / bytesTotal).toInt()
                } catch (throwable: Throwable) {
                    0
                }
                installationListener?.onDownloadProgressChange(progress)

                delay(100)
            } while (cursor.getInt(cursor.getColumnIndexOrThrow(DownloadManager.COLUMN_STATUS)) != DownloadManager.STATUS_SUCCESSFUL)
        }
    }

    private fun observeDownloadingFiles(downloadId: Long) {
        val onComplete = object : BroadcastReceiver() {
            override fun onReceive(context: Context, intent: Intent) {
                val id = intent.getLongExtra(DownloadManager.EXTRA_DOWNLOAD_ID, -1)
                if (downloadId == id) {
                    context.unregisterReceiver(this)
                    if (hasDownloaded())
                        installDownloadedApk()
                    else
                        installationListener?.onDownloadFailed()
                }
            }
        }

        context.registerReceiver(
            onComplete,
            IntentFilter(DownloadManager.ACTION_DOWNLOAD_COMPLETE)
        )
    }

    private fun installDownloadedApk() {
        val file = getFile()

        if (!hasDownloaded()) {
            installationListener?.onInstallationFailed(InstallationFailure.DownloadedFileNotFound)
            return
        }

        val packageInstaller = context.applicationContext.packageManager.packageInstaller
        val resolver = context.applicationContext.contentResolver

        resolver.openInputStream(file.toUri())?.use { apkStream ->

            val params =
                PackageInstaller.SessionParams(PackageInstaller.SessionParams.MODE_FULL_INSTALL)
            val sessionId = packageInstaller.createSession(params)
            val session = packageInstaller.openSession(sessionId)

            session.openWrite(apkName, 0, file.length()).use { sessionStream ->
                apkStream.copyTo(sessionStream)
                session.fsync(sessionStream)
            }

            val intent = Intent(
                context.applicationContext,
                FirebaseInstallReceiver::class.java
            ).putExtra("apkName", apkName)

            val flags =
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S)
                    PendingIntent.FLAG_MUTABLE or PendingIntent.FLAG_UPDATE_CURRENT
                else
                    PendingIntent.FLAG_UPDATE_CURRENT

            val pendingIntent =
                PendingIntent.getBroadcast(context.applicationContext, sessionId, intent, flags)

            session.commit(pendingIntent.intentSender)
            session.close()
        }
    }

    fun cancelDownload() {
        if (::downloadManager.isInitialized) {
            downloadManager.remove(downloadId)
        }
    }

    internal class FirebaseInstallReceiver : BroadcastReceiver() {
        override fun onReceive(context: Context, intent: Intent) {

            when (intent.getIntExtra(PackageInstaller.EXTRA_STATUS, -1)) {
                PackageInstaller.STATUS_PENDING_USER_ACTION -> {
                    installationListener?.onInstallationStart()
                    val activityIntent = intent.parcelable<Intent>(Intent.EXTRA_INTENT)

                    if (activityIntent == null) {
                        installationListener?.onInstallationFailed(
                            InstallationFailure.InstallFailed(
                                "Unexpected"
                            )
                        )
                        return
                    }

                    context.startActivity(activityIntent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK))
                }

                PackageInstaller.STATUS_SUCCESS -> {
                    installationListener?.onInstallationCompleted()
                }

                PackageInstaller.STATUS_FAILURE_INVALID -> {
                    handleFailedInstall(context, intent)

                }

                PackageInstaller.STATUS_FAILURE,
                PackageInstaller.STATUS_FAILURE_BLOCKED,
                PackageInstaller.STATUS_FAILURE_ABORTED,
                PackageInstaller.STATUS_FAILURE_CONFLICT,
                PackageInstaller.STATUS_FAILURE_STORAGE,
                PackageInstaller.STATUS_FAILURE_INCOMPATIBLE -> {
                    handleFailedInstall(context, intent)
                }

                else ->
                    installationListener?.onInstallationFailed(InstallationFailure.InstallFailed("Unexpected"))
            }
        }


        private fun handleFailedInstall(context: Context, intent: Intent) {
            val message =
                intent.getStringExtra(PackageInstaller.EXTRA_STATUS_MESSAGE).orEmpty()
            installationListener?.onInstallationFailed(InstallationFailure.InstallFailed(message))

            intent.extras?.getString("apkName")?.let {
                File(
                    context.getExternalFilesDir(Environment.DIRECTORY_DOWNLOADS),
                    it
                ).delete()
            }
        }
    }
}