package io.nearpay.install.core.data.remote

import android.content.Context
import android.util.Log
import com.jakewharton.retrofit2.converter.kotlinx.serialization.asConverterFactory
import io.nearpay.install.core.BuildConfig
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.json.Json
import okhttp3.Cache
import okhttp3.CipherSuite
import okhttp3.ConnectionSpec
import okhttp3.Interceptor
import okhttp3.MediaType.Companion.toMediaType
import okhttp3.OkHttpClient
import okhttp3.TlsVersion
import okhttp3.logging.HttpLoggingInterceptor
import retrofit2.Retrofit
import java.io.File
import java.util.Collections
import java.util.concurrent.TimeUnit

 object SdkApiClient {

    private const val CONNECT_TIMEOUT_IN_SECONDS = 10
    private const val READ_TIMEOUT_IN_SECONDS = 60
    private const val WRITE_TIMEOUT_IN_SECONDS = 60
    private const val CACHE_SIZE_BYTES = 10 * 1024 * 1024L // 10 MB

    operator fun invoke(
        context: Context,
        packageName: String,
        serverUrl: String,
        sdkVersion:String
    ): ServiceApi {

        val json = Json { isLenient = true; ignoreUnknownKeys = true }

        val headerInterceptor = Interceptor {
            val requestBuilder = it.request().newBuilder()

            if (packageName.matches("[a-zA-Z0-9._-]+".toRegex()))
                requestBuilder.addHeader("package", packageName)

            requestBuilder.addHeader("Content-Type", "application/json")
            requestBuilder.addHeader("sdk-version", sdkVersion)
            it.proceed(requestBuilder.build())
        }

        val httpCacheDirectory = File(context.cacheDir.absolutePath, "HttpCache")
        val cache = Cache(httpCacheDirectory, CACHE_SIZE_BYTES)

        val spec: ConnectionSpec = ConnectionSpec.Builder(ConnectionSpec.COMPATIBLE_TLS)
            .tlsVersions(TlsVersion.TLS_1_2, TlsVersion.TLS_1_1, TlsVersion.TLS_1_0)
            .cipherSuites(
                CipherSuite.TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256,
                CipherSuite.TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256,
                CipherSuite.TLS_DHE_RSA_WITH_AES_128_GCM_SHA256,
                CipherSuite.TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA,
                CipherSuite.TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA,
                CipherSuite.TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA,
                CipherSuite.TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA,
                CipherSuite.TLS_ECDHE_ECDSA_WITH_RC4_128_SHA,
                CipherSuite.TLS_ECDHE_RSA_WITH_RC4_128_SHA,
                CipherSuite.TLS_DHE_RSA_WITH_AES_128_CBC_SHA,
                CipherSuite.TLS_DHE_DSS_WITH_AES_128_CBC_SHA,
                CipherSuite.TLS_DHE_RSA_WITH_AES_256_CBC_SHA
            ).build()

        val loggingInterceptor = HttpLoggingInterceptor {
            Log.i("Network", it)
        }.setLevel(HttpLoggingInterceptor.Level.BODY)

        val httpClient = OkHttpClient.Builder()
            .connectionSpecs(Collections.singletonList(spec))
            .readTimeout(READ_TIMEOUT_IN_SECONDS.toLong(), TimeUnit.SECONDS)
            .writeTimeout(WRITE_TIMEOUT_IN_SECONDS.toLong(), TimeUnit.SECONDS)
            .connectTimeout(CONNECT_TIMEOUT_IN_SECONDS.toLong(), TimeUnit.SECONDS)
            .cache(cache)
            .addInterceptor(headerInterceptor)

        //Enable logging in debug mode only
        if (BuildConfig.DEBUG) {
            httpClient.addInterceptor(loggingInterceptor)
        }

        @OptIn(ExperimentalSerializationApi::class)
        val retrofit = Retrofit.Builder()
            .baseUrl(serverUrl)
            .client(httpClient.build())
            .addConverterFactory(json.asConverterFactory("application/json".toMediaType()))
            .build()

        return retrofit.create(ServiceApi::class.java)
    }
}