package io.nearpay.install.core.data.remote

import io.nearpay.install.core.data.dto.NiceErrors
import io.nearpay.install.core.data.dto.ServerErrorMessage
import kotlinx.serialization.SerializationException
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.json.Json
import retrofit2.HttpException
import java.net.UnknownHostException

fun resolveError(throwable: Throwable): ApiResponse.Error {
    return when (throwable) {
        is HttpException -> resolveHttpException(throwable)
        is SerializationException -> RequestException.parsingException(throwable)
        is UnknownHostException -> RequestException.networkError(throwable)
        else -> RequestException.unexpectedError(throwable)
    }.asFailure()
}

private fun resolveHttpException(e: HttpException): RequestException {

    fun HttpException.getServerErrorMsg(): ServerErrorMessage? {
        return try {
            val errorMessageObject = e.response()?.errorBody()?.string().orEmpty()
            val json = Json { ignoreUnknownKeys = true; isLenient = true }
            json.decodeFromString<ServerErrorMessage>(errorMessageObject)
        } catch (throwable: Throwable) {
            null
        }
    }

    val errorMessage = e.getServerErrorMsg() ?: NiceErrors.PARSING_ERROR_MSG_FAILED.messageError

    return when (e.code()) {
        401 -> RequestException.authenticationError(e, errorMessage)
        404 -> RequestException.notFountError(e, errorMessage)
        else -> RequestException.serviceError(e, errorMessage)
    }
}

private fun RequestException.asFailure() = ApiResponse.Error(this)